<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_yuboto\Unit;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformTokenManagerInterface;
use Drupal\webform_yuboto\Plugin\WebformHandler\WebformYubotoHandler;
use Drupal\webform_yuboto\YubotoApiService;
use PHPUnit\Framework\Attributes\CoversClass;

/**
 * Unit tests for WebformYubotoHandler token replacement behavior.
 *
 * @group webform_yuboto
 */
#[CoversClass(WebformYubotoHandler::class)]
final class WebformYubotoHandlerTokenReplacementTest extends UnitTestCase {

  /**
   * Sets a (possibly inherited) property via reflection.
   */
  private function setObjectProperty(object $object, string $property, mixed $value): void {
    $ref = new \ReflectionObject($object);
    while (!$ref->hasProperty($property)) {
      $ref = $ref->getParentClass();
      if ($ref === FALSE) {
        self::fail(sprintf('Property "%s" not found on %s or any parent class.', $property, get_class($object)));
      }
    }

    $prop = $ref->getProperty($property);
    $prop->setAccessible(TRUE);
    $prop->setValue($object, $value);
  }

  /**
   * Verifies message/sender/phone tokens are replaced before calling the API.
   *
   * This test focuses on the handler behavior (not the API service).
   */
  public function testPostSaveReplacesTokensBeforeCallingApi(): void {
    $webform_submission = $this->createMock(WebformSubmissionInterface::class);
    $webform_submission->method('id')->willReturn(123);

    // Submission data does NOT contain the eventual phone, ensuring the handler
    // uses the token-derived literal phone number branch.
    $webform_submission->method('toArray')
      ->with(TRUE)
      ->willReturn([
        'data' => [
          'some_other_field' => 'value',
        ],
      ]);

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::exactly(2))
      ->method('invokeAll')
      ->willReturnCallback(static function (string $hook, array $args): array {
        self::assertContains($hook, ['webform_yuboto_skip_handler', 'webform_yuboto_skip_sms']);
        return [];
      });

    $config = $this->createMock(Config::class);
    $config->method('getRawData')
      ->willReturn([
        'default_message' => 'Default message',
        'default_sender' => 'Default sender',
      ]);

    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config_factory->method('get')
      ->with('webform_yuboto.settings')
      ->willReturn($config);

    $token_manager = $this->createMock(WebformTokenManagerInterface::class);

    $replace_calls = 0;
    $token_manager->expects(self::exactly(4))
      ->method('replace')
      ->willReturnCallback(static function (mixed $value, WebformSubmissionInterface $submission, array $options = []) use (&$replace_calls): mixed {
        $replace_calls++;
        self::assertSame(123, $submission->id());
        self::assertSame(['clear' => TRUE], $options);

        // 1) Replace on configuration array.
        if (is_array($value)) {
          return [
            // Keep a token here; we'll resolve it in call #2.
            'phone_field' => '[webform_submission:values:phone]',
            // Simulate token replacement already happened for these.
            'message' => 'Hello John',
            'sender' => 'MySite',
          ];
        }

        // 2) Replace phone_field token to a literal phone number.
        if ($replace_calls === 2) {
          return '6912345678';
        }

        // 3) and 4) message/sender "additional token replacement" are no-ops.
        return $value;
      });

    $yuboto_api = $this->createMock(YubotoApiService::class);
    $yuboto_api->expects(self::once())
      ->method('sendSms')
      ->with('6912345678', 'Hello John', 'MySite')
      ->willReturn(['ErrorCode' => 0]);

    // Create handler instance without invoking the plugin constructor.
    try {
      $handler = (new \ReflectionClass(WebformYubotoHandler::class))
        ->newInstanceWithoutConstructor();
      $this->setObjectProperty($handler, 'yubotoApi', $yuboto_api);
      $this->setObjectProperty($handler, 'tokenManager', $token_manager);
      $this->setObjectProperty($handler, 'moduleHandler', $module_handler);
      $this->setObjectProperty($handler, 'configFactory', $config_factory);

      // Raw configuration values (contain tokens).
      $this->setObjectProperty($handler, 'configuration', [
        'phone_field' => '[webform_submission:values:phone]',
        'message' => 'Hello [webform_submission:sid]',
        'sender' => '[site:name]',
      ]);

      $handler->postSave($webform_submission, FALSE);
    }
    catch (\ReflectionException $e) {
      self::fail($e->getMessage());
    }
  }

}
