<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_yuboto\Unit;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\webform_yuboto\YubotoApiService;
use GuzzleHttp\ClientInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

/**
 * Unit tests for the Yuboto API service.
 *
 * @group webform_yuboto
 */
#[CoversClass(YubotoApiService::class)]
final class YubotoApiServiceTest extends UnitTestCase {

  /**
   * Creates a service with configurable config values and mocks.
   *
   * @param array $config_values
   *   Values returned by config->get($key).
   * @param \GuzzleHttp\ClientInterface|null $http_client
   *   Optional HTTP client mock.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface|null $module_handler
   *   Optional module handler mock.
   * @param \Drupal\Core\Logger\LoggerChannelInterface|null $logger
   *   Optional logger mock.
   *
   * @return \Drupal\webform_yuboto\YubotoApiService
   *   The service under test.
   */
  private function createService(
    array $config_values,
    ?ClientInterface $http_client = NULL,
    ?ModuleHandlerInterface $module_handler = NULL,
    ?LoggerChannelInterface $logger = NULL,
  ): YubotoApiService {
    $config = $this->createMock(Config::class);
    $config->method('get')
      ->willReturnCallback(static fn(string $key) => $config_values[$key] ?? NULL);

    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config_factory->method('get')
      ->with('webform_yuboto.settings')
      ->willReturn($config);

    $logger ??= $this->createMock(LoggerChannelInterface::class);

    $logger_factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger_factory->method('get')
      ->with('webform_yuboto')
      ->willReturn($logger);

    $http_client ??= $this->createMock(ClientInterface::class);
    $module_handler ??= $this->createMock(ModuleHandlerInterface::class);

    return new YubotoApiService($config_factory, $logger_factory, $http_client, $module_handler);
  }

  /**
   * Provides phone number validation test cases.
   *
   * @return array<string, array{0: string, 1: bool, 2: string, 3: string}>
   *   Dataset of (input, expected valid, expected normalized, expected error).
   */
  public static function phoneNumberProvider(): array {
    return [
      'empty' => ['', FALSE, '', 'Phone number is empty'],
      'ten_digits_valid' => ['6912345678', TRUE, '306912345678', ''],
      'twelve_digits_valid' => ['306912345678', TRUE, '306912345678', ''],
      'ten_digits_invalid_prefix' => [
        '6812345678',
        FALSE,
        '',
        'Phone number must be a Greek mobile number starting with 69',
      ],
      'twelve_digits_invalid_prefix' => [
        '306812345678',
        FALSE,
        '',
        'Phone number must be a Greek mobile number starting with 69',
      ],
      'invalid_length' => [
        '691234567',
        FALSE,
        '',
        'Invalid phone number format. Expected 10-digit Greek mobile number starting with 69, or 12-digit number with 30 country code',
      ],
      'non_digits_are_ignored' => ['+30 69 1234 5678', TRUE, '306912345678', ''],
    ];
  }

  /**
   * Tests phone number validation and normalization.
   *
   * @dataProvider phoneNumberProvider
   */
  #[DataProvider('phoneNumberProvider')]
  public function testValidateGreekMobileNumber(string $input, bool $expected_valid, string $expected_normalized, string $expected_error): void {
    $service = $this->createService([
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'dummy',
    ]);

    $result = $service->validateGreekMobileNumber($input);

    self::assertSame($expected_valid, $result['valid']);
    self::assertSame($expected_normalized, $result['normalized']);
    self::assertSame($expected_error, $result['error']);
  }

  /**
   * Tests sendSms() returns "disabled" when the module is disabled in config.
   */
  public function testSendSmsReturnsDisabledWhenDisabledInConfig(): void {
    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::never())
      ->method('request');

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::never())
      ->method('alter');

    $logger = $this->createMock(LoggerChannelInterface::class);
    $logger->expects(self::once())
      ->method('info');

    $service = $this->createService([
      'enabled' => FALSE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'key',
    ], $http_client, $module_handler, $logger);

    $result = $service->sendSms('6912345678', 'Hello', 'Sender');

    self::assertSame('disabled', $result['status']);
  }

  /**
   * Tests sendSms() returns an error when the API key is missing.
   */
  public function testSendSmsReturnsErrorWhenApiKeyMissing(): void {
    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::never())
      ->method('request');

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::never())
      ->method('alter');

    $service = $this->createService([
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => '',
    ], $http_client, $module_handler);

    $result = $service->sendSms('6912345678', 'Hello', 'Sender');

    self::assertSame('error', $result['status']);
    self::assertSame('Yuboto API key is not configured.', $result['message']);
  }

  /**
   * Tests sendSms() returns an error when the phone number is invalid.
   */
  public function testSendSmsReturnsErrorWhenPhoneInvalid(): void {
    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::never())
      ->method('request');

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::never())
      ->method('alter');

    $service = $this->createService([
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'key',
    ], $http_client, $module_handler);

    $result = $service->sendSms('6812345678', 'Hello', 'Sender');

    self::assertSame('error', $result['status']);
    self::assertStringContainsString('Invalid phone number:', $result['message']);
  }

  /**
   * Tests sendSms() returns an error when the API returns a non-zero ErrorCode.
   */
  public function testSendSmsReturnsApiErrorWhenErrorCodeNonZero(): void {
    $stream = $this->createMock(StreamInterface::class);
    $stream->method('getContents')
      ->willReturn('{"ErrorCode":1,"ErrorMessage":"No credit"}');

    $response = $this->createMock(ResponseInterface::class);
    $response->method('getBody')
      ->willReturn($stream);

    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::once())
      ->method('request')
      ->with(
        'POST',
        'https://services.yuboto.com/omni/v1/Send',
        self::callback(static fn($v): bool => is_array($v)),
      )
      ->willReturn($response);

    $logger = $this->createMock(LoggerChannelInterface::class);
    $logger->expects(self::once())
      ->method('error')
      ->with(self::stringContains('Yuboto API error 1: No credit'));

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::once())
      ->method('alter');

    $service = $this->createService([
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'key',
    ], $http_client, $module_handler, $logger);

    $result = $service->sendSms('6912345678', 'Hello', 'Sender');

    self::assertSame('error', $result['status']);
    self::assertStringContainsString('Yuboto API error 1: No credit', $result['message']);
  }

  /**
   * Tests sendSms() issues a POST request with expected headers and payload.
   */
  public function testSendSmsSuccessPostsWithExpectedHeadersAndBody(): void {
    $stream = $this->createMock(StreamInterface::class);
    $stream->method('getContents')
      ->willReturn('{"ErrorCode":0,"foo":"bar"}');

    $response = $this->createMock(ResponseInterface::class);
    $response->method('getBody')
      ->willReturn($stream);

    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::once())
      ->method('request')
      ->with(
        'POST',
        'https://services.yuboto.com/omni/v1/Send',
        self::callback(static function (array $options): bool {
          if (!isset($options['headers']['Authorization'])) {
            return FALSE;
          }
          if ($options['headers']['Authorization'] !== 'Basic ' . base64_encode('key')) {
            return FALSE;
          }
          if (($options['headers']['Accept'] ?? NULL) !== 'application/json') {
            return FALSE;
          }
          if (!isset($options['json']['contacts'][0]['phonenumber'])) {
            return FALSE;
          }
          if ($options['json']['contacts'][0]['phonenumber'] !== '306912345678') {
            return FALSE;
          }
          if (($options['json']['sms']['sender'] ?? NULL) !== 'Sender') {
            return FALSE;
          }
          if (($options['json']['sms']['text'] ?? NULL) !== 'Hello') {
            return FALSE;
          }
          return TRUE;
        })
      )
      ->willReturn($response);

    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $module_handler->expects(self::once())
      ->method('alter')
      ->with(
        'webform_yuboto_sms_data',
        self::callback(static fn($v): bool => is_array($v)),
        self::callback(static fn($v): bool => is_array($v)),
      );

    $service = $this->createService([
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'key',
    ], $http_client, $module_handler);

    $result = $service->sendSms('6912345678', 'Hello', 'Sender');

    self::assertSame(['ErrorCode' => 0, 'foo' => 'bar'], $result);
  }

  /**
   * Tests validateApiKey() with a temporary validation key.
   *
   * Confirms the method returns TRUE on a 200 response.
   */
  public function testValidateApiKeyUsesTemporaryValidationKeyAndReturnsTrueOn200(): void {
    $response = $this->createMock(ResponseInterface::class);
    $response->method('getStatusCode')
      ->willReturn(200);

    $http_client = $this->createMock(ClientInterface::class);
    $http_client->expects(self::once())
      ->method('request')
      ->with(
        'GET',
        'https://services.yuboto.com/omni/v1/UserBalance',
        self::callback(static function (array $options): bool {
          return ($options['headers']['Authorization'] ?? '') === 'Basic ' . base64_encode('temp_key');
        })
      )
      ->willReturn($response);

    $service = $this->createService([
      // Even if configured key exists, we will override it with temp key.
      'enabled' => TRUE,
      'debug_enabled' => FALSE,
      'yuboto_api_key' => 'configured_key',
    ], $http_client);

    $service->setApiKeyForValidation('temp_key');
    self::assertTrue($service->validateApiKey());
  }

}
