<?php

namespace Drupal\wedia_responsive_image\Plugin\Field\FieldFormatter;

use Drupal\breakpoint\BreakpointManager;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\LinkGeneratorInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\wedia\Plugin\Field\FieldType\WediaImageField;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Wedia responsive image field formatter.
 *
 * @FieldFormatter(
 *   id = "wedia_responsive_image_field_formatter",
 *   label = @Translation("Wedia responsive image formatter"),
 *   description = @Translation("Display the responsive wedia image"),
 *   field_types = {
 *     "wedia_image_field",
 *   }
 * )
 */
class WediaResponsiveImageFieldFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The image style entity storage.
   *
   * @var \Drupal\image\ImageStyleStorageInterface
   */
  protected $imageStyleStorage;

  /**
   * The responsive image style entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $responsiveImageStyleStorage;

  /**
   * The link generator.
   *
   * @var \Drupal\Core\Utility\LinkGeneratorInterface
   */
  protected $linkGenerator;

  /**
   * The breakpoint manager.
   *
   * @var \Drupal\breakpoint\BreakpointManager
   */
  protected $breakpointManager;

  /**
   * Constructs an WediaImageFieldFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityStorageInterface $image_style_storage
   *   The image style storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $responsive_image_style_storage
   *   The responsive image style storage.
   * @param \Drupal\Core\Utility\LinkGeneratorInterface $link_generator
   *   The link generator service.
   * @param \Drupal\breakpoint\BreakpointManager $breakpoint_manager
   *   The breakpoint manager service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, AccountInterface $current_user, EntityStorageInterface $image_style_storage, EntityStorageInterface $responsive_image_style_storage, LinkGeneratorInterface $link_generator, BreakpointManager $breakpoint_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->currentUser = $current_user;
    $this->imageStyleStorage = $image_style_storage;
    $this->responsiveImageStyleStorage = $responsive_image_style_storage;
    $this->linkGenerator = $link_generator;
    $this->breakpointManager = $breakpoint_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('current_user'),
      $container->get('entity_type.manager')->getStorage('image_style'),
      $container->get('entity_type.manager')->getStorage('responsive_image_style'),
      $container->get('link_generator'),
      $container->get('breakpoint.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'responsive_image_style' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $responsive_image_options = [];
    $responsive_image_styles = $this->responsiveImageStyleStorage->loadMultiple();
    uasort($responsive_image_styles, '\Drupal\responsive_image\Entity\ResponsiveImageStyle::sort');
    if ($responsive_image_styles && !empty($responsive_image_styles)) {
      foreach ($responsive_image_styles as $machine_name => $responsive_image_style) {
        if ($responsive_image_style->hasImageStyleMappings()) {
          $responsive_image_options[$machine_name] = $responsive_image_style->label();
        }
      }
    }

    $elements['responsive_image_style'] = [
      '#title' => $this->t('Responsive image style', [], ['context' => 'Wedia']),
      '#type' => 'select',
      '#default_value' => $this->getSetting('responsive_image_style') ?: NULL,
      '#required' => TRUE,
      '#options' => $responsive_image_options,
      '#description' => [
        '#markup' => $this->linkGenerator->generate($this->t('Configure Responsive Image Styles', [], ['context' => 'Wedia']), new Url('entity.responsive_image_style.collection')),
        '#access' => $this->currentUser->hasPermission('administer responsive image styles'),
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $responsive_image_style = $this->responsiveImageStyleStorage->load($this->getSetting('responsive_image_style'));
    if ($responsive_image_style) {
      $summary[] = $this->t('Responsive image style: @responsive_image_style', ['@responsive_image_style' => $responsive_image_style->label()], ['context' => 'Wedia']);
    }
    else {
      $summary[] = $this->t('Select a responsive image style.', [], ['context' => 'Wedia']);
    }

    return array_merge($summary, parent::settingsSummary());
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $cache_tags = [];

    // Early opt-out if the field is empty.
    if (empty($items)) {
      return $elements;
    }

    $responsive_image_style = $this->responsiveImageStyleStorage->load($this->getSetting('responsive_image_style'));
    $image_styles = $this->imageStyleStorage->loadMultiple($responsive_image_style->getImageStyleIds());

    // Collect cache tags to be added for each item in the field.
    $cache_tags = Cache::mergeTags($cache_tags, $responsive_image_style->getCacheTags());
    foreach ($image_styles as $image_style) {
      $cache_tags = Cache::mergeTags($cache_tags, $image_style->getCacheTags());
    }

    foreach ($items as $delta => $item) {
      $image = [];

      // Generate fallback image.
      $fallback = $responsive_image_style->getFallbackImageStyle();
      if ($fallback) {
        $image['fallback'] = $this->applyImageStyle($item, $image_styles[$fallback]);
      }
      // Generate sources for all brakpoints.
      $breakpoints = array_reverse($this->breakpointManager->getBreakpointsByGroup($responsive_image_style->getBreakpointGroup()));
      foreach ($responsive_image_style->getKeyedImageStyleMappings() as $breakpoint_id => $multipliers) {
        if (isset($breakpoints[$breakpoint_id])) {
          /*
           * TO DO: Extend to all multipliers.
           * Current functionality is limited to 1x multiplier.
           */
          $multiplier = reset($multipliers);
          switch ($multiplier['image_mapping_type']) {
            case 'sizes':
              /*
               * TO DO: Extend to sizes.
               * Currently sizes are not supported.
               */
              break;

            case 'image_style':
              $image['sources'][$breakpoint_id] = $this->applyImageStyle($item, $image_styles[$multiplier['image_mapping']], $multiplier['multiplier']);
              break;
          }
          $image['sources'][$breakpoint_id]['media'] = trim($breakpoints[$breakpoint_id]->getMediaQuery());
        }
      }

      // Render element.
      $elements[$delta] = [
        '#theme' => 'wedia_responsive_image_formatter',
        '#image' => $image,
        '#responsive_image_style_id' => $responsive_image_style ? $responsive_image_style->id() : '',
        '#cache' => [
          'tags' => $cache_tags,
        ],
      ];
    }

    return $elements;
  }

  /**
   * Get image attributes values.
   *
   * @param \Drupal\wedia\Plugin\Field\FieldType\WediaImageField $image
   *   The image object.
   * @param \Drupal\image\Entity\ImageStyle $image_style
   *   The image style.
   * @param string $multiplier
   *   The multiplier string value.
   *
   * @return array
   *   Returns image attributes array.
   */
  protected function applyImageStyle(WediaImageField $image, ImageStyle $image_style, string $multiplier = ''): array {
    $item = $image->getValue();
    $attributes = [
      'alt' => $item['alt'],
      'uri' => $item['uri'],
      'width' => $item['width'] ?: NULL,
      'height' => $item['height'] ?: NULL,
    ];

    // Only apply for images that use default template.
    $effects = $image_style->getEffects()->getConfiguration();
    if (empty($effects) || !str_contains($attributes['uri'], '?option=default')) {
      return $attributes;
    }

    // Apply Wedia effects.
    foreach ($effects as $effect) {
      /*
       * TO DO: Logic when multiple effects added.
       * Currently only last effect is applied.
       */
      $attributes = $this->applyWediaEffect($attributes, $effect);
    }

    // Add multiplier.
    $attributes['uri'] .= (!empty($multiplier) ? " {$multiplier}" : '');

    return $attributes;
  }

  /**
   * Update an existing GET parameter in a URL, or add it if no exist.
   *
   * @param array $attributes
   *   The image attributes.
   * @param array $effect
   *   The effect to apply.
   *
   * @return array
   *   The altered attributes.
   */
  protected function applyWediaEffect($attributes, $effect): array {
    $anchors = [
      'left-top' => ['x' => 0, 'y' => 0],
      'center-top' => ['x' => 0.5, 'y' => 0],
      'right-top' => ['x' => 1.0, 'y' => 0],
      'right-center' => ['x' => 1.0, 'y' => 0.5],
      'center-center' => ['x' => 0.5, 'y' => 0.5],
      'right-center' => ['x' => 1.0, 'y' => 0.5],
      'left-bottom' => ['x' => 0, 'y' => 1.0],
      'center-bottom' => ['x' => 0.5, 'y' => 1.0],
      'right-bottom' => ['x' => 1.0, 'y' => 1.0],
    ];
    $effects = [];
    /*
     * TO DO: Resize, Convert, Desaturate effects.
     * Not sure if all supported by Wedia.
     */
    switch ($effect['id']) {
      case 'image_crop':
        $effects[] = [
          't' => 'crop',
          'fx' => $anchors[$effect['data']['anchor']]['x'],
          'fy' => $anchors[$effect['data']['anchor']]['y'],
          'outputWidth' => $effect['data']['width'],
          'outputHeight' => $effect['data']['height'],
        ];
        break;

      // Scale and resize are treated the same in wedia.
      case 'image_scale':
      case 'image_resize':
        $effects[] = [
          't' => 'resize',
          'width' => $effect['data']['width'],
          'height' => $effect['data']['height'],
        ];
        break;

      case 'image_scale_and_crop':
        // First we need to resize image to percentage.
        if ($effect['data']['width'] < $attributes['width']) {
          $effects[] = [
            't' => 'resize',
            'width' => round($effect['data']['width'] / $attributes['width'], 2),
          ];
        }
        // After resize apply crop with focus points.
        if (($effect['data']['width'] / $attributes['width']) * $attributes['height'] > $effect['data']['height']) {
          $effects[] = [
            't' => 'crop',
            'fx' => $anchors[$effect['data']['anchor']]['x'],
            'fy' => $anchors[$effect['data']['anchor']]['y'],
            'outputHeight' => $effect['data']['height'],
          ];
        }
        break;
    }
    // Remove all previous effect.
    $attributes['uri'] = strtok($attributes['uri'], '?');
    // Apply effects.
    if (!empty($effects)) {
      for ($i = 0; $i < count($effects); $i++) {
        $separator = (!$i ? '?' : '&');
        $attributes['uri'] .= $separator . http_build_query($effects[$i]);
      }
    }
    $attributes['width'] = $effect['data']['width'];
    $attributes['height'] = $effect['data']['height'];

    return $attributes;
  }

}
