<?php

namespace Drupal\whatlinkshere\Commands;

use Drush\Commands\DrushCommands;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides Drush commands for What Links Here.
 */
final class WhatlinkshereCommands extends DrushCommands {

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct();
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Scan for inbound node references to one or more nodes.
   *
   * @command whatlinkshere:scan
   * @aliases wlhs:scan,whatlinkshere-scan
   * @arg nids Optional. One or more node IDs. You may pass a single ID, or a single comma/space-separated list (e.g. "1,2,3" or "1 2 3").
   * @usage drush whatlinkshere:scan
   *   Scan references for all nodes.
   * @usage drush whatlinkshere:scan 1
   *   Scan references to node 1.
   * @usage drush whatlinkshere:scan "1,2,3"
   *   Scan references to nodes 1, 2 and 3.
   */
  public function scan($nids = null): int {
    $target_nids = $this->normalizeNids($nids);

    if (empty($target_nids)) {
      // No IDs supplied: scan all nodes.
      $target_nids = $this->entityTypeManager->getStorage('node')
        ->getQuery()
        ->accessCheck(FALSE)
        ->execute();
      $this->io()->title('No node IDs provided – scanning all nodes.');
    }
    else {
      $this->io()->title(sprintf(
        'Scanning %d node(s): %s',
        count($target_nids),
        implode(', ', $target_nids)
      ));
    }

    return $this->processNodes($target_nids);
  }

  /**
   * Perform the actual scan work for the provided node IDs.
   *
   * This method intentionally exists to satisfy previous calls to
   * $this->processNodes() and to keep the logic testable.
   */
  protected function processNodes(array $target_nids) : int {
    if (empty($target_nids)) {
      $this->io()->warning('Nothing to do.');
      return self::EXIT_SUCCESS;
    }

    /** @var \Drupal\node\NodeStorageInterface $node_storage */
    $node_storage = $this->entityTypeManager->getStorage('node');

    // Load all bundles up front so we can examine their field definitions.
    $bundles = array_keys($this->entityTypeManager->getStorage('node_type')->loadMultiple());

    // Build a list of all node-reference fields across ALL node bundles.
    $node_reference_fields = [];
    /** @var \Drupal\Core\Entity\EntityFieldManagerInterface $efm */
    $efm = \Drupal::service('entity_field.manager');

    foreach ($bundles as $bundle) {
      $defs = $efm->getFieldDefinitions('node', $bundle);

      foreach ($defs as $field_name => $def) {
        if ($def instanceof FieldDefinitionInterface) {
          $type = $def->getType();
          $target = $def->getSetting('target_type');
          // Accept regular entity_reference and ER revisions if they target nodes.
          if (($type === 'entity_reference' || $type === 'entity_reference_revisions') && $target === 'node') {
            $node_reference_fields[$bundle][] = $field_name;
          }
        }
      }
    }

    if (empty($node_reference_fields)) {
      $this->io()->warning('No node-reference fields found on any node bundles.');
      return self::EXIT_SUCCESS;
    }

    $count_scanned = 0;
    $count_found_total = 0;

    // Process in chunks to avoid memory spikes.
    $chunks = array_chunk($target_nids, 50);
    foreach ($chunks as $chunk) {
      /** @var \Drupal\node\NodeInterface[] $nodes */
      $nodes = $node_storage->loadMultiple($chunk);

      foreach ($nodes as $node) {
        $count_scanned++;
        $referencers = $this->findReferencingNodeIds($node, $node_reference_fields);
        $count_found_total += count($referencers);

        if ($referencers) {
          $this->io()->writeln(sprintf(
            'Node %d (%s) is referenced by: %s',
            $node->id(),
            $node->bundle(),
            implode(', ', $referencers)
          ));
        }
        else {
          $this->io()->writeln(sprintf('Node %d (%s) has no inbound references.', $node->id(), $node->bundle()));
        }
      }
    }

    $this->io()->success(sprintf(
      'Done. Scanned %d node(s); found %d referencing node(s) total.',
      $count_scanned,
      $count_found_total
    ));

    return self::EXIT_SUCCESS;
  }

  /**
   * Find IDs of nodes that reference the given node.
   *
   * @param \Drupal\node\NodeInterface $target
   *   The node being referenced.
   * @param array<string, string[]> $node_reference_fields
   *   Map of bundle => array of field names that reference nodes.
   *
   * @return int[]
   *   Node IDs that reference $target.
   */
  protected function findReferencingNodeIds(NodeInterface $target, array $node_reference_fields) : array {
    $nids = [];

    foreach ($node_reference_fields as $bundle => $fields) {
      foreach ($fields as $field_name) {
        // Query nodes where the field contains the target node ID.
        $query = $this->entityTypeManager->getStorage('node')
          ->getQuery()
          ->condition('type', $bundle)
          ->condition($field_name, (int) $target->id())
          ->accessCheck(FALSE);
        $result = $query->execute();

        if (!empty($result)) {
          $nids = array_merge($nids, array_map('intval', array_values($result)));
        }
      }
    }

    sort($nids, SORT_NUMERIC);
    $nids = array_values(array_unique($nids));

    // Exclude self-reference if present.
    $nids = array_values(array_diff($nids, [(int) $target->id()]));

    return $nids;
  }

  /**
   * Normalise CLI args into a clean integer NID list.
   */
  protected function normalizeNids(?string $input): array {
    if ($input === null || $input === '') {
      return [];
    }

    $ids = [];
    foreach (explode(',', $input) as $part) {
      $part = trim($part);
      if (!empty($part)) {
        $ids[] = (int) $part;
      }
    }

    $ids = array_values(array_unique($ids));
    sort($ids, SORT_NUMERIC);

    return $ids;
  }

}
