<?php

namespace Drupal\whatsapp_bubble\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a WhatsApp Bubble block.
 *
 * @Block(
 *   id = "whatsapp_bubble_block",
 *   admin_label = @Translation("Whatsapp Floating Bubble")
 * )
 */
class WhatsappBubbleBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Admin context service.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected AdminContext $adminContext;

  /**
   * Constructs a new WhatsappBubbleBlock instance.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, AdminContext $admin_context) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->adminContext = $admin_context;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('router.admin_context')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Do not render on admin routes to match page-bottom behavior.
    if ($this->adminContext->isAdminRoute()) {
      return [];
    }

    $configs = $this->configFactory->get('whatsapp_bubble.config');

    $alignment = $configs->get('alignment') ?: 'right';
    $valignment = $configs->get('valignment') ?: 'bottom';
    $number = $configs->get('phone_number') ?: '';
    $message = $configs->get('message') ?: '';
    $inverse = $configs->get('is_inverse') ? 'inverse' : '';
    $message = urlencode($message);

    return [
      '#theme' => 'wab',
      '#phone_number' => $number,
      '#message' => $message,
      '#attributes' => [
        'class' => ['whatsapp-bubble', $alignment, $valignment, $inverse],
      ],
      '#attached' => [
        'library' => ['whatsapp_bubble/main'],
      ],
      '#cache' => [
        'tags' => ['config:whatsapp_bubble.config'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $is_enabled = (bool) $this->configFactory->get('whatsapp_bubble.config')->get('is_enabled');
    if ($is_enabled) {
      $url = Url::fromRoute('whatsapp_bubble.whatsapp_bubble_config_form')->toString();
      $form['whatsapp_bubble_warning'] = [
        '#type' => 'item',
        '#title' => $this->t('Potential duplicate bubble'),
        '#markup' => $this->t('Global WhatsApp bubble injection is currently enabled. If you also place this block, two bubbles may be visible on the same page. You can disable the global bubble in the <a href=":url">module settings</a>.', [
          ':url' => $url,
        ]),
        '#weight' => -10,
      ];
    }

    return $form;
  }

}
