<?php

namespace Drupal\whatsapp_bubble\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\AdminContext;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a WhatsApp Button block.
 *
 * @Block(
 *   id = "whatsapp_button_block",
 *   admin_label = @Translation("Whatsapp Button")
 * )
 */
class WhatsappButtonBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Admin context service.
   */
  protected AdminContext $adminContext;

  /**
   * Constructs a new WhatsappButtonBlock instance.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, AdminContext $admin_context) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->adminContext = $admin_context;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('router.admin_context')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'small_text' => 'Contact us via',
      'override_phone' => FALSE,
      'phone_number' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Do not render on admin routes (consistency with other block/template).
    if ($this->adminContext->isAdminRoute()) {
      return [];
    }

    $configs = $this->configFactory->get('whatsapp_bubble.config');
    $number = $configs->get('phone_number') ?: '';
    if (!empty($this->configuration['override_phone']) && !empty($this->configuration['phone_number'])) {
      $number = $this->configuration['phone_number'];
    }

    return [
      '#theme' => 'wab_button',
      '#phone_number' => $number,
      '#small_text' => $this->configuration['small_text'] ?? 'Contact us via',
      '#attached' => [
        'library' => ['whatsapp_bubble/main'],
      ],
      '#cache' => [
        'tags' => ['config:whatsapp_bubble.config'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    $form['small_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Small text label'),
      '#description' => $this->t('Text displayed above “WhatsApp” inside the button.'),
      '#default_value' => $this->configuration['small_text'] ?? 'Contact us via',
      '#required' => FALSE,
    ];

    $global_number = (string) $this->configFactory->get('whatsapp_bubble.config')->get('phone_number') ?: '';
    $form['override_phone'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override phone number for this block'),
      '#default_value' => !empty($this->configuration['override_phone']),
    ];
    $form['phone_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone Number'),
      '#description' => $this->t('WhatsApp destination number for this block. Leave empty to use the number configured in the global settings (@number).', ['@number' => $global_number]),
      '#default_value' => $this->configuration['phone_number'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="settings[override_phone]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['small_text'] = $form_state->getValue('small_text');
    $this->configuration['override_phone'] = (bool) $form_state->getValue('override_phone');
    $this->configuration['phone_number'] = $form_state->getValue('phone_number') ?? '';
  }

}
