<?php

declare(strict_types=1);

namespace Drupal\whatsapp_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Url;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\telephone\Plugin\Field\FieldFormatter\TelephoneLinkFormatter;

/**
 * Plugin implementation of the 'Whatsapp Number' formatter.
 *
 * @FieldFormatter(
 *   id = "whatsapp_number",
 *   label = @Translation("Whatsapp Number"),
 *   field_types = {
 *     "telephone"
 *   }
 * )
 */
final class WhatsappNumberFormatter extends TelephoneLinkFormatter {

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    // Keep the same logic as the core telephone number, only changing the URL.
    $element = parent::viewElements($items, $langcode);

    foreach ($items as $delta => $item) {
      // Clean the data to make sure the number somewhat complies with the
      // format described on
      // https://faq.whatsapp.com/5913398998672934#create-your-own-link.
      // This, however, does not guarantee the final result will be correct. For
      // that, you need input validation on the telephone field.
      // @todo This is something that could use a unit test, and smarter regex.
      // Strip leading zeros or zero wrapped in parentheses as people often do.
      $value = preg_replace('/^\+?0*|(\(0+\))*/', '', (string) $item->value);
      // Remove any remaining non-digit characters.
      $value = preg_replace('/\D/', '', $value);

      $element[$delta]['#url'] = Url::fromUri("https://wa.me/$value");
      $element[$delta]['#options']['attributes']['target'] = '_blank';
    }

    return $element;
  }

}
