<?php

namespace Drupal\wisski_adapter_sparql11_pb\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\LocalRedirectResponse;
use Drupal\Core\Url;
use Drupal\wisski_salz\Plugin\wisski_salz\Engine\Sparql11Engine;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;


enum QueryType {
  case QUERY;
  case UPDATE;
}

/**
 * Generic controller for the WissKI API.
 */
class Sparql11EndpointController extends ControllerBase implements ContainerInjectionInterface {


  /**
   * The route match interface for getting the current route.
   *
   * @var Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a Drupal\wisski_adapter_sparql11_pb/Controller\Sparql11EndpointController object.
   *
   * @param Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The EntityTypeManager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  public function query(Request $request, string $adapter_id) {
    return $this->handleRequest($request, $adapter_id, QueryType::QUERY);
  }

  public function update(Request $request, string $adapter_id) {
    return $this->handleRequest($request, $adapter_id, QueryType::UPDATE);
  }

  public function handleRequest(Request $request, string $adapter_id, QueryType $query_type) {
    $method = $request->getMethod();

    // Read out accept header
    $accept = explode(',', $request->headers->get("accept", "text/html"));
    $accept = array_map("trim", $accept);

    // If HTML is requested redirect to the forms based on request method.
    if (in_array("text/html", $accept)) {
      match ($query_type) {
        QueryType::QUERY => $redirect_route = 'entity.wisski_salz_adapter.query',
        QueryType::UPDATE => $redirect_route = 'entity.wisski_salz_adapter.update',
      };

      $query_params = $request->query->all();
      $url = Url::fromRoute($redirect_route, ["wisski_salz_adapter" => $adapter_id], ["query" => $query_params]);
      return new LocalRedirectResponse($url->toString());
    }

    /** @var \Drupal\wisski_salz\Entity\Adapter | NULL */
    $adapter = $this->entityTypeManager()->getStorage("wisski_salz_adapter")->load($adapter_id);
    if (!$adapter) {
      return  new Response(
        $this->t("No such adapter: '@adapter'", ["@adapter" => $adapter_id]),
        404,
        ["Content-Type" => "text/plain"],
      );
    }

    /** @var \Drupal\wisski_salz\Plugin\wisski_salz\Engine\Sparql11Engine */
    $engine = $adapter->getEngine();
    $engine_config = $engine->getConfiguration();

    match ($query_type) {
      QueryType::QUERY => $url = $engine_config["read_url"] ?? NULL,
      QueryType::UPDATE => $url = $engine_config["write_url"] ?? NULL,
    };

    // Convert the weird Guzzle format headers into array
    $request_headers = [];
    foreach ($request->headers->all() as $key => $value) {
      $request_headers[$key] = implode(', ', $value);
    }

    // Pass these headers through to the triplestore
    $passthrough_request_headers = [
      "Accept",
      "Content-Type"
    ];
    $filtered_request_headers = $this->filterHeaders($request_headers, $passthrough_request_headers);
    // Add Authorization header
    $filtered_request_headers = array_merge($this->getAuthHeader($engine));

    $request_params = [
      "headers" => $filtered_request_headers,
      "query" => $request->query->all(),
    ];

    // Add body if POST request.
    if ($method == "POST") {
      $request_params["body"] = $request->getContent();
    }

    try {
      $ts_response = \Drupal::httpClient()->request($method, $url, $request_params);
    } catch (\GuzzleHttp\Exception\RequestException $exception) {
      return new Response($exception->getMessage(), $exception->getCode());
    }
    $body = $ts_response->getBody();

    // Pass on these headers in the response.
    $passthrough_response_headers = [
      "Content-Type",
    ];
    $response_headers = $this->filterHeaders($ts_response->getHeaders(), $passthrough_response_headers);

    $response = new Response($body, $ts_response->getStatusCode(), $response_headers);
    return $response;

  }


  /**
   * Get number of triples in a particular engine.
   *
   * @param Sparql11Engine $engine
   *
   * @return integer
   *  The number of triples
   */
  public static function getRepoSize(Sparql11Engine $engine): int {
    // TODO: this should probably be a request to /repositories/{repo_id}/size instead of SPARQL query.
    // Question is if all TS solutions support this route...
    $count_query = "SELECT (COUNT(*) as ?count) WHERE { { GRAPH ?g { ?s ?p ?o } } UNION { ?s ?p ?o }}";
    try {
      $results = $engine->directQuery($count_query);
    }
    catch (\EasyRDF\Exception) {
      return -1;
    }

    foreach ($results as $result) {
      return $result->count->getValue();
    }
    return -1;
  }


  /**
   * Get the Authorization header for an engine.
   *
   * @param Sparql11Engine $engine
   *  The engine.
   *
   * @return array
   *  The header as an associative array: ["Authorization" => VALUE] or empty array.
   */
  public static function getAuthHeader(Sparql11Engine $engine): array {
    $engine_config = $engine->getConfiguration();
    if ($engine_config["use_token"] && $engine_config["token"]) {
      return ["Authorization" => "Token " . $engine_config["token"]];
    }
    elseif (!$engine_config["use_token"] && $engine_config["header"]) {
      return ["Authorization" => "Basic " . $engine_config["header"]];
    }
    else {
      return [];
    }
  }

  /**
   * Filter headers.
   *
   * @param $headers
   *  Incoming headers to filter.
   * @param array $allowed_headers
   *  List of allowed headers.
   *
   * @return array
   *  The filtered headers.
   */
  private function filterHeaders($headers, array $allowed_headers): array {
    $new_headers = [];
    foreach ($headers as $key => $value) {
      // Inore case.
      if (in_array(strtolower($key), array_map('strtolower', $allowed_headers))) {
        $new_headers[$key] = $value;
      }
    }
    return $new_headers;
  }

}
