<?php

namespace Drupal\wisski_adapter_sparql11_pb\Form;

use Drupal\file\Entity\File;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\wisski_adapter_sparql11_pb\Controller\Sparql11EndpointController;


class Sparql11EndpointImportForm extends FormBase {

  /**
   * Mapping file extnetions to MIME Types.
   * This is needed since the Drupal MimetypeGuesser does not always output the correct types.
   */
  const MIME_TYPE_MAP = [
    "rdf" => "application/rdf+xml",
    "xml" => "application/rdf+xml",
    "owl" => "application/rdf+xml",
    "ttl" => "text/tutle",
    "nt" => "application/n-triples",
    "nq" => "application/n-quads",
    "trig" => "application/trig",
    "jsonld" => "application/ld+json"
  ];

  /**
   * @Inheritdoc
   */
  public function getFormId() {
    return "wisski_adapter_sparql11_pb_endpoint_import_form";
  }

  /**
   * @Inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state, $wisski_salz_adapter = NULL) {
    $endpoint_id = $wisski_salz_adapter;

    // Display warning if non existing adapter was requested
    /** @var \Drupal\wisski_salz\AdapterInterface[] */
    $adapters = \Drupal::entityTypeManager()->getStorage("wisski_salz_adapter")->loadMultiple();
    if (!in_array($endpoint_id, array_keys($adapters))) {
      $this->messenger()->addWarning($this->t("No such adapter '@requested'! Available adapters are: @adapters", [
        "@adapters" => join(", ", array_keys($adapters)),
        "@requested" => $endpoint_id
      ]));
      return $form;
    }

    $adapter = $adapters[$endpoint_id];
    $engine = $adapter->getEngine();
    $engine_config = $engine->getConfiguration();

    // Save engine for submit
    $form["engine"] = [
      "#type" => "value",
      "#value" => $engine,
    ];

    $form["file"] = [
      "#type" => "managed_file",
      "#title" => $this->t("Upload a file to import"),
      "#description" => $this->t("Valid formats: @formats", ["@formats" => join(", ", array_keys(self::MIME_TYPE_MAP))]),
      "#upload_validators" => [
        "FileExtension" => [
          "extensions" => join(" ", array_keys(self::MIME_TYPE_MAP)),
        ],
      ],
    ];

    $form["graph"] = [
      "#type" => "textfield",
      "#title" => $this->t("Graph"),
      "#description" => $this->t("The graph into which the uploaded file is imported to"),
      "#default_value" => $engine_config["default_graph"],
    ];

    $form["submit"] = [
      "#type" => "submit",
      "#value" => $this->t("Import"),
    ];

    $form["#title"] = "Import into $endpoint_id Endpoint";
    return $form;
  }

  /**
   * @Inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_file = $form_state->getValue("file", 0);
    $file = File::load($form_file[0]);

    $uri = $file->getFileUri();
    $content = file_get_contents($uri);

    // Get params from engine_config
    /** @var \Drupal\wisski_salz\EngineInterface */
    $engine = $form_state->getValue("engine");
    $write_url = $engine->getConfiguration()["write_url"];

    // Add context to URL if set
    $graph = $form_state->getValue("graph");
    $url = $write_url;
    if ($graph) {
      $encoded_graph = urlencode("<" . $graph . ">");
      $url .= "?context=$encoded_graph";
    }

    // Guess the mimetype and set headers
    // Sadly the \Drupal::service("file.mime_type.guesser") does not output
    // the correct types for some extensions so we have to do it ourselves
    $filename = $file->getFilename();
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $mime_type = self::MIME_TYPE_MAP[$extension];
    $headers = ["Content-Type" => $mime_type];
    $headers = array_merge($headers, Sparql11EndpointController::getAuthHeader($engine));

    $size_before = Sparql11EndpointController::getRepoSize($engine);

    try {
      \Drupal::httpClient()->post($url, [
          "headers" => $headers,
          "body" => $content
        ]
      );
    } catch (\GuzzleHttp\Exception\RequestException $exception) {
      $this->messenger()->addError($exception->getMessage());
      return;
    }

    // Display repo size
    $size_after = Sparql11EndpointController::getRepoSize($engine);
    $message = $this->t("The number of statements did not change");
    $diff = $size_after - $size_before;
    if ($diff > 0) {
      $message = $this->t("Imported @diff statement(s)", ["@diff" => $diff]);
    }
    $this->messenger()->addMessage($message);

    // we want to get back where we came from
    $form_state->setRebuild(TRUE);
    $form_state->disableRedirect(TRUE);
  }
}
