<?php

namespace Drupal\wisski_adapter_sparql11_pb\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\wisski_adapter_sparql11_pb\Controller\Sparql11EndpointController;

class Sparql11EndpointUpdateForm extends FormBase {

  /**
   * @Inheritdoc
   */
  public function getFormId() {
    return "wisski_adapter_sparql11_pb_endpoint_update_form";
  }

  /**
   * @Inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state, $wisski_salz_adapter = NULL) {

    $adapter_id = $wisski_salz_adapter;

    // Display warning if non existing adapter was requested
    $adapters = \Drupal::entityTypeManager()->getStorage("wisski_salz_adapter")->loadMultiple();
    if (!in_array($adapter_id, array_keys($adapters))) {
      $this->messenger()->addWarning($this->t("No such adapter '@requested'! Available adapters are: @adapters", [
        "@adapters" => join(", ", array_keys($adapters)),
        "@requested" => $adapter_id
      ]));
      return $form;
    }

    $query = \Drupal::request()->query->get("query") ?? "";

    $query_fs = $form_state->getUserInput();
    if (!empty($query_fs["query"])) {
      $query = $query_fs["query"];
    }

    $form["#title"] = "Update $adapter_id Endpoint";

    $form["query"] = array(
      "#type" => "textarea",
      "#attributes" => [
        "class" => ["codemirror"],
      ],
      "#value" => $query
    );

    $form["submit"] = [
      "#type" => "submit",
      "#value" => $this->t("Execute Query"),
    ];

    $form["#attached"]["library"][] = "wisski_adapter_sparql11_pb/codemirror";

    if (empty($query)) {
      $form["query"]["#value"] = "INSERT DATA { <http://there.is> <http://nothing.to> <http://see.here> }";
      return $form;
    }

    /** @var \Drupal\wisski_salz\Entity\Adapter */
    $adapter = $adapters[$adapter_id];
    /** @var \Drupal\wisski_salz\Plugin\wisski_salz\Engine\Sparql11Engine */
    $engine = $adapter->getEngine();
    $size_before = Sparql11EndpointController::getRepoSize($engine);
    if ($size_before < 0) {
      // TODO: display some errror... this should not happen or only when the endpoint is not reachable.
      return $form;
    }
    try {
      $engine->directUpdate($query);
    } catch (\EasyRDF\Exception $e) {
      // TODO: For this to work it requires the error to be thrown in directQuery and not caught...
      // Check with Mark if that is possible or if WissKI completely breaks if this error gets out of directQuery.
      // If that is not possible at least put this message into the messenger in directQuery.
      $this->messenger()->addError($e->getMessage());
      return $form;
    }

    // We expect this to work if the previous size checks and update query worked...
    $size_after = Sparql11EndpointController::getRepoSize($engine);


    // Query results
    $form["results"] = [
      "#type" => "container"
    ];

    $message = $this->t("The number of statements did not change");
    $diff = $size_after - $size_before;
    if ($diff > 0) {
      $message = $this->t("Added @diff statement(s)", ["@diff" => $diff]);
    } elseif ($diff < 0) {
      $diff *= -1;
      $message = $this->t("Removed @diff statement(s)", ["@diff" => $diff]);
    }

    $this->messenger()->addMessage($message);

    return $form;
  }



  /**
   * @Inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Put the query into the query params in order for the pager to work
    $query = $form_state->getValue("query");
    if ($query) {
      \Drupal::request()->query->set("query", $query);
    }

    $form_state->setValue("query", $query);

    // we want to get back where we came from
    $form_state->setRebuild(TRUE);
    $form_state->disableRedirect(TRUE);
  }
}
