<?php

namespace Drupal\wisski_adapter_sparql11_pb\Query;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\wisski_pathbuilder\Entity\WisskiPathEntity;
use Drupal\Core\Entity\Query\ConditionInterface;
use Drupal\wisski_salz\AdapterHelper;
use Drupal\wisski_salz\Query\ConditionAggregate;
use Drupal\wisski_salz\Query\WisskiQueryBase;

/**
 *
 */
class Query extends WisskiQueryBase {

  /**
   * Holds the pathbuilders that this query object is responsible for.
   * This variable should not be accessed directly, use $this->getPbs()
   * instead.
   */
  private $pathbuilders = NULL;

  /**
   * Add a variable for dependent query parts.
   */
  private $dependent_parts = [];

  /**
   * A counter used for naming variables in multi-path sparql queries.
   *
   * @var int
   */
  protected $varCounter = 0;

  /**
   * A string of vars to order by in this query.
   *
   * @var array
   */
  protected $orderby = "";

  /**
   *
   */
  public function setOrderBy($orderby) {
    $this->orderby = $orderby;
  }

  /**
   * A function to add dependent parts
   * typically a SERVICE-string like:
   * SERVICE <http...serviceurl> { ?s ?p ?o }
   * hopefully it uses the correct variables....
   */
  public function addDependentParts($parts) {
    $this->dependent_parts[] = $parts;
  }

  /**
   * A function to set dependent parts
   * via an array
   */
  public function setDependentParts($parts) {
    $this->dependent_parts = $parts;
  }

  /**
   * Get an array of query parts to build up the dependent queries.
   * currently it contains:
   * - The where string
   * - The eids (values-part)
   * - The order string
   */
  public function getQueryParts() {
    // @todo Iterate over the tree in this::Groups.
    [$where_clause, $entity_ids] = $this->makeQueryConditions($this->condition);

    return ["where" => $where_clause, "eids" => $entity_ids, "order" => $this->orderby];
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    // $micro = microtime(TRUE);
    // dpm("yay!");
    // dpm($this);
    // NOTE: this is not thread-safe... shouldn't bother!
    $this->varCounter = 0;

    // dpm($this->condition->conditions(),$this->getEngine()->adapterId().': '.__METHOD__);
    // wisski_tick();
    // Speed hack:
    // if there is a bundle-condition
    // and there are other field-conditions concerning wisski-fields
    // then these are probably more specific, so we can skip the bundle condition.
    // dpm($this->isPathQuery(), "yep?");
    // dpm($this->condition, "condition?");.
    $bundlekey = -1;

    $pathkey = -1;

    if ($this->condition->getConjunction() == "AND") {
      // Iterate through all conditions.
      foreach ($this->condition->conditions() as $key => $cond) {

        $field = $cond['field'];

        if (is_object($field)) {
          // By Mark: This is a problem
          // if this takes place it is some kind of grouping
          // so it should be iterated.
          // for now we just skip it but we should go in there and fetch the things from it!
          // dpm($field, "this is an object, but it should be a string");.
          foreach ($field->conditions() as $keynew => $condnew) {

            $key = $keynew;
            $cond = $condnew;

            $field = $cond['field'];

          }
        }

        // if we have nothing anymore then it is a problem
        // also if it is still a condition, then it is more complex...
        if (empty($field) || is_object($field)) {
          continue;
        }

        if ($field == "bundle") {
          // Store the bundle key if there is a bundle condition.
          $bundlekey = $key;
        }

        // Get all pbs and their ids.
        $pbs = $this->getPbs();

        $pbids = array_keys($pbs);

        // If it is a pb field, it has a dot in it.
        if (strpos($field, '.') !== FALSE) {
          $pb_and_path = explode(".", $field);

          // Is it really a pb field?
          if (count($pb_and_path) != 2) {
            // Bad encoding! can't handle
            // drupal_set_message(new \Drupal\Core\StringTranslation\TranslatableMarkup('Bad pathbuilder and path id "%id" in entity query condition', ['%id' => $field]));.
            // With next condition.
            continue;
          }
          $pbid = $pb_and_path[0];

          // If the id is a pathbuilder id, it is a pathquery - so handle it accordingly!
          if (in_array($pbid, $pbids) !== FALSE) {
            $pathkey = $key;
          }
        }
      }

      // dpm($cond, "cond?");
      // if(strpos($field, '.')
    }

    // dpm($bundlekey, "bk? " . $pathkey);.
    // If we now have a bundlekey and a pathkey, we can omit the bundle-part probably.
    if ($bundlekey > -1 && $pathkey > -1) {
      $conditions = &$this->condition->conditions();

      // dpm("I unset" . $bundlekey);.
      unset($conditions[$bundlekey]);
    }

    // Compile the condition clauses into
    // sparql graph patterns and
    // a list of entity ids that the pattern should be restricted to
    //    dpm($this->condition, "condition?");.
    [$where_clause, $entity_ids] = $this->makeQueryConditions($this->condition);

    // dpm($this->orderby, "order?");.
    // dpm($where_clause, "where clause in adapter query");
    // dpm($entity_ids, "eids in list");
    // dpm($this->dependent_parts, "dep");
    // If we have dependent parts, we always want to go to buildAndExecute...
    // We can only opt out if there really is nothing!
    if (empty($where_clause) && empty($entity_ids) && empty($this->dependent_parts)) {
      $return = $this->count ? 0 : [];
    }
    elseif (empty($where_clause) && empty($this->dependent_parts)) {
      // We can use the entity ids if we have no other dependencies.
      [$limit, $offset] = $this->getPager();
      if ($limit !== NULL) {
        $entity_ids = array_slice($entity_ids, $offset, $limit, TRUE);
      }
      $return = $this->count ? count($entity_ids) : array_keys($entity_ids);
    }
    elseif (empty($entity_ids) || !empty($this->dependent_parts)) {
      // By Mark: I dont know if we want to have dependent parts here....
      [$limit, $offset] = $this->getPager();
      // dpm($this->orderby, "order");.
      $return = $this->buildAndExecSparql($where_clause, NULL, $this->count, $limit, $offset, $this->orderby);
      // dpm(serialize($return), "got: ");.
      if (!$this->count) {
        $return = array_keys($return);
      }
      else {
        // dpm($return, "ret!");.
      }
    }
    // This should only happen if there are no dependent parts!!!
    else {
      // There are conditions left and found entities.
      // this can only occur if the conjunction of $this->condition is OR.
      [$limit, $offset] = $this->getPager();
      // We must not use count directly (3rd param, see above)
      $entity_ids_too = $this->buildAndExecSparql($where_clause, NULL, FALSE, $limit, $offset);
      // dpm($entity_ids_too, "too");
      // dpm($entity_ids, "too2");
      // Combine the resulting entities with the ones already found.
      // we have to OR them: an AND conjunction would have been resolved in
      // makeQueryConditions().
      // @todo Check AND! This might be wrong again (by Mark)
      $entity_ids = $this->join('OR', $entity_ids, $entity_ids_too);
      // dpm($entity_ids, "too3");
      // Now we again have to apply the pager.
      if ($limit !== NULL) {
        $entity_ids = array_slice($entity_ids, $offset, $limit, TRUE);
      }
      $return = $this->count ? count($entity_ids) : array_keys($entity_ids);
    }
    // dpm([$limit, $offset], 'pager');.
    // \Drupal::logger('query adapter ' . $this->getEngine()->adapterId())->debug('query result is {result}', array('result' => serialize($return)));
    // wisski_tick("end query with num ents:" . (is_int($return) ? $return : count($return)));
    // dpm($return, "what");
    // dpm(microtime(TRUE) - $micro, "took: ");.
    return $return;

  }

  /**
   *
   */
  public function getPager() {
    $limit = $offset = NULL;
    if (!empty($this->pager) || !empty($this->range)) {
      $limit = $this->range['length'] ? : NULL;
      $offset = $this->range['start'] ? : 0;
    }
    return [$limit, $offset];
  }

  /**
   * Gets all the pathbuilders that this query is responsible for.
   *
   * @return an array of pathbuilder objects keyed by their ID
   */
  protected function getPbs() {

    // As the pbs won't change during query execution, we cache them.
    if ($this->pathbuilders === NULL) {
      $aid = $this->getEngine()->adapterId();
      $pbids = \Drupal::service('wisski_pathbuilder.manager')->getPbsForAdapter($aid);
      // $this->pathbuilders = entity_load_multiple('wisski_pathbuilder', $pbids);
      $this->pathbuilders = \Drupal::entityTypeManager()->getStorage('wisski_pathbuilder')->loadMultiple($pbids);
    }
    return $this->pathbuilders;

  }

  /**
   * Descends the conjunction field until it finds an AND/OR string
   * If none is found, returns the $default.
   *
   * We need this function as the conditions' conjunction field may itself
   * contain a condition.
   */
  protected function getConjunction($condition, $default = 'AND') {
    $conj = $condition->getConjunction();
    if (is_object($conj) && $conj instanceof ConditionInterface) {
      return $this->getConjunction($conj, $default);
    }
    elseif (is_string($conj)) {
      $conj = strtoupper($conj);
      if ($conj == 'AND' || $conj == 'OR') {
        return $conj;
      }
    }
    return $default;
  }

  /**
   * Helper function to join two arrays of entity id => uri pairs according
   * to the query conjunction
   */
  protected function join($conjunction, $array1, $array2) {
    // Update the result set only if we really have executed a condition.
    if ($array1 === NULL) {
      return $array2;
    }
    elseif ($array2 === NULL) {
      return $array1;
    }
    elseif ($conjunction == 'AND') {
      return array_intersect_key($array1, $array2);
    }
    else {
      // OR
      // This seems to be wrong because it does renumbering
      // but the key is an eid here -> renumbering is evil!
      // return array_merge($array1, $array2);.
      return $array1 + $array2;

    }

  }

  /**
   * Recursively go through $condition tree and match entities against it.
   */
  protected function makeQueryConditions(ConditionInterface $condition) {
    // dpm("yay???");
    // These fields cannot be queried with this adapter.
    $skip_field_ids = [
    // 'langcode',
      'name',
      'preview_image',
      'status',
      'uuid',
      'uid',
      'vid',
    ];

    // Get the conjunction (AND/OR)
    $conjunction = $this->getConjunction($condition);

    // Here we collect entity ids.
    $entity_ids = NULL;
    // ... and query parts
    $query_parts = [];

    $sort_query_parts = "";

    // Fetch the bundle in advance e.g. for title generation.
    $needs_a_bundle = NULL;
    $is_a_pathquery = FALSE;
    $contributes_to_pathquery = FALSE;
    // dpm($condition, "cond");.
    foreach ($condition->conditions() as $ij => $cond) {
      $field = $cond['field'];
      $value = $cond['value'];

      if ($field instanceof ConditionInterface) {
        // We don't handle this here!
        continue;
      }

      // assertion: "$field" is a string!
      if ($field == "bundle") {
        $needs_a_bundle = current((array) $value);
      }

      // dpm($this->isPathQuery(), "ispathquery?");
      // dpm($condition, "cond?");.
      if ($this->isPathQuery() && strpos($field, '.') !== FALSE) {
        $is_a_pathquery = TRUE;
        $pb_and_path = explode(".", $field);
        // dpm($pb_and_path, "pb_and_path");.
        if (count($pb_and_path) != 2) {
          // Bad encoding! can't handle.
          \Drupal::messenger()->addStatus(new TranslatableMarkup('Bad pathbuilder and path id "%id" in entity query condition', ['%id' => $field]));
          // With next condition.
          continue;
        }
        $pbid = $pb_and_path[0];
        $pbs = $this->getPbs();

        // If this is not set the field can not contribute.
        if (!isset($pbs[$pbid])) {
          continue;
        }
        else {
          // dpm("it contributes!!!" . $pbid, "yes!");.
          $contributes_to_pathquery = TRUE;
        }
      }
    }

    // dpm(serialize($this->isPathQuery()) . " and " . serialize($contributes_to_pathquery));
    // If this is a pathquery and it does not contribute - we stop here.
    if ($is_a_pathquery && !$contributes_to_pathquery) {
      // dpm("it is a path query and it does not contribute!");
      // Get out here.
      return ['', NULL];
    }
    else {

      // $condition is actually a tree of checks that can be OR'ed or AND'ed.
      // We walk the tree and build up sparql conditions / a where clause in
      // $query_parts.
      //
      // We must handle the special case of an entity id and title/label
      // condition, which is not executed against the triple store but the RDB.
      // We keep track of these entities in $entity_ids and perform sparql
      // subqueries in case the ids and the clauses have to be mixed
      // (holds for ANDs).
      foreach ($condition->conditions() as $ij => $cond) {

        $field = $cond['field'];
        $value = $cond['value'];
        $operator = $cond['operator'];
        // Just to be sure!
        $operator = strtoupper($operator);
        // dpm(serialize($cond), "going into field");
        // wisski_tick($field instanceof ConditionInterface ? "recurse in nested condition" : "now for '".join(";",(array)$value)."' in field '$field'");
        // \Drupal::logger('query path cond')->debug("$ij::$field::$value::$operator::$conjunction");.
        // We dispatch over the field.
        if ($field instanceof ConditionInterface) {
          // This is a nested condition so we have to recurse.
          [$qp, $eids] = $this->makeQueryConditions($field);
          $entity_ids = $this->join($conjunction, $entity_ids, $eids);
          if ($entity_ids !== NULL && count($entity_ids) == 0 && $conjunction == 'AND') {
            // The condition evaluated to an empty set of entities
            // and we have to AND; so the result set will be empty.
            // The rest of the conditions can be skipped.
            return ['', []];
          }
          $query_parts[] = $qp;

        }

        elseif ($field == "eid") {
          // dpm($field, "I ask for eid!");.
          if (\Drupal::keyValue('system.schema')->get("wisski_core") >= 8006) {
            if (!empty($value) && (is_numeric($value) || is_array($value))) {
              $query_parts[] = $this->makeTSEntityIdCondition($operator, $value);
              // If it is a string or it is not fulfillable otherwise we just skip.
            }
            else {
              return ['', []];
            }
          }
          else {

            // Directly ask Drupal's entity id.
            $eids = $this->executeEntityIdCondition($operator, $value);
            $entity_ids = $this->join($conjunction, $entity_ids, $eids);
            if ($entity_ids !== NULL && count($entity_ids) == 0 && $conjunction == 'AND') {
              // The condition evaluated to an empty set of entities
              // and we have to AND; so the result set will be empty.
              // The rest of the conditions can be skipped.
              return ['', []];
            }
          }
        }
        elseif ($field == "bundle") {
          // The bundle is being mapped to pb groups.
          $query_parts[] = $this->makeBundleCondition($operator, $value);
        }
        elseif ($field == "langcode") {

          $eids = $this->executeLangcodeCondition($operator, $value, $needs_a_bundle);

          $entity_ids = $this->join($conjunction, $entity_ids, $eids);
          if ($entity_ids !== NULL && count($entity_ids) == 0 && $conjunction == 'AND') {
            return ['', []];
          }
        }
        elseif ($field == "status") {
          // dpm("yay, status!");
          // In $needs_a_bundle there is the bundle if there is something like that.
          $eids = $this->executeStatusCondition($operator, $value, $needs_a_bundle);

          $entity_ids = $this->join($conjunction, $entity_ids, $eids);
          if ($entity_ids !== NULL && count($entity_ids) == 0 && $conjunction == 'AND') {
            return ['', []];
          }

        }
        elseif ($field == "title" || $field == 'label') {
          // We treat label and title the same (there really should be no difference)
          // directly ask the title.
          // @todo we could handle the special case of title+bundle query as this
          // can be packed into one db query and not unintentionally explode the
          // intermediate result set.
          // dpm("yay!");
          $eids = $this->executeEntityTitleCondition($operator, $value, $needs_a_bundle);
          $entity_ids = $this->join($conjunction, $entity_ids, $eids);
          // dpm($entity_ids, "ids?");.
          if ($entity_ids !== NULL && count($entity_ids) == 0 && $conjunction == 'AND') {
            // The condition evaluated to an empty set of entities
            // and we have to AND; so the result set will be empty.
            // The rest of the conditions can be skipped.
            return ['', []];
          }

        }
        elseif (in_array($field, $skip_field_ids)) {
          // dpm("does not work!");
          // These fields are not supported on purpose
          // $this->missingImplMsg("Field '$field' intentionally not queryable in entity query", array('condition' => $condition));.
        }
        // For the rest of the fields we need to distinguish between field and path
        // query mode
        // .
        // @todo we should not need to distinguish between both modes as we can
        // tell them apart by the dot. This would make query more flexible and
        // allow for queries that contain both path and field conditions.
        elseif ($this->isPathQuery() || strpos($field, '.') !== FALSE) {
          // The field is actually a path so we can query it directly
          // dpm("path!!");
          // The search field id encodes the pathbuilder id and the path id:
          // decode them!
          // @todo we could omit the pb and search all pbs the contain the path.
          $pb_and_path = explode(".", $field);
          if (count($pb_and_path) != 2) {
            // dpm("bad encoding?");
            // Bad encoding! can't handle.
            \Drupal::messenger()->addStatus(('Bad pathbuilder and path id "%id" in entity query condition'));
            // With next condition.
            continue;
          }
          // dpm($pb_and_path, "pbandp?");.
          $pbid = $pb_and_path[0];
          $pbs = $this->getPbs();
          if (!isset($pbs[$pbid])) {
            // We cannot handle this path as its pb belongs to another engine's
            // pathbuilder.
            // With next condition.
            continue;
          }
          $pb = $pbs[$pbid];
          // Get the path.
          $path_id = $pb_and_path[1];
          $path = WisskiPathEntity::load($path_id);
          if (empty($path)) {
            \Drupal::messenger()->addStatus($this->t('Bad path id "%id" in entity query', ['%id' => $path_id]));
            // With next condition.
            continue;
          }

          $new_query_part = $this->makePathCondition($pb, $path, $operator, $value);
          // dpm($new_query_part, "yay!");
          // dpm($value, "val?");
          // dpm($pb, "pb");
          // dpm($path, "path");
          // dpm($operator, "op");
          // dpm($value, "val");.
          if (is_null($new_query_part)) {
            if ($conjunction == 'AND') {
              // The condition would definitely evaluate to an empty set of
              // entities and we have to AND; so the result set will be empty.
              // The rest of the conditions can be skipped.
              return ['', []];
            }
            // else: we are in OR mode so we can just skip the condition that
            // would evaluate to an empty set.
          }
          else {
            if (!empty($new_query_part)) {
              $query_parts[] = $new_query_part;
            }
          }

        }
        else {
          // The field must be mapped to one or many paths which are then queried.
          // dpm("could not make map!");.
          $new_query_part = $this->makeFieldCondition($field, $operator, $value);
          if (is_null($new_query_part)) {
            if ($conjunction == 'AND') {
              // The condition would definitely evaluate to an empty set of
              // entities and we have to AND; so the result set will be empty.
              // The rest of the conditions can be skipped.
              return ['', []];
            }
            // else: we are in OR mode so we can just skip the condition that
            // would evaluate to an empty set.
          }
          else {
            if (!empty($new_query_part)) {
              $query_parts[] = $new_query_part;
            }
          }
        }
      }
    }

    // If we have a query part that is NULL, this means that the field is not
    // supported by this adapter. If we are in AND mode, this means that the
    // whole condition is not satisfiable and we return the empty set.
    // In OR mode we can omit the query part.
    foreach ($query_parts as $i => $part) {
      if ($part === NULL) {
        if ($conjunction == 'AND') {
          // dpm($entity_ids, "return if part is null");.
          return ['', []];
        }
        // OR.
        else {
          unset($query_parts[$i]);
        }
      }
    }

    // dpm($query_parts, "qp");.
    // Flatten query parts array.
    if (empty($query_parts)) {
      $query_parts = '';
    }
    elseif (count($query_parts) == 1) {
      $query_parts = $query_parts[0];
    }
    elseif ($conjunction == 'AND') {
      $query_parts = join(' ', $query_parts);
    }
    else {
      // OR.
      $query_parts = ' {{ ' . join(' } UNION { ', $query_parts) . ' }} ';
    }

    // dpm($query_parts, "qpout");.
    // This means we had an empty condition... why should we query without a condition?!
    if (empty($query_parts)) {
      return ['', $entity_ids];
      // Return "";.
    }
    // $query_parts = " GRAPH ?somethingsb0rg { ?x0 ?this_should ?not_happen } ";
    // Handle sorting
    $sort_params = "";

    foreach ($this->sort as $sortkey => $elem) {
      // dpm($elem, "sort");
      // if($elem['field'] == "title") {
      // $select->orderBy('ngram', $elem['direction']);
      // }.
      $field = $elem['field'];

      if (strpos($field, "wisski_path_") === 0 && strpos($field, "__") !== FALSE) {

        $pb_and_path = explode("__", substr($field, 12), 2);
        if (count($pb_and_path) != 2) {
          \Drupal::messenger()->addError("Bad field id for Wisski views: $field");
        }
        else {
          $pb = \Drupal::service('entity_type.manager')->getStorage('wisski_pathbuilder')->load($pb_and_path[0]);
          if (!in_array($pb, $this->getPbs())) {
            continue;
          }
          $path = \Drupal::service('entity_type.manager')->getStorage('wisski_path')->load($pb_and_path[1]);
          $engine = $this->getEngine();
          if (!$pb) {
            \Drupal::messenger()->addError("Bad pathbuilder id for Wisski views: $pb_and_path[0]");
          }
          elseif (!$path) {
            \Drupal::messenger()->addError("Bad path id for Wisski views: $pb_and_path[1]");
          }
          else {
            // $starting_position = $pb->getRelativeStartingPosition($path, TRUE);
            $starting_position = 0;

            $vars[$starting_position] = "x0";
            $i = $this->varCounter++;
            for ($j = count($path->getPathArray()); $j > $starting_position; $j--) {
              $vars[$j] = "c{$i}_x$j";
            }
            $vars['out'] = "c{$i}_out";

            $sort_part = $this->getEngine()->generateTriplesForPath($pb, $path, "", NULL, NULL, 0, $starting_position, FALSE, '=', 'field', FALSE, $vars);

            $sort = " OPTIONAL { " . $sort_part . " } ";

            // dpm($sort_part, "sorti?");.
            // foreach($query_parts as $iter => $query_part) {.
            $query_parts = $query_parts . $sort;
            $sort_query_parts = $sort_query_parts . $sort;
            // }
            if (!empty($path->id()) && !empty($pb->getPbPath($path->id())) && $pb->getPbPath($path->id())["fieldtype"] == "decimal" || $pb->getPbPath($path->id())["fieldtype"] == "number") {
              $sort_params = $elem['direction'] . "(xsd:integer(?c{$i}_out)) ";
            }
            else {
              $sort_params = $elem['direction'] . "(STR(?c{$i}_out)) ";
            }

            $this->orderby = $this->orderby . $sort_params;

            // dpm($query_parts);
            // $query_parts
          }
        }
      }
      else {
        if ($field == "rand") {
          $this->orderby = $this->orderby . ' RAND() ';
        }

        if ($field == "eid") {
          global $base_url;
          $my_url = $base_url . "/wisski/navigate/";
          $eid_sort = " OPTIONAL { GRAPH ?g_xz { ?x0 owl:sameAs ?sort . FILTER(STRSTARTS(STR(?sort), '" . $my_url . "')) }} ";
          $sort_params = $elem['direction'] . "(strlen(str(?sort))) " . $elem['direction'] . "(?sort) ";
          // dpm($eid_sort, "yay!");.
          $query_parts = $query_parts . $eid_sort;
          $this->orderby = $this->orderby . $sort_params;

        }
      }
    }

    if ($entity_ids === NULL) {
      return [$query_parts, $entity_ids];
    }
    else {
      if (count($entity_ids) == 0) {
        // Implies OR conjunction; AND is handled above inline.
        // no entities selected so far, treat as if there was no such condition.
        return [$query_parts, NULL];
      }
      elseif (empty($query_parts)) {
        // We can just pass on the entity ids.
        return ['', $entity_ids];
      }
      elseif ($conjunction == 'AND') {
        // We have clauses and entity ids which we combine for AND as we
        // don't know if the parent condition is OR in which case
        // the clauses and ids would produce a cross product.
        // this subquery is (hopefully) much faster.
        // By Mark: This might have been faster, but it is a pain in multi-storage-systems
        // we can't do a query on a single store here... it will not have all results!
        // $entity_ids = $this->buildAndExecSparql($query_parts, $entity_ids);
        // return array('', $entity_ids);
        // Do it if we have a single store system.
        if (empty($this->dependent_parts)) {
          // By Mark: We need the sorting parameter here, as it will delivery wrong results otherwise.
          // $entity_ids = $this->buildAndExecSparql($query_parts, $entity_ids);.
          // dpm($entity_ids, "enten!");.
          $entity_ids = $this->buildAndExecSparql($query_parts, $entity_ids, FALSE, 0, 0, $this->orderby);

          // If we return nothing for the query here it is rather dangerous because if somebody
          // uses order by, the query should use this, too.
          // so we should at least give back the order by part of the query!
          // return array('', $entity_ids);.
          // dpm($sort_query_parts, "sort query parts!");
          // This is a guess! If there is no query that may follow up, we can safely skip the order by part.
          if (count($this->dependent_parts) == 0) {
            return ['', $entity_ids];
          }
          else {
            return [$sort_query_parts, $entity_ids];
          }
        }
        else {
          // dpm("it is an AND!");
          // Therefore we do the full thing!
          // we have to check that!!!
          return [$query_parts, $entity_ids];
        }
      }
      else {
        // OR
        // we just can pass both on.
        return [$query_parts, $entity_ids];
      }
    }

  }

  /**
   * Builds a Sparql SELECT query from the given parameter and sends it to the
   * query's adapter for execution.
   *
   * @param $query_parts
   *   the where clause of the query. The query always asks
   *   about ?x0 so query_parts must contain this variable.
   * @param $entity_ids
   *   an assoc array of entity id => uri pairs that the
   *   resulting array is restricted to.
   * @param $count
   *   whether this is a count query
   * @param $limit
   *   max number of returned entities / the pager limit
   * @param $offset
   *   the offset in combination with $limit
   *
   * @return an assoc array of matched entities in the form of entity_id => uri
   *   or an integer $count is TRUE.
   */
  protected function buildAndExecSparql($query_parts, $entity_ids, $count = FALSE, $limit = 0, $offset = 0, $sort_params = "") {

    if ($count) {
      // We don't do this anymore...
      $select = 'SELECT (COUNT(DISTINCT ?x0) as ?cnt) WHERE { ';
    }
    else {
      $select = 'SELECT DISTINCT ?x0 WHERE { ';
    }

    if (count($this->dependent_parts) == 0) {

      // Special case - if dep is empty and entity ids is empty then we probably search for
      // anything? Search api suddenly does this from time to time, I don't know why...
      // dpm(serialize($query_parts), "qp?");.
      if (empty($entity_ids) && empty($query_parts)) {
        $query_parts = " GRAPH ?x0_nothing { ?x0 ?p ?o . } . ";
      }

      // We restrict the result set to the entities in $entity_ids by adding a
      // VALUES statement in front of the rest of the where clause.
      // entity_ids is an assoc array where the keys are the ids and the values
      // are the corresp URIs. When there is no URI (for the adapter) the URI is
      // empty and needs to be filtered out for the VALUES construct.
      $filtered_uris = NULL;
      if (!empty($entity_ids)) {
        $filtered_uris = array_filter($entity_ids);
      }
      if (!empty($filtered_uris)) {
        $select .= 'VALUES ?x0 { <' . join('> <', $filtered_uris) . '> } ';
      }
      $select .= $query_parts;
    }
    else {

      $first = TRUE;
      // Add dependent parts?
      foreach ($this->dependent_parts as $part) {

        if (!$first) {
          $select .= " UNION ";
        }
        else {
          if (count($this->dependent_parts) > 1) {
            $select .= " { ";
            $part .= " } ";
          }
        }

        $select .= $part;

        $first = FALSE;
      }
    }

    $select .= ' }';

    // dpm($select, "select is?");.
    // dpm($sort_params, "sort?");.
    if ($sort_params) {
      // BY Mark: We don't add GROUP BY HERE due to multiple hits
      // if we do that sorting does not work anymore
      // we would need sparql 1.2 for that.
      $select .= " ORDER BY " . $sort_params;
    }

    if ($limit) {
      $select .= " LIMIT $limit OFFSET $offset";
    }

    $timethis[] = microtime(TRUE);
    // dpm(microtime(), "before");
    // dpm( $select , "query on engine " . $this->getEngine()->adapterId());
    // dpm(serialize($this), "this");.
    $result = $this->getEngine()->directQuery($select);
    // dpm($result, "resquery");.
    $timethis[] = microtime(TRUE);
    $adapter_id = $this->getEngine()->adapterId();
    // drupal_set_message("I answered: " . $adapter_id);.
    if (WISSKI_DEVEL) {
      \Drupal::logger("query adapter $adapter_id")->debug('(sub)query {query} yielded result count {cnt}: {result}', ['query' => $select, 'result' => $result, 'cnt' => $result->count()]);
    }
    if ($result === NULL) {
      throw new \Exception("query failed (null): $select");
    }
    elseif ($result->numRows() == 0) {
      $return = $count ? 0 : [];
    }
    elseif ($count) {
      $return = $result[0]->cnt->getValue();
    }
    else {
      // Make the assoc array from the results.
      $return = [];
      foreach ($result as $row) {
        if (!empty($row) && !empty($row->x0)) {
          $uri = $row->x0->getUri();
          if (!empty($uri)) {
            $timethat = microtime(TRUE);
            $entity_id = AdapterHelper::getDrupalIdForUri($uri, TRUE, $adapter_id);
            $timethis[] = "$timethat " . (microtime(TRUE) - $timethat) . " " . ($timethis[1] - microtime(TRUE));
            if (!empty($entity_id)) {
              $return[$entity_id] = $uri;
            }
          }
        }
      }
    }
    // drupal_set_message("I return for $adapter_id and query " . $select . " data: " . serialize($return));
    return $return;

  }

  /**
   *
   */
  protected function executeStatusCondition($operator, $value, $bundleid = '%') {
    $entity_ids = NULL;

    // It would be good to supply the bundle here, too...
    $query = \Drupal::database()->select('wisski_entity_field_properties', 't')
      ->distinct()
      ->fields('t', ['eid', 'ident'])
    // ->condition('adapter_id', $this->getEngine()->adapterId())
      ->condition('fid', 'status', '=')
      ->condition('bid', $bundleid, '=')
      ->condition('ident', $value, '=');
    $entity_ids = $query->execute()->fetchAllKeyed();

    $out = [];

    $adapter = \Drupal::service('entity_type.manager')->getStorage('wisski_salz_adapter')->load($this->getEngine()->adapterId());

    foreach ($entity_ids as $eid => $bla) {
      // NOTE: getUrisForDrupalId returns one uri as string as we have
      // given the adapter.
      $out[$eid] = '' . AdapterHelper::getUrisForDrupalId($eid, $adapter) . '';
    }

    return $out;

  }

  /**
   *
   */
  protected function executeLangcodeCondition($operator, $value, $bundleid = '%') {
    $entity_ids = NULL;

    $langcode = current($value);

    if ($langcode == "***LANGUAGE_language_interface***") {
      $langcode = \Drupal::service('language_manager')->getCurrentLanguage()->getId();
    }

    // dpm("I am looking for $langcode");
    // It would be good to supply the bundle here, too...
    $query = \Drupal::database()->select('wisski_entity_field_properties', 't')
      ->distinct()
      ->fields('t', ['eid', 'ident'])
    // ->condition('adapter_id', $this->getEngine()->adapterId())
      ->condition('fid', 'langcode', '=')
      ->condition('bid', $bundleid, '=')
      ->condition('ident', $langcode, '=');
    $entity_ids = $query->execute()->fetchAllKeyed();
    // dpm($bundleid, "bun?");
    // dpm($value, "val?");.
    $out = [];

    $adapter = \Drupal::service('entity_type.manager')->getStorage('wisski_salz_adapter')->load($this->getEngine()->adapterId());

    foreach ($entity_ids as $eid => $bla) {
      // NOTE: getUrisForDrupalId returns one uri as string as we have
      // given the adapter.
      $out[$eid] = '' . AdapterHelper::getUrisForDrupalId($eid, $adapter) . '';
    }

    return $out;

  }

  /**
   *
   */
  protected function executeEntityIdCondition($operator, $value) {
    $entity_ids = NULL;

    // dpm($value, "value?");.
    if (empty($value)) {
      // If no value is given, then condition is always true.
      // this may be the case when a field's mere existence is checked;
      // as the eid always exists, this is true for every entity
      // => do nothing.
    }
    else {
      // We directly access the entity table.
      // @todo this is a hack but faster than talking with the AdapterHelper
      // NOTE: an empty operator value means IN as it is the default. This is
      // necessary at least since Drupal 8.4, as the quickedit module makes
      // such queries.
      if ($operator == 'IN' || $operator == "=" || empty($operator)) {
        $values = (array) $value;
        $query = \Drupal::database()->select('wisski_salz_id2uri', 't')
          ->distinct()
          ->fields('t', ['eid', 'uri'])
          ->condition('adapter_id', $this->getEngine()->adapterId())
          ->condition('eid', $values, 'IN')
          ->condition('uri', '%/wisski/navigate/%', 'NOT LIKE');
        $entity_ids = $query->execute()->fetchAllKeyed();
      }
      elseif ($operator == 'BETWEEN') {
        $values = (array) $value;
        $query = \Drupal::database()->select('wisski_salz_id2uri', 't')
          ->distinct()
          ->fields('t', ['eid', 'uri'])
          ->condition('adapter_id', $this->getEngine()->adapterId())
          ->condition('eid', $values, 'BETWEEN')
          ->condition('uri', '%/wisski/navigate/%', 'NOT LIKE');
        $entity_ids = $query->execute()->fetchAllKeyed();
      }
      elseif ($operator == '>' || $operator == '<') {
        $values = (array) $value;
        $query = \Drupal::database()->select('wisski_salz_id2uri', 't')
          ->distinct()
          ->fields('t', ['eid', 'uri'])
          ->condition('adapter_id', $this->getEngine()->adapterId())
          ->condition('eid', $values, $operator)
          ->condition('uri', '%/wisski/navigate/%', 'NOT LIKE')
          ->condition('uri', 'http%', 'LIKE');
        $entity_ids = $query->execute()->fetchAllKeyed();
        // dpm($this->condition, "condition");
        // dpm($entity_ids, "out");
        // return array();
      }
      else {
        $this->missingImplMsg("Operator '$operator' in eid field query", ['condition' => $this->condition]);
      }
      // dpm($this->condition, "condition");
      // dpm($entity_ids, "out");.
    }
    // dpm($value, "val");
    // dpm($operator, "op");
    // dpm($entity_ids, "ent");.
    return $entity_ids;
  }

  /**
   *
   */
  protected function executeEntityTitleCondition($operator, $value, $bundleid = NULL) {
    $entity_ids = NULL;
    $out_entities = [];
    // dpm($value, "val!");.
    if (empty($value)) {
      // If no value is given, then condition is always true.
      // this may be the case when a field's mere existence is checked;
      // as the title always exists, this is true for every entity
      // => do nothing.
    }
    else {
      // We directly access the title cache table. this is the only way to
      // effeciently query the title. However, this may not always return
      // all expected entity ids as
      // - a title may not yet been written to the table.
      // NOTE: This query is not aware of bundle conditions that may sort out
      // titles that are associated with "wrong" bundles.
      // E.g: an entity X is of bundle A and B. A query on bundle A and title
      // pattern xyz is issued. xyz matches entity title, but for bundle B.
      // The query will still deliver X as it matches both conditions
      // seperately, but not combined!
      // First fetch all entity ids that match the title pattern.
      $select = \Drupal::service('database')
        ->select('wisski_title_n_grams', 'w')
        ->fields('w', ['ent_num']);

      if ($operator == '=' || $operator == "!=" || $operator == "LIKE") {
        $select->condition('ngram', $value, $operator);
      }
      elseif ($operator == 'CONTAINS' || $operator == "STARTS_WITH" || $operator == "ENDS_WITH") {
        $select->condition('ngram', ($operator == 'CONTAINS' ? "%" : "") . $select->escapeLike($value) . "%", 'LIKE');
      }
      else {
        $this->missingImplMsg("Operator '$operator' in title field query", ['condition' => $value]);
        // NULL.
        return $entity_ids;
      }

      // dpm($bundleid, "bundleid!");.
      if ($bundleid) {
        $select->condition('bundle', $bundleid);
      }

      // Handle sorting - currently only for title.
      foreach ($this->sort as $elem) {
        if ($elem['field'] == "title") {
          $select->orderBy('ngram', $elem['direction']);
        }
      }
      // dpm($select, "sel!");.
      $rows = $select
        ->execute()
        ->fetchAll();

      foreach ($rows as $row) {
        $entity_ids[$row->ent_num] = $row->ent_num;
      }

      // dpm($entity_ids, "eids!");
      // Now fetch the uris for the eids as we have to return both.
      $query = \Drupal::database()->select('wisski_salz_id2uri', 't')
        ->distinct()
        ->fields('t', ['eid', 'uri'])
        ->condition('adapter_id', $this->getEngine()->adapterId())
      // We need to add this line below as the wisski navigate url is not the one we need...
        ->condition('eid', $entity_ids, 'IN')
        ->condition('uri', '%/wisski/navigate/%', 'NOT LIKE');
      $entity_ids = $query->execute()->fetchAllKeyed();
      // dpm($entity_ids, "sec");
      //      $out_entities = array();
      // Redo the sorting.
      foreach ($rows as $row) {
        if (isset($entity_ids[$row->ent_num])) {
          $out_entities[$row->ent_num] = $entity_ids[$row->ent_num];
        }
      }
      // dpm( $out_entities, "out!");.
      $entity_ids = $out_entities;

    }
    return $entity_ids;
  }

  /**
   *
   */
  protected function makeTSEntityIdCondition($operator, $value) {
    // dpm($operator ,"op?");
    // dpm($value, "val?");.
    $query_parts = "";

    if (empty($operator)) {
      $operator = "=";
      // dpm(" argh!");.
    }

    if (empty($value)) {
      return "";
    }

    $basefieldinfo = $this->getEngine()->getBaseFieldGraph();

    $basefield = "eid";
    $basefieldurl = $this->getEngine()->getDefaultDataGraphUri() . $basefield;

    if (is_array($value) && count($value) == 1) {
      $value = current($value);
      if ($operator == "IN") {
        $operator = "=";
      }
    }

        /** 
     * Only add the baseField filter on writable engines 
     * in the case if we want to use a engine as a read only 
     * authority file at remote SPARQL repositories (i .e. konchylien). 
     */ 
    if (!is_array($value)) {
      if ($this->getEngine()->isWritable()) {
        $query_parts = "{ GRAPH <$basefieldinfo> { ?x0 <$basefieldurl> ?eid . <$basefieldurl> a owl:AnnotationProperty . FILTER( ?eid $operator $value ) . }} ";
      }
    }
    else {
      if ($this->getEngine()->isWritable()) {
        $query_parts = "{ GRAPH <$basefieldinfo> { ?x0 <$basefieldurl> ?eid . <$basefieldurl> a owl:AnnotationProperty . FILTER( ?eid $operator (";
      } else {
        $query_parts = "{{ ((";
      }
      foreach ($value as $one_val) {
        $query_parts .= "$one_val, ";
      }
      $query_parts = substr($query_parts, 0, -2);
      $query_parts .= ")) . }} "; 
    }

    return $query_parts;
  }

  /**
   *
   */
  protected function makeBundleCondition($operator, $value) {

    $query_parts = [];

    if (empty($operator) || $operator == 'IN' || $operator == '=') {
      $bundle_ids = (array) $value;
      $engine = $this->getEngine();

      $i = $this->varCounter++;

      // We have to igo thru all the groups that belong to this bundle.
      foreach ($this->getPbs() as $pb) {
        foreach ($bundle_ids as $bid) {
          $groups = $pb->getGroupsForBundle($bid);
          foreach ($groups as $group) {

            // Build up an array for separating the variables of the sparql
            // subqueries.
            // only the first var x0 get to be the same so that everything maps
            // to the same entity
            // NOTE: we set the first var to x0 although it's not x0.
            $starting_position = $pb->getRelativeStartingPosition($group, FALSE);
            // drupal_set_message(serialize($group));
            // drupal_set_message(serialize($starting_position));
            $vars[$starting_position] = 'x0';
            for ($j = count($group->getPathArray()); $j > $starting_position; $j--) {
              $vars[$j] = "c{$i}_x$j";
            }
            $vars['out'] = "c{$i}_out";

            $sparql_part = $engine->generateTriplesForPath($pb, $group, '', NULL, NULL, 0, $starting_position, FALSE, '=', 'field', TRUE, $vars);

            if (!in_array($sparql_part, $query_parts)) {
              $query_parts[] = $sparql_part;
            }
          }
        }
      }
    }
    else {
      $this->missingImplMsg("Operator '$operator' in bundle fieldquery", [func_get_args()]);
    }

    if (empty($query_parts)) {
      // The bundle is not handled by this adapter
      // we signal that this query should be skipped.
      return NULL;
    }
    else {
      $query_parts = '{{ ' . join('} UNION {', $query_parts) . '}} ';

      return $query_parts;
    }

  }

  /**
   *
   */
  protected function makeFieldCondition($field, $operator, $value) {

    $query_parts = [];

    $count = 0;
    $path_available = FALSE;
    foreach ($this->getPbs() as $pb) {
      $path = $pb->getPathForFid($field);
      if (!empty($path)) {
        $path_available = TRUE;
        $pbarray = $pb->getPbPath($path->id());
        $new_query_part = $this->makePathCondition($pb, $path, $operator, $value, $pbarray['parent']);
        if ($new_query_part !== NULL) {
          $query_parts[] = $new_query_part;
          $count++;
        }
      }
    }

    if (!$path_available) {
      // The adapter is not responsible for this field.
      // we just skip this condition.
      // @todo should we rather declare the condition as failed? (return NULL)
      return '';
    }
    elseif ($count == 0) {
      return NULL;
    }
    elseif ($count == 1) {
      return $query_parts[0];
    }
    else {
      $query_parts = '{{ ' . join('} UNION {', $query_parts) . '}} ';
      return $query_parts;
    }

  }

  /**
   *
   */
  protected function makePathCondition($pb, $path, $operator, $value, $starting_group = NULL) {

    if (!$operator) {
      $operator = '=';
    }
    if ($starting_group === NULL) {
      $starting_position = 0;
    }
    else {
      $cur_group = \Drupal::service('entity_type.manager')->getStorage('wisski_path')->load($starting_group);
      if (!$cur_group) {
        // No valid group given:
        // treat it as relative path.
        $starting_position = $pb->getRelativeStartingPosition($path, FALSE);
      }
      else {
        // The starting position is where the group ends (the group's class)
        // is included, however.
        // NOTE: ATTENTION: starting position is counted in old WissKI style,
        // ie. only concepts are counted.
        $starting_position = (count($cur_group->getPathArray()) - 1) / 2;
      }
    }
    // \Drupal::logger('query path cond')->debug("start path cond:".$this->varCounter.";$operator:$value;".($path->getDatatypeProperty()?:"no dt"));
    $dt_prop = $path->getDatatypeProperty();
    $obj_uris = [];
    if ((empty($dt_prop) || $dt_prop == 'empty') && !$path->isGroup()) {
      // We have a regular path without datatype property.
      // @todo if value is an array how do we want to treat it?
      if (!is_array($value)) {
        // If value is a scalar we treat it as title pattern and do a search
        // for these entities first.
        // Determine at which position the referred/object uris are in the path.
        $obj_pos = $path->getDisamb() ? $path->getDisamb() * 2 - 2 : (count($path->getPathArray()) - 1);
        $referred_concept = $path->getPathArray()[$obj_pos];

        // We have to find out the bundle(s)
        $bundles = \Drupal::service('wisski_pathbuilder.manager')->getBundlesWithStartingConcept($referred_concept);
        // Top bundles are preferred.
        $preferred_bundles = NULL;
        foreach ($bundles as $bid => $info) {
          if ($info['is_top_bundle']) {
            $preferred_bundles[$bid] = $bid;
            unset($bundles[$bid]);
          }
        }

        $entity_ids = [];

        // dpm($preferred_bundles, "pref?");
        // dpm($value, "val?");
        // dpm($operator, "op");.
        // Special case if we get a constraint from a view that asks for a special entity
        // id. this is used in condition filters e.g.
        if (is_numeric($value) && ($operator == "HAS_EID" || $operator == "has_eid")) {
          // dpm("we take value...");.
          $entity_ids = [$value => $value];
        }

        // Only do that if we really ask for something.
        if (empty($entity_ids)) {
          if (!empty($preferred_bundles)) {
            if ($operator == "NOT_EMPTY" || $operator == "EMPTY") {
              // By Mark:
              // in this case the lower part is doing the work, so we don't have to do anything here
              // it would just slow everything.
            }
            else {
              $entity_ids = $this->queryReferencedEntities($preferred_bundles, $value, $operator);
            }
            // dpm($entity_ids, "ents?");.
          }
        }

        // If there are no preferred bundles or querying them yielded no result
        // we search in all the other bundles.
        if (empty($entity_ids) && !empty($bundles)) {
          // We have to take the keys as the values are the info structs.
          $entity_ids = $this->queryReferencedEntities(array_keys($bundles), $value, $operator);
        }

        if (empty($entity_ids) && !($operator == "NOT_EMPTY" || $operator == "EMPTY")) {
          // There are no entities that match the title, therefore the whole
          // condition cannot be satisfied and we have to abort.
          return NULL;
          // $obj_uris = 'UNDEF'; // this leads to an unbound sparql var
        }

        // Get the uris for the entity ids.
        $adapter = \Drupal::service('entity_type.manager')->getStorage('wisski_salz_adapter')->load($this->getEngine()->adapterId());
        foreach ($entity_ids as $eid) {
          // NOTE: getUrisForDrupalId returns one uri as string as we have
          // given the adapter.
          $obj_uris[] = '<' . AdapterHelper::getUrisForDrupalId($eid, $adapter) . '>';
        }

      }
    }
    else {
      // It has a datatype-property?
      // This is a special case for the eid-thingies...
      // for example if there is a filter...
      if (is_numeric($value) && ($operator == "HAS_EID" || $operator == "has_eid")) {
        $entity_ids = [$value => $value];

        $obj_pos = $path->getDisamb() ? $path->getDisamb() * 2 - 2 : (count($path->getPathArray()) - 1);

        // Reset value as it is not so important anymore... we have the eid!
        $value = NULL;

        // Get the uris for the entity ids.
        $adapter = \Drupal::service('entity_type.manager')->getStorage('wisski_salz_adapter')->load($this->getEngine()->adapterId());
        foreach ($entity_ids as $eid) {
          // NOTE: getUrisForDrupalId returns one uri as string as we have
          // given the adapter.
          $obj_uris[] = '<' . AdapterHelper::getUrisForDrupalId($eid, $adapter) . '>';
        }
      }
    }

    // Build up an array for separating the variables of the sparql
    // subqueries.
    // only the first var x0 get to be the same so that everything maps
    // to the same entity.
    $vars[$starting_position * 2] = "x0";
    $i = $this->varCounter++;
    for ($j = count($path->getPathArray()); $j > $starting_position * 2; $j--) {
      $vars[$j] = "c{$i}_x$j";
    }
    $vars['out'] = "c{$i}_out";

    // Arg 11 ($relative) must be FALSE, otherwise fields of subgroups yield
    // the entities of the subgroup.
    $query_part = $this->getEngine()->generateTriplesForPath($pb, $path, $value, NULL, NULL, 0, $starting_position, FALSE, $operator, 'field', FALSE, $vars);
    // dpm($query_part, "qp");
    // return $query_part;.
    // dpm($operator, "op?");.
    if (!empty($obj_uris) && $operator != "EMPTY") {
      $query_part .= ' VALUES ?' . $vars[$obj_pos] . ' { ' . join(' ', $obj_uris) . ' }';
    }

    // This might be a hack - we search for the first . and
    // put the optional there.
    /*
    if($operator == "EMPTY") {
    $pos = strpos($query_part, "} .");
    $query_part = substr_replace($query_part, "} . OPTIONAL { ", $pos, 3);
    $dt = $path->getDataTypeProperty();
    if(!empty($dt) && $dt != "empty")
    $query_part = $query_part . " } . FILTER( !bound(?" . $vars['out'] . " ) )";
    else // if it has no dt we use the last element.
    $query_part = $query_part . " } . FILTER( !bound(?" . $vars[count($path->getPathArray())-1] . " ) )";
    #      dpm($query_part, "qp1");
    #      dpm($path, "path");
    }
     */
    return $query_part;

  }

  /**
   *
   */
  protected function queryReferencedEntities($bundle_ids, $title_search_string, $operator) {
    // We start a new query.
    $result = \Drupal::entityQuery('wisski_individual')
      ->accessCheck(TRUE)
      ->condition('title', $title_search_string, $operator)
      ->condition('bundle', $bundle_ids, 'IN')
      ->execute();
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function existsAggregate($field, $function, $langcode = NULL) {
    return $this->conditionAggregate->exists($field, $function, $langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function notExistsAggregate($field, $function, $langcode = NULL) {
    return $this->conditionAggregate->notExists($field, $function, $langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function conditionAggregateGroupFactory($conjunction = 'AND') {
    return new ConditionAggregate($conjunction, $this);
  }

  /**
   * Places a screen and log message for functionality that is not implemented (yet).
   */
  protected function missingImplMsg($msg, $data) {
    // dpm("bump", "bump");.
    \Drupal::messenger()->addError("Missing entity query implementation: $msg. See log for details.");
    \Drupal::logger("wisski entity query")->warning("Missing entity query implementation: $msg. Data: {data}", ['data' => serialize($data)]);
  }

}
