<?php

namespace Drupal\wisski_authority_document\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Http\ClientFactory;
use Drupal\wisski_autocomplete\Controller\WisskiAutocompleteController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for handling authority document autocomplete requests.
 */
class AutocompleteController extends ControllerBase {

  /**
   * The authority document settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $authorityDocumentSettings;

  /**
   * The WissKI autocomplete controller.
   *
   * @var \Drupal\wisski_autocomplete\Controller\WisskiAutocompleteController
   */
  protected $autocompleteController;

  /**
   * The HTTP client factory.
   *
   * @var \Drupal\Core\Http\ClientFactory
   */
  protected $httpClientFactory;

  /**
   * Constructs an AutocompleteController object.
   *
   * @param \Drupal\wisski_autocomplete\Controller\WisskiAutocompleteController $autocompleteController
   *   The WissKI autocomplete controller.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Http\ClientFactory $httpClientFactory
   *   The HTTP client factory.
   */
  public function __construct(WisskiAutocompleteController $autocompleteController, ConfigFactoryInterface $configFactory, ClientFactory $httpClientFactory) {
    $this->autocompleteController = $autocompleteController;
    $this->authorityDocumentSettings = $configFactory->get('wisski_authority_document.settings');
    $this->httpClientFactory = $httpClientFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wisski.autocomplete.controller'),
      $container->get('config.factory'),
      $container->get('http_client_factory')
    );
  }

  /**
   * Returns autocomplete suggestions from authority document APIs.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing matching results.
   */
  public function autocomplete(Request $request) {
    $matches = [];
    $input = $request->query->get('q');
    $authorityType = $request->query->get('authority_type');
    $entityTypes = $request->query->get('entity_type') ? explode(',', $request->query->get('entity_type')) : [];
    $fieldId = $request->query->get('field_id');
    $customTermsPreferredNameField = $this->authorityDocumentSettings->get('custom_terms')['custom_terms_preferred_name_field'];
    $customTermsAlternativeNameField = $this->authorityDocumentSettings->get('custom_terms')['custom_terms_alternative_name_field'];

    if (strlen($input) < 3) {
      return new JsonResponse($matches);
    }

    // Convert authority_type to lowercase for consistency.
    $authorityType = strtolower($authorityType);

    // Get configuration.
    $config = $this->config('wisski_authority_document.settings');

    switch ($authorityType) {
      case 'custom':
        if (!empty($customTermsPreferredNameField)) {
          try {
            $altName = FALSE;
            // Call the autocomplete method which returns a JsonResponse.
            $response = $this->autocompleteController->autocomplete($request, $customTermsPreferredNameField, 100, TRUE);

            // Get the content from the JsonResponse.
            $matches = json_decode($response->getContent(), TRUE);

            if (empty($matches)) {
              $response = $this->autocompleteController->autocomplete($request, $customTermsAlternativeNameField, 100, TRUE);
              $matches = json_decode($response->getContent(), TRUE);
              $altName = TRUE;
            }

            // If we have matches, add URI to the label for each item.
            if (!empty($matches) && is_array($matches)) {
              foreach ($matches as &$match) {
                if (isset($match['label']) && isset($match['disambUri'])) {
                  // Append the URI to the label.
                  $baseValue = explode('(', $match['value'])[0];
                  $sourceInfo = $altName ? ' (via ' . $baseValue . ')' : '';
                  $match['value'] = $match['label'];
                  $match['label'] = $match['label'] . $sourceInfo . ' - ' . $match['disambUri'];
                  $match['uri'] = $match['disambUri'];
                }
              }
              // Unset the reference to avoid issues.
              unset($match);
            }

          }
          catch (\Exception $e) {
            $this->getLogger('wisski_authority_document')->error('Error in WissKI autocomplete: @error', ['@error' => $e->getMessage()]);
          }
        }
        break;

      case 'gnd':
        $matches = $this->getGndMatches($input, $entityTypes);
        break;

      case 'getty':
        // Check if Getty is enabled in settings.
        if ($config->get('getty_enabled')) {
          $matches[] = [
            'value' => 'not yet implemented',
            'label' => 'not yet implemented',
            'uri' => 'not yet implemented',
          ];
        }
        break;

      case 'geonames':
        // Check if GeoNames is enabled in settings.
        if ($config->get('geonames_enabled')) {
          $matches[] = [
            'value' => 'not yet implemented',
            'label' => 'not yet implemented',
            'uri' => 'not yet implemented',
          ];
        }
        break;
    }

    return new JsonResponse($matches);
  }

  /**
   * Gets autocomplete suggestions from the GND API.
   *
   * @param string $input
   *   The search query.
   * @param array|null $entityTypes
   *   Optional entity type restriction.
   *
   * @return array
   *   An array of suggestions.
   */
  protected function getGndMatches($input, $entityTypes = NULL) {
    $matches = [];
    $client = $this->httpClientFactory->fromOptions([
      'timeout' => 5,
      'connect_timeout' => 3,
    ]);

    try {
      // Get settings from configuration.
      $config = $this->config('wisski_authority_document.settings');
      $gndApiUrl = $config->get('gnd_api_url') ?: 'https://services.dnb.de/sru/authorities';
      $maxResults = $config->get('gnd_max_results') ?: 10;
      $debugEnabled = $config->get('debug_enabled') ?: FALSE;

      // Build CQL query for SRU.
      $input = $input;
      $query = '';
      $indexes = [];
      $gndTypeFilter = '';
      // Add entity type filter if provided.
      if (!empty($entityTypes)) {
        // Map entity type to GND BBG code for filtering.
        $gndTypeFilter = ' and (BBG=' . implode('* or BBG=', $entityTypes) . '*)';
        // @see https://www.dnb.de/SharedDocs/Downloads/EN/Professionell/Metadatendienste/Rundschreiben/rundschreiben20180403ZdbAnkuendigungIndiezKatalogZdbSruSchnittstelle.pdf?__blob=publicationFile&v=3
        foreach ($entityTypes as $entityType) {
          switch ($entityType) {
            case 'Tb':
              // Körperschaft.
              $indexes += ['KOE'];
              break;

            case 'Tf':
              // Kongress.
              $indexes += ['TIT', 'TST'];
              break;

            case 'Tp':
              // Person.
              $indexes += ['PER'];
              break;

            case 'Ts':
              // Sachbegriff.
              $indexes += ['SW', 'SPR', 'TIT'];
              break;

            case 'Tu':
              // Werk.
              $indexes += ['TIT', 'TST'];
              break;

            default:
              // Alle Begriffe.
              break;
          }
        }
      }
      else {
        $indexes = ['WOE'];
      }

      // Remove redundant items from indexes array.
      $indexes = array_unique($indexes);

      // Build query without double encoding.
      $queryParts = [];
      if (count($indexes) > 1) {
        foreach ($indexes as $index) {
          $queryParts[] = $index . '=' . $input;
        }
        $query = implode(' or ', $queryParts);
      }
      else {
        $query = $indexes[0] . '=' . $input;
      }

      if (!empty($gndTypeFilter)) {
        $query .= $gndTypeFilter;
      }

      // Set a reasonable batch size for pagination.
      $batchSize = 100;
      $startRecord = 1;
      $totalRecords = 0;
      $processedRecords = 0;

      do {
        // Construct SRU request URL with proper
        // encoding and pagination parameters.
        $url = $gndApiUrl . '?version=1.1&operation=searchRetrieve&query=' . urlencode($query);
        $url .= '&maximumRecords=' . $batchSize;
        $url .= '&startRecord=' . $startRecord;
        $url .= '&recordSchema=RDFxml';

        $response = $client->get($url);
        $xml = simplexml_load_string($response->getBody());

        if (!$xml) {
          $this->getLogger('wisski_authority_document')->error('Failed to parse XML response from GND SRU API.');
          throw new \Exception('Invalid XML response from GND SRU API.');
        }

        // Register namespaces for parsing.
        $namespaces = $xml->getNamespaces(TRUE);

        // Debug the namespaces if enabled.
        if ($debugEnabled) {
          $this->getLogger('wisski_authority_document')->debug('XML namespaces: @namespaces', [
            '@namespaces' => print_r($namespaces, TRUE),
          ]);
        }

        // Get total number of records if this is the first request.
        if ($startRecord === 1 && isset($xml->numberOfRecords)) {
          $totalRecords = (int) $xml->numberOfRecords;
          if ($debugEnabled) {
            $this->getLogger('wisski_authority_document')->debug('Total records found: @count', [
              '@count' => $totalRecords,
            ]);
          }
        }

        if (!isset($xml->records) || !isset($xml->records->record)) {
          break;
        }

        // Process records.
        foreach ($xml->records->record as $record) {
          if (!isset($record->recordData)) {
            continue;
          }

          // For debugging purposes, log the XML structure.
          if ($debugEnabled) {
            $this->logXmlStructure($record->recordData);
          }

          // First access the RDF namespace.
          $rdfData = $record->recordData->children($namespaces['rdf']);

          // Check if RDF element exists in the expected structure.
          if (!isset($rdfData->RDF)) {
            continue;
          }

          // Find Description element in RDF namespace.
          $description = NULL;
          foreach ($rdfData->RDF->children($namespaces['rdf']) as $child) {
            if ($child->getName() === 'Description') {
              $description = $child;
              break;
            }
          }

          // If Description not found directly, try another
          // approach for SimpleXML navigation.
          if ($description === NULL) {
            if (isset($rdfData->RDF->Description)) {
              $description = $rdfData->RDF->Description;
            }
            else {
              continue;
            }
          }

          // Access GNDO namespace elements within Description.
          $gndoElements = $description->children($namespaces['gndo']);

          // Get preferred name and variant names.
          $label = '';
          $variantNames = [];

          foreach ($entityTypes as $entityType) {
            switch ($entityType) {
              // Prioritize preferredNameForTheWork
              // since it's what we're looking for.
              case 'Tb':
                // Körperschaft.
                if (isset($gndoElements->preferredNameForTheCorporateBody)) {
                  $label = (string) $gndoElements->preferredNameForTheCorporateBody;
                  foreach ($gndoElements->variantNameForTheCorporateBody as $variant) {
                    $variantNames[] = (string) $variant;
                  }
                  break;
                }
              case 'Tf':
                // Kongress.
                if (isset($gndoElements->preferredNameForTheConferenceOrEvent)) {
                  $label = (string) $gndoElements->preferredNameForTheConferenceOrEvent;
                  foreach ($gndoElements->variantNameForTheConferenceOrEvent as $variant) {
                    $variantNames[] = (string) $variant;
                  }
                  break;
                }
              case 'Tp':
                // Person.
                if (isset($gndoElements->preferredNameForThePerson)) {
                  $label = (string) $gndoElements->preferredNameForThePerson;
                  foreach ($gndoElements->variantNameForThePerson as $variant) {
                    $variantNames[] = (string) $variant;
                  }
                  break;
                }
              case 'Ts':
                // Sachbegriff.
                if (isset($gndoElements->preferredNameForTheSubjectHeading)) {
                  $label = (string) $gndoElements->preferredNameForTheSubjectHeading;
                  foreach ($gndoElements->variantNameForTheSubjectHeading as $variant) {
                    $variantNames[] = (string) $variant;
                  }
                  break;
                }
              case 'Tu':
                // Werk.
                if (isset($gndoElements->preferredNameForTheWork)) {
                  $label = (string) $gndoElements->preferredNameForTheWork;
                  foreach ($gndoElements->variantNameForTheWork as $variant) {
                    $variantNames[] = (string) $variant;
                  }
                  break;
                }
              default:
                // Alle Begriffe.
                break;
            }
          }

          if (empty($label)) {
            if (isset($gndoElements->preferredName)) {
              $label = (string) $gndoElements->preferredName;
              foreach ($gndoElements->variantName as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            elseif (isset($gndoElements->preferredNameForTheCorporateBody)) {
              $label = (string) $gndoElements->preferredNameForTheCorporateBody;
              foreach ($gndoElements->variantNameForTheCorporateBody as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            elseif (isset($gndoElements->preferredNameForTheConferenceOrEvent)) {
              $label = (string) $gndoElements->preferredNameForTheConferenceOrEvent;
              foreach ($gndoElements->variantNameForTheConferenceOrEvent as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            elseif (isset($gndoElements->preferredNameForThePerson)) {
              $label = (string) $gndoElements->preferredNameForThePerson;
              foreach ($gndoElements->variantNameForThePerson as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            elseif (isset($gndoElements->preferredNameForTheSubjectHeading)) {
              $label = (string) $gndoElements->preferredNameForTheSubjectHeading;
              foreach ($gndoElements->variantNameForTheSubjectHeading as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            elseif (isset($gndoElements->preferredNameForTheWork)) {
              $label = (string) $gndoElements->preferredNameForTheWork;
              foreach ($gndoElements->variantNameForTheWork as $variant) {
                $variantNames[] = (string) $variant;
              }
            }
            else {
              $label = "no label found";
            }
          }

          // Check if the search term is part of the label or variant names, normalizing Umlauts and special characters.
          $normalizedInput = transliterator_transliterate('Any-Latin; Latin-ASCII', $input);
          $normalizedLabel = transliterator_transliterate('Any-Latin; Latin-ASCII', $label);
          $foundMatch = FALSE;

          // Check preferred name first.
          if (stripos($normalizedLabel, $normalizedInput) !== FALSE) {
            $foundMatch = TRUE;
          }

          $tooltip = "";

          // If not found in preferred name, check variant names.
          if (!$foundMatch && !empty($variantNames)) {
            foreach ($variantNames as $variantName) {
              $normalizedVariant = transliterator_transliterate('Any-Latin; Latin-ASCII', $variantName);
              if (stripos($normalizedVariant, $normalizedInput) !== FALSE) {
                $foundMatch = TRUE;
                $tooltip = $this->t('@label (via @variant)', ['@label' => $label, '@variant' => $variantName]);
                break;
              }
            }
          }

          if (!$foundMatch) {
            // Skip this result if the search term is not found in the label or variants.
            continue;
          }

          // Get URI (about attribute)
          $uri = '';
          $rdfAttributes = $description->attributes($namespaces['rdf']);
          if (isset($rdfAttributes['about'])) {
            $uri = (string) $rdfAttributes['about'];
          }

          // Skip if no URI found.
          if (empty($uri)) {
            continue;
          }

          // Get additional information (dates for persons, etc.).
          $additional_info = [];

          // For persons, include dates.
          if (isset($gndoElements->dateOfBirth)) {
            $additional_info[] = 'geboren: ' . (string) $gndoElements->dateOfBirth;
          }
          if (isset($gndoElements->dateOfDeath)) {
            $additional_info[] = 'gestorben: ' . (string) $gndoElements->dateOfDeath;
          }

          // Add profession/occupation if available.
          if (isset($gndoElements->professionOrOccupation)) {
            $professions = [];
            foreach ($gndoElements->professionOrOccupation as $profession) {
              $professions[] = (string) $profession;
            }
            if (!empty($professions)) {
              $additional_info[] = implode(', ', $professions);
            }
          }

          // Build display label with additional info.
          $display_label = $label;
          // Filter out empty values and
          // whitespace-only strings from additional info.
          $additional_info = array_filter($additional_info, function ($item) {
            return !empty($item) && trim($item) !== '';
          });
          if (!empty($additional_info)) {
            $display_label .= ' (' . implode('; ', $additional_info) . ')';
          }

          $display_label .= ' - ' . $uri;

          $matches[] = [
            'value' => $label,
            'label' => $display_label,
            'tooltip' => $tooltip,
            'uri' => $uri,
          ];

          // Sort matches by exact match first, then alphabetically.
          usort($matches, function ($a, $b) use ($input) {
            // Check for exact matches (prioritize them).
            $a_exact = strtolower($a['value']) === strtolower($input);
            $b_exact = strtolower($b['value']) === strtolower($input);

            if ($a_exact && !$b_exact) {
              return -1;
            }
            elseif (!$a_exact && $b_exact) {
              return 1;
            }

            // If exactness is the same, sort alphabetically.
            return strcasecmp($a['value'], $b['value']);
          });

        }

        // Update processed records count and start record for next batch.
        $processedRecords += count($xml->records->record);
        $startRecord += $batchSize;

      } while ($processedRecords < $totalRecords);

      if ($debugEnabled) {
        $this->getLogger('wisski_authority_document')->debug('Retrieved @count records out of @total total records.', [
          '@count' => count($matches),
          '@total' => $totalRecords,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->getLogger('wisski_authority_document')->error('Error querying GND SRU API: @error', ['@error' => $e->getMessage()]);
    }

    // Limit the number of matches to the maximum results limit.
    $matches = array_slice($matches, 0, $maxResults);

    return $matches;
  }

  /**
   * Helper function to log the XML response structure.
   *
   * @param \SimpleXMLElement $element
   *   The XML element to log.
   * @param int $depth
   *   The current depth in the XML tree.
   * @param string $path
   *   The current path in the XML tree.
   */
  protected function logXmlStructure($element, $depth = 0, $path = '') {
    if ($depth > 10) {
      // Prevent infinite recursion.
      return;
    }

    $indent = str_repeat('  ', $depth);
    $namespaces = $element->getNamespaces(TRUE);

    foreach ($namespaces as $prefix => $ns) {
      $nsElements = $element->children($ns);
      foreach ($nsElements as $name => $nsElement) {
        $currentPath = $path . '/' . ($prefix ? "$prefix:" : '') . $name;
        $this->getLogger('wisski_authority_document')->debug('@indent@path: @value', [
          '@indent' => $indent,
          '@path' => $currentPath,
          '@value' => (string) $nsElement,
        ]);
        $this->logXmlStructure($nsElement, $depth + 1, $currentPath);
      }
    }
  }

}
