<?php

namespace Drupal\wisski_authority_document\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'authority_document_autocomplete' widget.
 */
#[FieldWidget(
  id: 'authority_document_autocomplete',
  label: new TranslatableMarkup('Authority Document Autocomplete'),
  field_types: ['string', 'link'],
)]
class AuthorityDocumentAutocompleteWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'authority_type' => 'gnd',
      'authority_type_field' => 'field__auth_doc__authority_type',
      'search_field' => 'field__auth_doc__search_term',
      'link_field' => 'field__auth_doc__link',
      'entity_type_field' => 'field__auth_doc__entity_type',
      'min_chars' => 3,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['authority_type_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Authority type field id'),
      '#description' => $this->t('Enter the field id for the authority type.'),
      '#default_value' => $this->getSetting('authority_type_field'),
      '#required' => TRUE,
    ];

    $elements['entity_type_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity Type Field ID'),
      '#description' => $this->t('Restrict the autocomplete suggestions to a specific entity type.'),
      '#default_value' => $this->getSetting('entity_type_field'),
      '#required' => FALSE,
    ];

    $elements['link_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link Field ID'),
      '#description' => $this->t('The field ID where links will be stored.'),
      '#default_value' => $this->getSetting('link_field'),
      '#required' => TRUE,
    ];

    $elements['min_chars'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum characters for autocomplete'),
      '#description' => $this->t('The minimum number of characters required to trigger autocomplete.'),
      '#default_value' => $this->getSetting('min_chars'),
      '#min' => 1,
      '#max' => 10,
      '#required' => TRUE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $authority_type_field = $this->getSetting('authority_type_field') ?? '';
    $custom_terms_settings = $this->getSetting('custom_terms') ?? [];
    $custom_terms_preferred_name_field = $custom_terms_settings['custom_terms_preferred_name_field'] ?? '';
    $custom_terms_alternative_name_field = $custom_terms_settings['custom_terms_alternative_name_field'] ?? '';
    $entity_type_field = $this->getSetting('entity_type_field') ?? '';
    $link_field = $this->getSetting('link_field') ?? '';
    $min_chars = $this->getSetting('min_chars') ?? 3;

    $summary[] = $this->t('Authority type field id: @type', ['@type' => $authority_type_field]);
    $summary[] = $this->t('Custom terms preferred name field: @field', ['@field' => $custom_terms_preferred_name_field]);
    $summary[] = $this->t('Custom terms alternative name field: @field', ['@field' => $custom_terms_alternative_name_field]);
    $summary[] = $this->t('Entity type field id: @type', ['@type' => $entity_type_field]);
    $summary[] = $this->t('Link field: @field', ['@field' => $link_field]);
    $summary[] = $this->t('Min chars: @chars', ['@chars' => $min_chars]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $value = $items[$delta]->value ?? '';

    $element['value'] = [
      '#type' => 'textfield',
      '#title' => $element['#title'] ?? $this->fieldDefinition->getLabel(),
      '#description' => $element['#description'] ?? $this->fieldDefinition->getDescription(),
      '#default_value' => $value,
      '#autocomplete_route_name' => 'wisski_authority_document.autocomplete',
      '#attributes' => [
        'class' => ['wisski-authority-autocomplete'],
        'data-authority-type-field' => $this->getSetting('authority_type_field'),
        'data-entity-type-field' => $this->getSetting('entity_type_field'),
        'data-field-id' => $this->fieldDefinition->getName(),
        'data-link-field' => $this->getSetting('link_field'),
        'data-min-chars' => $this->getSetting('min_chars'),
      ],
      '#attached' => [
        'library' => ['wisski_authority_document/autocomplete'],
      ],
    ];

    return $element;
  }

}
