<?php

namespace Drupal\wisski_autocomplete\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Field\WidgetInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\HttpClient\HttpClient;
use Drupal\Core\Url;

/**
 * Plugin implementation of the 'wisski_autocomplete' widget.
 *
 * @FieldWidget(
 *   id = "wisski_autocomplete_widget",
 *   label = @Translation("WissKI autocomplete widget"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class WissKIAutocompleteWidget extends WidgetBase implements WidgetInterface {

  /**
   * The config storage of the field widget.
   *
   * @var Drupal/Core/Config/Config
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'fieldId' => NULL,
      'size' => 60,
      'placeholder' => '',
      'autocompleteLimit' => 10,
      'useTitlePattern' => TRUE,
      'allowOnlyExisting' => FALSE,
    ] + parent::defaultSettings();
  }


  /**
   * Saves single key in settings array in field widget config.
   *
   * @param string $key
   *   Array key of the settings array.
   *
   * @param mixed $value
   *   Array value of the settings array.
   *
   * @return void
   */
  public function setSetting($key, $value){
    parent::setSetting($key, $value);
    $fieldId = $this->getSetting('fieldId');
    if($fieldId){
      // TODO: move $this->config into the constructor.
      $this->config = $this->config ?? \Drupal::service('config.factory')->getEditable("wisski.autocomplete");
      $this->config->set($fieldId, $this->settings)->save();
    }
  }

  /**
   * Saves setting array in field widget config.
   *
   * @param array $settings
   * @return void
   */
  public function setSettings(array $settings)
  {
    parent::setSettings($settings);
    $fieldId = $this->getSetting('fieldId');
    if($fieldId){
      // TODO: move $this->config into the constructor.
      $this->config = $this->config ?? \Drupal::service('config.factory')->getEditable("wisski.autocomplete");
      $this->config->set($fieldId, $this->settings)->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {

    $element['size'] = [
      '#type' => 'number',
      '#title' => t('Size of textfield'),
      '#default_value' => $this->getSetting('size'),
      '#required' => TRUE,
      '#min' => 1,
    ];
    $element['placeholder'] = [
      '#type' => 'textfield',
      '#title' => t('Placeholder'),
      '#default_value' => $this->getSetting('placeholder'),
      '#description' => t('Text that will be shown inside the field until a value is entered. This hint is usually a sample value or a brief description of the expected format.'),
    ];
    $element['autocompleteLimit'] = [
      '#type' => 'number',
      '#title' => t('Limit of autocomplete suggestions shown'),
      '#default_value' => $this->getSetting('autocompleteLimit'),
      '#description' => t('Limits the suggestions from the query results.'),
      '#min' => 5,
    ];

    $element['useTitlePattern'] = [
      '#type' => 'checkbox',
      '#title' => t('Use title pattern?'),
      '#default_value' => $this->getSetting('useTitlePattern'),
      '#description' => t('If checked, use title pattern, instead of field value.'),
    ];

    $element['allowOnlyExisting'] = [
      '#type' => 'checkbox',
      '#title' => t('Allow only existing values'),
      '#default_value' => $this->getSetting('allowOnlyExisting'),
      '#description' => t('Allow only existing Values to be selected - now new values may be created.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = t('Textfield size: @size', ['@size' => $this->getSetting('size')]);
    $placeholder = $this->getSetting('placeholder');
    $autocompletelimit = $this->getSetting('autocompleteLimit');
    $useTitlePattern = $this->getSetting('useTitlePattern');
    $allowOnlyExisting = $this->getSetting('allowOnlyExisting');
    if (!empty($autocompletelimit)) {
      $summary[] = t('Autocomplete limit: @autocompletelimit', ['@autocompletelimit' => $autocompletelimit]);
    }
    if (!empty($placeholder)) {
      $summary[] = t('Placeholder: @placeholder', ['@placeholder' => $placeholder]);
    }
    if (!empty($useTitlePattern)) {
      $summary[] = t('Show title pattern.');
    } else {
      $summary[] = t('Show field value.');
    }
    
    if (!empty($allowOnlyExisting)) {
      $summary[] = t('Only existing values may be selected.');
    } else {
      $summary[] = t('New values may be created.');
    }

    return $summary;
  }
  
    /**
   * Form validation handler for widget elements.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateElement(array $element, FormStateInterface $form_state) {

    // if it is empty we can opt out as that means we delete the value. There is nothing to check here then.
    if(empty($element['#value']))
      return;
    
    // only handle fields that do have an autocomplete!
    if(isset($element['#autocomplete_route_name'])) {

      // if we use the title pattern and there are brackets in the title (probably an entity id?)
      if(!empty($element['#useTitlePattern']) && preg_match("/ \(\d+\)$/", $element['#value'])) {

        $value = $element['#value'];
        $eid = substr($value, strrpos($value, '(', -1) + 1, (strrpos($value, ')', -1) - strrpos($value, '(', -0) - 1));
        $string = substr($value, 0, strrpos($value, '(', -1) - 1);

        $select = \Drupal::service('database')
          ->select('wisski_title_n_grams','w');
        

        $rows = $select
          ->fields('w', array('ngram'))
          ->condition('ngram', "%" . $select->escapeLike($string) . "%", 'LIKE')
          ->condition('ent_num', $eid, '=')
          ->range(0, 1)
          ->execute()
          ->fetchCol();
 
      
        if(empty($rows)) {
          $form_state->setError($element, t('@name field may only be filled with existing values and @value does not exist yet in the database.', ['@name' => $element['#title'], '@value' => $element['#value']]));
        }

        
      } else {
          
        // call the autocomplete controller and see if we get something there.
        // The problem is, that although there is "use title pattern" checked
        // the entity loading loads without that, so if you edit an existing
        // entity you get a field value without the entity id in the end and 
        // that might be evil...
        $url = Url::fromRoute($element['#autocomplete_route_name'], $element['#autocomplete_route_parameters'], ['query'=> ['q' =>$element['#value']], 'absolute' => TRUE]);

        $client = \Drupal::httpClient();
                
        $response = $client->get($url->toString(), [
          'headers' => [
            'Accept' => 'application/json',
          ],
        ]);
        
        $data = json_decode($response->getBody(), TRUE);
      
        $found = FALSE;
        // By Mark:
        // this is sloppy as we check for all these values. In theory this might result in wrong hits... 
        // but the wisski backend will handle these anyway :)
        foreach($data as $dat) {
          if($dat['value'] == $element['#value'] || $dat['label'] == $element['#value'] || $dat['real_value'] == $element['#value']) {
            $found = TRUE;
          }
        }
      
        if($found == FALSE) {
          $form_state->setError($element, t('@name field may only be filled with existing values and @value does not exist yet in the database.', ['@name' => $element['#title'], '@value' => $element['#value']]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $fieldId = $items[$delta]->getFieldDefinition()->get('field_name');

    $this->setSetting('fieldId', $fieldId);

    // add a validator in case "allowonlyexisting" is checked
    if($this->getSetting('allowOnlyExisting'))
      $element['#element_validate'][] = [static::class, 'validateElement'];

    $element['value'] = $element + [
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->value ?? NULL,
      '#size' => $this->getSetting('size'),
      '#placeholder' => $this->getSetting('placeholder'),
      '#useTitlePattern' => $this->getSetting('useTitlePattern'),
      '#maxlength' => $this->getFieldSetting('max_length'),
      '#attributes' => ['class' => ['js-text-full', 'text-full']],
      '#autocomplete_route_name' => 'wisski.wisski_autocomplete.autocomplete',
      '#autocomplete_route_parameters' => [
        'fieldId' => $fieldId,
      ],
    ];

    // We need to set the paramenter within some settings?
    return $element;
  }

}
