<?php

namespace Drupal\wisski_core\Entity;

use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\wisski_core\WisskiEntityInterface;
use Drupal\wisski_salz\AdapterHelper;

/**
 * Defines the entity class.
 *
 * @ContentEntityType(
 *   id = "wisski_individual",
 *   label = @Translation("Wisski Entity"),
 *   bundle_label = @Translation("Wisski Bundle"),
 *   handlers = {
 *     "storage" = "Drupal\wisski_core\WisskiStorage",
 *     "view_builder" = "Drupal\wisski_core\WisskiEntityViewBuilder",
 *     "views_data" = "Drupal\wisski_core\WisskiEntityViewsData",
 *     "list_builder" = "Drupal\wisski_core\Controller\WisskiEntityListBuilder",
 *     "list_controller" = "Drupal\wisski_core\Controller\WisskiEntityListController",
 *     "form" = {
 *       "default" = "Drupal\wisski_core\Form\WisskiEntityForm",
 *       "edit" = "Drupal\wisski_core\Form\WisskiEntityForm",
 *       "add" = "Drupal\wisski_core\Form\WisskiEntityForm",
 *       "delete" = "Drupal\wisski_core\Form\WisskiEntityDeleteForm",
 *     },
 *     "inline_form" = "Drupal\wisski_core\Form\WisskiInlineEntityForm",
 *     "access" = "Drupal\wisski_core\Controller\WisskiEntityAccessHandler",
 *   },
 *   render_cache = FALSE,
 *   field_cache = FALSE,
 *   persistent_cache = FALSE,
 *   entity_keys = {
 *     "id" = "eid",
 *     "revision" = "vid",
 *     "bundle" = "bundle",
 *     "label" = "label",
 *     "langcode" = "langcode",
 *     "uuid" = "uuid",
 *     "published" = "published"
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log"
 *   },
 *   bundle_entity_type = "wisski_bundle",
 *   permission_granularity = "bundle",
 *   admin_permission = "administer wisski",
 *   fieldable = TRUE,
 *   field_ui_base_route = "entity.wisski_bundle.edit_form",
 *   links = {
 *     "canonical" = "/wisski/navigate/{wisski_individual}",
 *     "delete-form" = "/wisski/navigate/{wisski_individual}/delete",
 *     "add-form" = "/wisski/create/{wisski_bundle}",
 *     "edit-form" = "/wisski/navigate/{wisski_individual}/edit",
 *     "admin-form" = "/admin/structure/wisski_core/{wisski_bundle}/edit",
 *   },
 *   base_table = "wisski_basetable",
 *   data_table = "wisski_datatable",
 *   revision_table = "wisski_revision",
 *   revision_data_table = "wisski_data_revision",
 *   show_revision_ui = TRUE,
 *   translatable = TRUE
 * )
 */
class WisskiEntity extends EditorialContentEntityBase implements WisskiEntityInterface {

  use StringTranslationTrait;

  /**
   * The original values of the entity.
   *
   * @var array
   */
  protected $originalValues;

  /**
   * The label of the entity.
   *
   * @var string
   */
  protected $label = NULL;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {

    $fields = parent::baseFieldDefinitions($entity_type);
    $fields['eid'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Entity ID'))
      ->setDescription(t('The ID of this entity.'))
      ->setReadOnly(TRUE)
      ->setSetting('unsigned', TRUE);

    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setDescription(t("This entity's UUID."))
      ->setReadOnly(TRUE);

    $fields['vid'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Revision ID'))
      ->setDescription(t('The revision ID.'))
      ->setReadOnly(TRUE)
      ->setSetting('unsigned', TRUE);

    $fields['bundle'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Bundle'))
      ->setDescription(t('The bundle.'))
      ->setSetting('target_type', 'wisski_bundle')
      ->setReadOnly(TRUE)
      ->setDisplayOptions('view', [
        'region' => 'hidden',
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['description'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Description'))
      ->setDescription(t('The description.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['wisski_uri'] = BaseFieldDefinition::create('uri')
      ->setLabel(t('WissKI URI'))
      ->setDescription(t('The WissKI URI from the TS (in case you need it)'))
      ->setReadOnly(TRUE)
      ->setDisplayOptions('view', [
        'region' => 'hidden',
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language'))
      ->setDescription(t('Language of the entity.'))
      ->setReadOnly(TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'region' => 'hidden',
        'label' => 'hidden',
      ]);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity name'))
      ->setDescription(t('The human readable name of this entity.'))
      ->setRequired(FALSE)
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
        'region' => 'hidden',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'region' => 'hidden',
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Creator ID'))
      ->setDescription(t('The user ID of the entity creator.'))
      ->setRevisionable(TRUE)
      ->setDefaultValue(0)
      ->setSetting('target_type', 'user')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the WissKI Entity was created.'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'region' => 'hidden',
        'type' => 'timestamp',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'region' => 'hidden',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the WissKI Entity was last edited.'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Published'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE);

    $set = \Drupal::configFactory()->getEditable('wisski_core.settings');
    $use_status = $set->get('enable_published_status_everywhere');

    if ($use_status) {
      $fields['status']->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => 120,
      ])
        ->setDisplayConfigurable('form', TRUE);
    }

    $fields['preview_image'] = BaseFieldDefinition::create('image')
      ->setLabel(t('Preview Image'))
      ->setDescription(t('A reference to an image file that is used as the preview image of the entity'))
      ->setSetting('target_type', 'file')
      ->setDefaultValue(NULL)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    $title = parent::label();

    if ($this->label === NULL) {
      $title = wisski_core_generate_title($this);
      if (isset($title[0])) {
        $title = $this->t("@title (new)", ['@title' => $title[0]["value"]]);
      }
      else {
        // We probably got a language.
        if (is_array($title)) {
          $title = current($title);
          $title = $this->t("@title (new)", ['@title' => $title[0]["value"]]);
        }
      }
      $this->label = $title;
    }

    return $title;
  }

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = "canonical", array $options = []) {
    $url = parent::toUrl($rel, $options);
    $use_permalink = \Drupal::service('config.factory')->getEditable('wisski_core.settings')->get('use_get_canonical');
    if (!$use_permalink) {
      return $url;
    }
    // Catch the canonical route links.
    if ($url->getRouteName() == "entity.wisski_individual.canonical") {
      // @todo Sometimes the wisski_uri is not loaded into the entity.
      // Not sure why, but we manually lookup the uri from the adapter
      // in that case.
      $uri = $this->get("wisski_uri")->value;
      if (empty($uri)) {
        // Take the first URI, hopefully that's the correct one.
        $uri = current(AdapterHelper::getUrisForDrupalId($this->id(), NULL, FALSE));
      }
      // Replace canonical route by /wisski/get route.
      $url = Url::fromRoute("entity.wisski_individual.lod_get", ['uri' => $uri], $options);
    }
    return $url;
  }

  /**
   * Returns information about fields.
   *
   * @return array
   *   An array containing information about the requested fields.
   *   The array contains the following keys:
   *   - key: The key of the field.
   *   - field_name: The name of the field.
   *   - property: The property of the field.
   *   - value: The value of the field.
   */
  public function tellMe() {
    $keys = func_get_args();
    $return = [];
    foreach ($keys as $key) {
      $field_name = $this->getEntityType()->getKey($key);
      $definition = $this->getFieldDefinition($field_name);
      $property = $definition->getFieldStorageDefinition()->getMainPropertyName();
      $value = $this->get($field_name)->$property;
      $return[$key] = [
        'key' => $key,
        'field_name' => $field_name,
        'property' => $property,
        'value' => $value,
      ];
    }
    return $return;
  }

  /**
   * Saves the original values of the entity.
   *
   * @param \Drupal\wisski_core\WisskiStorage $storage
   *   The storage object.
   */
  public function saveOriginalValues($storage) {
    // If there were already original values - do nothing.
    if (empty($this->originalValues)) {
      $this->originalValues = $this->extractFieldData($storage);
    }
  }

  /**
   * Returns the original values of the entity.
   *
   * @return array
   *   An array containing the original values of the entity.
   */
  public function getOriginalValues() {

    return $this->originalValues;
  }

  /**
   * Returns the values of the entity.
   *
   * @param \Drupal\wisski_core\WisskiStorage $storage
   *   The storage object.
   * @param bool $save_field_properties
   *   Whether to save the field properties.
   */
  public function getValues($storage, $save_field_properties = FALSE) {
    return [$this->extractFieldData($storage, $save_field_properties), $this->originalValues];
  }

  /**
   * Extracts field data from the entity.
   *
   * @param object $storage
   *   The storage object.
   * @param bool $save_field_properties
   *   Whether to save field properties.
   *
   * @return array
   *   The extracted field data.
   */
  protected function extractFieldData($storage, $save_field_properties = FALSE) {
    $out = [];

    $fields_to_save = [];

    // This might be x-default, so we don't use that here.
    $language = $this->language()->getId();

    if ($save_field_properties) {
      // Clear the field values for this field in entity in bundle.
      \Drupal::database()->delete('wisski_entity_field_properties')
        ->condition('eid', $this->id())
        ->condition('bid', $this->bundle())
        ->condition('lang', $language)
        ->execute();

      // Prepare the insert query.
      // @todo Drupal Rector Notice: Please delete the following comment after
      // you've made any necessary changes.
      // You will need to use `\Drupal\core\Database\Database::getConnection()`
      // if you do not yet have access to the container here.
      $query = \Drupal::database()->insert('wisski_entity_field_properties')
        ->fields([
          'eid',
          'bid',
          'fid',
          'delta',
          'ident',
          'properties',
          'lang',
        ]);

    }

    // $this is iterable itself, iterates over field list.
    foreach ($this as $field_name => $field_item_list) {

      $out[$field_name] = [];

      // The main property is for all items of the field the same,
      // so we buffer it here.
      $main_property = NULL;

      foreach ($field_item_list as $weight => $field_item) {

        $field_values = $field_item->getValue();
        $field_def = $field_item->getFieldDefinition()->getFieldStorageDefinition();

        if (!empty($field_values) && method_exists($field_def, 'getDependencies') && in_array('file', $field_def->getDependencies()['module'])) {
          // When loading we assume $target_id to be the file uri.
          // This is a workaround since Drupal File IDs do not
          // carry any information when not in drupal context.
          if (!isset($field_values['target_id'])) {
            continue;
          }
          $field_values['target_id'] = $storage->getPublicUrlFromFileId($field_values['target_id']);
        }

        // If it is empty it is probably not correctly initialized.
        if (empty($main_property)) {
          $main_property = $field_item->mainPropertyName();
        }

        // If it is not initalized by now we do it by hand.
        if (empty($main_property) || (!empty($field_values) && empty($field_values[$main_property]))) {

          // This is not the best heuristic.
          // Better save something that is bigger...
          if (!empty($field_values)) {
            $main_property = current(array_keys($field_values));
          }
          else {
            $main_property = "value";
          }
        }

        // We transfer the main property name to the adapters.
        $out[$field_name]['main_property'] = $main_property;
        // Gathers the ARRAY of field properties for each field list item.
        // e.g. $out[$field_name][] = [
        // 'value' => 'Hans Wurst',
        // 'format' => 'basic_html',
        // ].
        $out[$field_name][$weight] = $field_values;

        if ($save_field_properties && !empty($this->id())) {

          if (!isset($field_values[$main_property])) {
            \Drupal::messenger()->addWarning("I could not store value " . serialize($field_values) . " for this field because the main property (" . $main_property . ") is not in there.");
            continue;
          }

          $fields_to_save = [
            'eid' => $this->id(),
            'bid' => $this->bundle(),
            'fid' => $field_name,
            'delta' => $weight,
            'ident' => strlen($field_values[$main_property]) > 1000 ? mb_substr($field_values[$main_property], 0, 1000) : $field_values[$main_property],
            // This formerly was in here.
            // The problem however is that this could never be written,
            // because we don't know what is the disamb...
            'properties' => serialize($field_values),
            'lang' => $language,
          ];

          // Add the values to the insert query.
          $query->values($fields_to_save);
        }
      }

      // Do not do this per field, do it as a bunch.
      // Execute the insert query.
      if ($save_field_properties && !empty($this->id())) {
        $query->execute();
      }

      if (!isset($out[$field_name][0]) || empty($out[$field_name][0]) || empty($out[$field_name][0][$main_property])) {
        unset($out[$field_name]);
      }
    }

    if ($save_field_properties && !empty($this->id())) {
      // @todo Somehow this seems to be triggered twice?!
      $query->execute();
    }

    return $out;
  }

  /**
   * Gets field data types.
   *
   * @return array
   *   Array of field data types.
   */
  public function getFieldDataTypes() {
    $types = [];

    // Gather a list of referenced entities.
    foreach ($this->getFields() as $field_name => $field_items) {
      foreach ($field_items as $field_item) {
        // Loop over all properties of a field item.
        foreach ($field_item->getProperties(TRUE) as $property_name => $property) {
          $types[$field_name][$property_name][] = get_class($property);
        }
      }
    }

    return $types;
  }

  /**
   * {@inheritdoc}
   *
   * Is the entity new? We cannot answer that question with certainty, so we
   * always say NO unless we definitely know it better.
   */
  public function isNew() {
    return !empty($this->enforceIsNew);
  }

}
