<?php

namespace Drupal\wisski_pathbuilder\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\wisski_pathbuilder\PathbuilderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to display and manage missing and (semi-)orphaned paths.
 */
class PathUsageForm extends FormBase {

  /**
   * The pathbuilder manager service.
   *
   * @var \Drupal\wisski_pathbuilder\PathbuilderManager
   */
  protected $pathbuilderManager;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new PathUsageForm object.
   *
   * @param \Drupal\wisski_pathbuilder\PathbuilderManager $pathbuilder_manager
   *   The pathbuilder manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(PathbuilderManager $pathbuilder_manager, EntityTypeManagerInterface $entity_type_manager) {
    $this->pathbuilderManager = $pathbuilder_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wisski_pathbuilder.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'wisski_pathbuilder_path_usage';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $usage = $this->pathbuilderManager->getOrphanedPaths();
    $missing_paths = $this->getMissingPaths();

    $table = [
      '#type' => 'tableselect',
      '#header' => [
        'name' => $this->t('Name'),
        'pid' => $this->t('ID'),
        'pb' => $this->t('Pathbuilder'),
        'usage' => $this->t('Usage'),
      ],
      '#empty' => $this->t('No paths available'),
      // Filled below.
      '#options' => [],
    ];

    // Handle missing/ominous paths that are referenced but don't exist.
    if (!empty($missing_paths)) {
      foreach ($missing_paths as $missing_id => $pb_info) {
        $rowid = "missing.{$pb_info['pbid']}.$missing_id";
        $table['#options'][$rowid] = [
          '#attributes' => ['class' => 'missing-path'],
          'name' => $this->t('Missing Path'),
          'pid' => $missing_id,
          'pb' => $pb_info['pb_name'],
          'usage' => $this->t('Referenced but not found'),
        ];
      }
    }

    if (!empty($usage['orphaned']) || !empty($usage['semiorphaned'])) {

      $paths = $this->entityTypeManager->getStorage('wisski_path')->loadMultiple();
      $pbs = $this->entityTypeManager->getStorage('wisski_pathbuilder')->loadMultiple();

      foreach ($usage['orphaned'] as $pid) {
        /** @var \Drupal\wisski_path\Entity\Path $path */
        $path = $paths[$pid];
        $rowid = "orphaned.none.$pid";
        $table['#options'][$rowid] = [
          '#attributes' => ['class' => 'orphaned'],
          'name' => $path->getName(),
          'pid' => $pid,
          'pb' => '',
          'usage' => $this->t('In no Pathbuilder'),
        ];
      }

      foreach ($usage['semiorphaned'] as $pid => $pbids) {
        /** @var \Drupal\wisski_path\Entity\Path $path */
        $path = $paths[$pid];
        foreach ($pbids as $pbid) {
          $rowid = "semiorphaned.$pbid.$pid";

          /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
          $pb = $pbs[$pbid];
          $pb_info = $pb->getName();
          if (isset($usage['home'][$pid])) {
            $pb_info .= '; Regular in ' . implode(", ", $usage['home'][$pid]);
          }
          $table['#options'][$rowid] = [
            '#attributes' => ['class' => 'semi-orphaned'],
            'name' => $path->getName(),
            'pid' => $pid,
            'pb' => $pb_info,
            'usage' => $this->t('In list but not in tree'),
          ];
        }
      }

    }

    // Add CSS library for styling.
    $form['#attached']['library'][] = 'wisski_pathbuilder/path_usage';

    // Add legend explaining the color coding.
    $form['legend'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['path-usage-legend']],
      'title' => [
        '#type' => 'html_tag',
        '#tag' => 'h3',
        '#value' => $this->t('Legend'),
      ],
      'missing' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['path-usage-legend-item']],
        'color' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['path-usage-legend-color', 'missing']],
          '#value' => '',
        ],
        'text' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#value' => $this->t('Missing paths - referenced but not found as path entity'),
        ],
      ],
      'orphaned' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['path-usage-legend-item']],
        'color' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['path-usage-legend-color', 'orphaned']],
          '#value' => '',
        ],
        'text' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#value' => $this->t('Orphaned paths - not in any pathbuilder'),
        ],
      ],
      'semi_orphaned' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['path-usage-legend-item']],
        'color' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['path-usage-legend-color', 'semi-orphaned']],
          '#value' => '',
        ],
        'text' => [
          '#type' => 'html_tag',
          '#tag' => 'span',
          '#value' => $this->t('Semi-orphaned paths - in list but not in tree'),
        ],
      ],
    ];

    $form['statistics'] = [
      '#type' => 'markup',
      '#value' => $this->t(
        'There are @o orphaned, @s semi-orphaned, and @m missing paths.',
        [
          '@o' => count($usage['orphaned']),
          '@s' => count($usage['semiorphaned']),
          '@m' => count($missing_paths),
        ]
      ),
    ];

    // Add CSS class to table for styling.
    $table['#attributes']['class'][] = 'path-usage-table';
    $form['table'] = $table;
    $form['actions'] = [
      'delete' => [
        '#type' => 'submit',
        '#value' => $this->t('Delete checked'),
        '#submit' => ['::deleteChecked'],
      ],
    ];

    return $form;
  }

  /**
   * Delete checked paths.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function deleteChecked($form, FormStateInterface $form_state) {

    $values = array_filter($form_state->getValue('table'));

    $pbs = $this->entityTypeManager->getStorage('wisski_pathbuilder')->loadMultiple();
    $pbpaths = [];
    $pb_trees = [];

    $cnt_semi = 0;
    $cnt_orph = 0;
    $cnt_missing = 0;
    foreach ($values as $rowid) {
      [$usage, $pbid, $pid] = explode('.', $rowid);

      if ($usage == 'semiorphaned') {
        if (!isset($pbpaths[$pbid])) {
          /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
          $pb = $pbs[$pbid];
          $pbpaths[$pbid] = $pb->getPbPaths();
        }
        unset($pbpaths[$pbid][$pid]);
        $cnt_semi++;
      }
      elseif ($usage == 'orphaned') {
        $path = $this->entityTypeManager->getStorage('wisski_path')->load($pid);
        if (!empty($path)) {
          $path->delete();
        }
        $cnt_orph++;
      }
      elseif ($usage == 'missing') {
        // Remove missing path references from pathbuilder.
        if (!isset($pbpaths[$pbid])) {
          /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
          $pb = $pbs[$pbid];
          $pbpaths[$pbid] = $pb->getPbPaths();
        }
        // Remove from pbpaths.
        unset($pbpaths[$pbid][$pid]);

        // Remove from path tree as well.
        if (!isset($pb_trees[$pbid])) {
          /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
          $pb = $pbs[$pbid];
          $pb_trees[$pbid] = $this->removeFromPathTree($pb->getPathTree(), $pid);
        }
        else {
          $pb_trees[$pbid] = $this->removeFromPathTree($pb_trees[$pbid], $pid);
        }
        $cnt_missing++;
      }
    }

    // Update pathbuilders with cleaned pbpaths and trees.
    foreach ($pbpaths as $pbid => $pbpath_array) {
      /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
      $pb = $pbs[$pbid];
      $pb->setPbPaths($pbpath_array);
      if (isset($pb_trees[$pbid])) {
        $pb->setPathTree($pb_trees[$pbid]);
      }
      $pb->save();
    }

    // Update pathbuilders that only had tree changes.
    foreach ($pb_trees as $pbid => $tree) {
      if (!isset($pbpaths[$pbid])) {
        /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
        $pb = $pbs[$pbid];
        $pb->setPathTree($tree);
        $pb->save();
      }
    }

    $this->messenger()->addStatus($this->t('@c orphaned paths have been deleted.', ['@c' => $cnt_orph]));
    $this->messenger()->addStatus($this->t('@c semi-orphaned paths have been removed from their pathbuilders.', ['@c' => $cnt_semi]));
    $this->messenger()->addStatus($this->t('@c missing path references have been cleaned from pathbuilders.', ['@c' => $cnt_missing]));
    if ($cnt_semi) {
      $this->messenger()->addStatus($this->t('There may be paths that have become orphaned as they have been removed from their pathbuilders.'));
    }

  }

  /**
   * Get missing paths that are referenced but don't exist as entities.
   *
   * @return array
   *   Array of missing path IDs with their pathbuilder information.
   */
  protected function getMissingPaths() {
    $missing_paths = [];
    $pbs = $this->entityTypeManager->getStorage('wisski_pathbuilder')->loadMultiple();
    $paths = $this->entityTypeManager->getStorage('wisski_path')->loadMultiple();

    foreach ($pbs as $pbid => $pb) {
      /** @var \Drupal\wisski_pathbuilder\Entity\Pathbuilder $pb */
      $pbpaths = $pb->getPbPaths();
      $tree_paths = $this->getPathIdsInPathTree($pb);

      // Check pbpaths for missing path entities.
      foreach ($pbpaths as $pid => $pbpath) {
        if (!isset($paths[$pid])) {
          $missing_paths[$pid] = [
            'pbid' => $pbid,
            'pb_name' => $pb->getName(),
          ];
        }
      }

      // Check tree paths for missing path entities.
      foreach ($tree_paths as $pid) {
        if (!isset($paths[$pid]) && !isset($missing_paths[$pid])) {
          $missing_paths[$pid] = [
            'pbid' => $pbid,
            'pb_name' => $pb->getName(),
          ];
        }
      }
    }

    return $missing_paths;
  }

  /**
   * Get all path ids in a path tree.
   *
   * @param object $pb
   *   The pathbuilder entity.
   *
   * @return array
   *   Array of path IDs.
   */
  protected function getPathIdsInPathTree($pb) {
    $ids = [];
    $agenda = $pb->getPathTree();
    while ($node = array_shift($agenda)) {
      $ids[$node['id']] = $node['id'];
      $agenda = array_merge($agenda, $node['children']);
    }
    return $ids;
  }

  /**
   * Remove a path ID from the path tree structure.
   *
   * @param array $tree
   *   The path tree array.
   * @param string $remove_id
   *   The path ID to remove.
   *
   * @return array
   *   The cleaned tree array.
   */
  protected function removeFromPathTree(array $tree, $remove_id) {
    $clean_tree = [];
    foreach ($tree as $node) {
      if ($node['id'] !== $remove_id) {
        if (!empty($node['children'])) {
          $node['children'] = $this->removeFromPathTree($node['children'], $remove_id);
        }
        $clean_tree[] = $node;
      }
    }
    return $clean_tree;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Does not occur.
  }

}
