<?php

namespace Drupal\wisski_adapter_sparql11_pb\Form;

use \Drupal\Core\Form\FormBase;
use \Drupal\Core\Form\FormStateInterface;
use \Drupal\Core\Pager\Pager;

class Sparql11EndpointQueryForm extends FormBase {

  /**
   * @inheritdoc
   */
  public function getFormId() {
    return "wisski_adapter_sparql11_pb_endpoint_query_form";
  }

  /**
   * @inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state, $wisski_salz_adapter = NULL) {

    $adapter_id = $wisski_salz_adapter;

    // Display warning if non existing adapter was requested
    $adapters = \Drupal::entityTypeManager()->getStorage("wisski_salz_adapter")->loadMultiple();
    if (!in_array($adapter_id, array_keys($adapters))) {
      $this->messenger()->addWarning($this->t("No such adapter '@requested'! Available adapters are: @adapters", [
        "@adapters" => join(", ", array_keys($adapters)),
        "@requested" => $adapter_id
      ]));
      return $form;
    }

    $query = \Drupal::request()->query->get("query") ?? "";

    $query_fs = $form_state->getUserInput();
    if(!empty($query_fs["query"])){
      $query = $query_fs["query"];
    }

    $form["#title"] = "Query $adapter_id Endpoint";
    $form["text"]["#markup"] = "Any SPARQL 1.1 SELECT or ASK Query can be stated here - no updates. You can also use it as a dynamic endpoint via GET/POST with the query parameter.";

    $form["query"] = [
      "#type" => "textarea",
      "#attributes" => [
        "class" => ["codemirror"],
      ],
      "#value" => $query
    ];

    $form["submit"] = [
      "#type" => "submit",
      "#value" => $this->t("Execute Query"),
    ];

    $form["#attached"]["library"][] = "wisski_adapter_sparql11_pb/codemirror";

    if (empty($query)) {
      $form["query"]["#value"]= "SELECT * WHERE { ?s ?p ?o }";
      return $form;
    }


    /** @var \Drupal\wisski_salz\Entity\Adapter */
    $adapter = $adapters[$adapter_id];
    /** @var \Drupal\wisski_adapter_sparql11_pb\Plugin\wisski_salz\Engine\Sparql11EngineWithPB */
    $engine = $adapter->getEngine();
    try {
      // Ideally we would want this in the submitForm, but then result pagination breaks.
      $result = $engine->directQuery($query);
    }
    catch(\EasyRDF\Exception $e) {
      // TODO: For this to work it requires the error to be thrown in directQuery and not caught...
      // Check with Mark if that is possible or if WissKI completely breaks if this error gets out of directQuery.
      // If that is not possible at least put this message into the messenger in directQuery.
      $this->messenger()->addError($e->getMessage());
      return $form;
    }

    // Query results
    $form["results"] = [
      "#type" => "container"
    ];

    $items_per_page = 10;
    $total = count((array) $result);
    // Set up pager
    $pager = $this->createPager($total, $items_per_page);
    $current_page = $pager->getCurrentPage();
    // Chunk items
    $chunks = array_chunk((array) $result, $items_per_page);
    $current_items = $chunks[$current_page];


    // Build the header for the table
    $header = [];
    foreach ($result->getFields() as $key) {
      $header[$key] = [
        "data" => $key,
        "specifier" => $key,
      ];
    }
    // Build the rows for the table
    $rows = [];
    foreach ($current_items as $values) {
      $row = [];
      foreach (array_keys($header) as $key) {
        $value = property_exists($values, $key) ? $values->$key : NULL;
        if ($value instanceof \EasyRDF\Resource) {
          $row[$key] = $value->getUri();
        }
        elseif ($value instanceof \EasyRDF\Literal) {
          $row[$key] = $value->getValue();
        }
        else {
          $row[$key] = $value;
        }

      }
      $rows[] = $row;
    }

    // Calculate indexes of first/last page items
    $page_start = $current_page * $items_per_page;
    $page_end = $page_start + count($current_items);

    $form["result_info"] = [
      "#markup" => "Showing results $page_start - $page_end of $total",
      "#group" => "results",
    ];
    $form["result"] = [
      "#theme" => "table",
      "#title" => "Results",
      "#header" => $header,
      "#rows" => $rows,
      "#attributes" => [
        "class" => ["result-table"],
      ],
      "#group" => "results",
    ];
    $form["pager"] = [
      "#type" => "pager",
      "#group" => "results",
    ];

    return $form;
  }

  /**
   * Set up a pager for the given items
   *
   * @param int $total
   *   Total number of items to paginate
   *
   * @param integer $limit
   *   Number of items per page
   *
   * @return \Drupal\Core\Pager\Pager
   */
  function createPager(int $total, int $limit): Pager {
    /** @var \Drupal\Core\Pager\PagerManagerInterface */
    $pager_manager = \Drupal::service("pager.manager");
    return $pager_manager->createPager($total, $limit);
  }

  /**
   * @inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Put the query into the query params in order for the pager to work
    $query = $form_state->getValue("query");
    if ($query) {
      \Drupal::request()->query->set("query", $query);
    }

    $form_state->setValue("query", $query);

    // we want to get back where we came from
    $form_state->setRebuild(TRUE);
    $form_state->disableRedirect(TRUE);
  }
}
