<?php

namespace Drupal\wisski_adapter_sparql11_pb\Plugin\wisski_salz\Engine;

use Drupal\wisski_pathbuilder\Entity\WisskiPathEntity;
use Drupal\wisski_pathbuilder\Entity\WisskiPathbuilderEntity;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\wisski_salz\Entity\Adapter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\wisski_salz\Plugin\wisski_salz\Engine\Sparql11Engine;
use Drupal\wisski_pathbuilder\PathbuilderEngineInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Language\LanguageInterface;


use Drupal\wisski_adapter_sparql11_pb\Query\Query;
use EasyRdf\Format as EasyRdf_Format;

/**
 * Standard Sparql 1.1 endpoint adapter engine.
 *
 * @Engine(
 *   id = "sparql11_with_pb",
 *   name = @Translation("Sparql 1.1 With Pathbuilder"),
 *   description = @Translation("Provides access to a SPARQL 1.1 endpoint and
 *   is configurable via a Pathbuilder")
 * )
 */
class Sparql11EngineWithPB extends Sparql11Engine implements PathbuilderEngineInterface {

  protected $allow_inverse_property_pattern;

  /**
   *
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'allow_inverse_property_pattern' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  // Public function setBaseFieldFromStoreForUri($uri, $basefield, $value) {
  // dpm("s1et base field was called with $uri, $basefield and $value.");
  //    Return NULL;
  //  }.

  /**
   * {@inheritdoc}
   */
  public function supportsOntology() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function supportsTriples() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {

    // This does not exist.
    parent::setConfiguration($configuration);
    $this->allow_inverse_property_pattern = $this->configuration['allow_inverse_property_pattern'];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return [
      'allow_inverse_property_pattern' => $this->allow_inverse_property_pattern,
    ] + parent::getConfiguration();
  }

  /*******************
   * BASIC Pathbuilder Support .***********************/

  /**
   * {@inheritdoc}
   */
  public function providesFastMode() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function providesCacheMode() {

    return TRUE;
  }

  /**
   * Returns TRUE if the cache is pre-computed and ready to use, FALSE otherwise.
   */
  public function isCacheSet() {
    // See $this->doTheReasoning()
    // and $this->getPropertiesFromCache() / $this->getClassesFromCache
    // the rasoner sets all reasoning based caches i.e. it is sufficient to check, that one of them is set
    // if ($cache = \Drupal::cache()->get('wisski_reasoner_properties')) return TRUE;
    // return FALSE;.
    return $this->isPrepared();
  }

  /**
   * {@inheritdoc}
   * returns the possible next steps in path creation, if
   * $this->providesFastMode() returns TRUE then this MUST react fast i.e. in
   * the blink of an eye if $fast_mode = TRUE and it MUST return the complete
   * set of options if $fast_mode=FALSE otherwise it should ignore the
   * $fast_mode parameter
   */
  public function getPathAlternatives($history = [], $future = [], $fast_mode = FALSE, $empty_uri = 'empty') {

    // \Drupal::logger('WissKI path alternatives: '.($fast_mode ? 'fast mode' : "normal mode"))->debug('History: '.serialize($history)."\n".'Future: '.serialize($future));
    $search_properties = NULL;

    $last = NULL;
    if (!empty($history)) {
      $candidate = array_pop($history);
      $inv_sign = '';
      if ($this->allow_inverse_property_pattern && $candidate[0] == '^') {
        $candidate = substr($candidate, 1);
        $inv_sign = '^';
      }
      if ($candidate === $empty_uri) {
        // \Drupal::logger('WissKI path alternatives')->error('Not a valid URI: "'.$candidate.'"');
        // as a fallback we assume that the full history is given so that every second step is a property
        // we have already popped one element, so count($history) is even when we need a property
        $search_properties = (0 === count($history) % 2);
      }
      elseif ($this->isValidUri('<' . $candidate . '>')) {
        $last = "$inv_sign$candidate";
        if ($this->isAProperty($candidate) === FALSE) {
          $search_properties = TRUE;
        }
      }
      else {
        if (WISSKI_DEVEL) {
          \Drupal::logger('WissKI path alternatives')
            ->debug('invalid URI ' . $candidate);
        }
        return [];
      }
    }

    $next = NULL;
    if (!empty($future)) {
      $candidate = array_shift($future);
      $inv_sign = '';
      if ($this->allow_inverse_property_pattern && $candidate[0] == '^') {
        $candidate = substr($candidate, 1);
        $inv_sign = '^';
      }
      if ($candidate !== $empty_uri) {
        if ($this->isValidUri('<' . $candidate . '>')) {
          $next = "$inv_sign$candidate";
          if ($search_properties === NULL) {
            if ($this->isAProperty($candidate) === FALSE) {
              $search_properties = TRUE;
            }
          }
          elseif ($this->isAProperty($candidate) === $search_properties) {
            $this->messenger()->addError('History and Future are inconsistent');
          }
        }
        else {
          if (WISSKI_DEVEL) {
            \Drupal::logger('WissKI path alternatives')
              ->debug('invalid URI ' . $candidate);
          }
          return [];
        }
      }
    }
    // dpm(serialize($search_properties), "sp");
    // \Drupal::logger('WissKI next '.($search_properties ? 'properties' : 'classes'))->debug('Last: '.$last.', Next: '.$next);
    // $search_properties is TRUE if and only if last and next are valid URIs and no owl:Class-es.
    if ($search_properties) {
      $return = $this->nextProperties($last, $next, $fast_mode);
    }
    else {
      $return = $this->nextClasses($last, $next, $fast_mode);
    }

    // dpm(func_get_args()+array('result'=>$return),__FUNCTION__);
    return $return;
  }

  /**
   * @{inheritdoc}
   */
  // Public function getPathAlternatives($history = [], $future = []) {
  //
  //  \Drupal::logger('WissKI SPARQL Client')->debug("normal mode");
  //    if (empty($history) && empty($future)) {
  //
  //      return $this->getClasses();
  //
  //    } elseif (!empty($history)) {
  //
  //      $last = array_pop($history);
  //      $next = empty($future) ? NULL : $future[0];
  //
  //      if ($this->isaProperty($last)) {
  //        return $this->nextClasses($last, $next);
  //      } else {
  //        return $this->nextProperties($last, $next);
  //      }
  //    } elseif (!empty($future)) {
  //      $next = $future[0];
  //      if ($this->isaProperty($next))
  //        return $this->getClasses();
  //      else
  //        return $this->getProperties();
  //    } else {
  //      return [];
  //    }
  //
  //
  //  }.

  /**
   * @{inheritdoc}
   */
  public function getPrimitiveMapping($step) {

    $out = $this->retrieve('primitives', 'property', 'class', $step);
    // dpm($step);
    // dpm($out);
    if (!empty($out)) {
      return $out;
    }

    // In case of properties we can skip this.
    if ($step[0] == '^') {
      return [];
    }

    $info = [];

    // This might need to be adjusted for other standards than rdf/owl.
    $query =
      "SELECT DISTINCT ?property "
      . "WHERE { { "
      . "?property a owl:DatatypeProperty . "
      . "?property rdfs:domain ?d_superclass. ";
    // ."<$step> rdfs:subClassOf* ?d_superclass. } }"
    // ."{ GRAPH ?g3 { ?property rdfs:subPropertyOf* ?d_subprop } } . ";
    // ."{ GRAPH ?g3 { ?property rdfs:subPropertyOf* ?d_subprop } } . { GRAPH ?g4 { ?d_subprop rdfs:domain ?d_superclass . } . } . "
    // ."{ GRAPH ?g5 { <$step> rdfs:subClassOf* ?d_superclass. } . } . }"
    // By Mark: TODO: Please check this. I have absolutely
    // no idea what this does, I just copied it from below
    // and I really really hope that Dorian did know what it
    // does and it will work forever.
    $query .=
      "{"
      . "{?d_def_prop rdfs:domain ?d_def_class.}"
      . " UNION "
      . "{"
      . "?d_def_prop owl:inverseOf ?inv. "
      . "?inv rdfs:range ?d_def_class. "
      . "}"
      . "} "
      . "<$step> rdfs:subClassOf* ?d_def_class. "
      . "{"
      . "{?d_def_prop rdfs:subPropertyOf* ?property.}"
      . " UNION "
      . "{ "
      . "?property rdfs:subPropertyOf+ ?d_def_prop. "
      . " FILTER NOT EXISTS {"
      . "{ "
      . "?mid_prop rdfs:subPropertyOf+ ?d_def_prop. "
      . "?property rdfs:subPropertyOf* ?mid_prop. "
      . "}"
      . "{"
      . "{?mid_prop rdfs:domain ?any_domain.}"
      . " UNION "
      . "{ "
      . "?mid_prop owl:inverseOf ?mid_inv. "
      . "?mid_inv rdfs:range ?any_range. "
      . "}"
      . "}"
      . "}"
      . "}"
      . "}}}";

    $result = $this->directQuery($query);
    // dpm($query, 'res');.
    if (count($result) == 0) {
      return [];
    }

    $output = [];
    foreach ($result as $obj) {
      $prop = $obj->property->getUri();
      $output[$prop] = $prop;
    }
    uksort($output, 'strnatcasecmp');
    return $output;
  }

  /**
   *
   */
  public function getStepInfo($step, $history = [], $future = []) {

    $info = [];

    $query = "SELECT DISTINCT ?label WHERE { GRAPH ?g { <$step> <http://www.w3.org/2000/01/rdf-schema#label> ?label . } } LIMIT 1";
    $result = $this->directQuery($query);
    if (count($result) > 0) {
      $info['label'] = $result[0]->label->getValue();
    }

    $query = "SELECT DISTINCT ?comment WHERE { GRAPH ?g { <$step> <http://www.w3.org/2000/01/rdf-schema#comment> ?comment . } } LIMIT 1";
    $result = $this->directQuery($query);
    if (count($result) > 0) {
      $info['comment'] = $result[0]->comment->getValue();
    }

    return $info;
  }

  /**
   *
   */
  public function isaProperty($p) {

    // You cannot use GRAPH in an ASK query
    // return $this->directQuery("ASK { GRAPH ?g { <$p> a owl:ObjectProperty . } }")->isTrue();
    // we obviously have to solve it via SELECT.
    $result = $this->directQuery(
      "SELECT * WHERE {"
      . "VALUES ?x {<$p>} "
      . "GRAPH ?g { "
      . "{ ?x a owl:ObjectProperty . } UNION { ?x a rdf:Property.}"
      . "}"
      . "}"
    );
    return $result->numRows() > 0;
  }

  /**
   *
   */
  public function getClasses() {

    $out = $this->retrieve('classes', 'class');
    if (!empty($out)) {
      return $out;
    }
    $query = "SELECT DISTINCT ?class WHERE { "
      . "{ GRAPH ?g1 {?class a owl:Class} }"
      . "UNION "
      . "{ GRAPH ?g2 {?class a rdfs:Class} }"
      . "UNION "
      . "{ GRAPH ?g3 {?ind a ?class. ?class a ?type} }"
      . " . FILTER(!isBlank(?class))"
      . "} ";
    $result = $this->directQuery($query);

    if ($result) {
      if (count($result) > 0) {
        $out = [];
        foreach ($result as $obj) {
          $class = $obj->class->getUri();
          $out[$class] = $class;
        }
        uksort($out, 'strnatcasecmp');
        return $out;
      }
    }
    else {
      return FALSE;
    }
  }

  /**
   *
   */
  public function getProperties() {

    $out = $this->retrieve('properties', 'property');
    if (!empty($out)) {
      return $out;
    }
    $query = "SELECT DISTINCT ?property WHERE { "
      . "{ GRAPH ?g1 {?property a owl:ObjectProperty .} } "
      . "UNION "
      . "{ GRAPH ?g2 {?property a rdf:Property .} } "
      . " . "
      . "FILTER(!isBlank(?property))"
      . "} ";
    $result = $this->directQuery($query);

    if ($result) {
      if (count($result) > 0) {
        $out = [];
        foreach ($result as $obj) {
          $class = $obj->property->getUri();
          $out[$class] = $class;
        }
        uksort($out, 'strnatcasecmp');
        return $out;
      }
    }
    else {
      return FALSE;
    }
  }

  /**
   *
   */
  public function nextProperties($class = NULL, $class_after = NULL, $fast_mode = FALSE) {

    if (!isset($class) && !isset($class_after)) {
      return $this->getProperties();
    }
    // \Drupal::logger(__METHOD__)->debug('class: '.$class.', class_after: '.$class_after);
    $output = $this->getPropertiesFromCache($class, $class_after);

    if ($output === FALSE) {
      // drupal_set_message('none in cache');.
      $output = $this->getPropertiesFromStore($class, $class_after, $fast_mode);
    }

    if ($this->allow_inverse_property_pattern) {
      // We get all the inverse properties by reverting class before and class
      // after and adding a "^".
      $output2 = $this->getPropertiesFromCache($class_after, $class);
      if ($output2 === FALSE) {
        // drupal_set_message('none in cache');.
        $output2 = $this->getPropertiesFromStore($class_after, $class, $fast_mode);
      }
      foreach ($output2 as $p) {
        $output["^$p"] = "^$p";
      }
    }

    uksort($output, 'strnatcasecmp');

    return $output;
  }

  /**
   * Returns an array of properties for which domain and/or range match the
   * input
   *
   * @param an associative array with keys 'domain' and/or 'range'
   *
   * @return array of matching properties | FALSE if there was no cache data
   */
  protected function getPropertiesFromCache($class, $class_after = NULL) {

    /* cache version
    $dom_properties = array();
    $cid = 'wisski_reasoner_reverse_domains';
    if ($cache = \Drupal::cache()->get($cid)) {
    $dom_properties = $cache->data[$class]?:array();
    } else return FALSE;
    $rng_properties = array();
    if (isset($class_after)) {
    $cid = 'wisski_reasoner_reverse_ranges';
    if ($cache = \Drupal::cache()->get($cid)) {
    $rng_properties = $cache->data[$class_after]?:array();
    } else return FALSE;
    } else return $dom_properties;
    return array_intersect_key($dom_properties,$rng_properties);
     */

    // DB version.
    $dom_properties = $this->retrieve('domains', 'property', 'class', $class);
    if (isset($class_after)) {
      $rng_properties = $this->retrieve('ranges', 'property', 'class', $class_after);
    }
    else {
      return $dom_properties;
    }
    return array_intersect_key($dom_properties, $rng_properties);
  }

  /**
   * Gets all graphs that are considered containing ontology information /
   * triples by this engine.
   *
   * This defaults to the graphs retrieved from the triple store but may also
   * be overridden by the user.
   *
   * @return an array of graph URIs
   */
  public function getOntologyGraphs() {
    $graph_uris = $this->ontology_graphs;
    if (empty($graph_uris) || (isset($graph_uris[0]) && empty($graph_uris[0]))) {
      $graph_uris = $this->queryOntologyGraphsFromStore();
    }
    return $graph_uris;
  }

  /**
   * Gets all graphs that contain an ontology.
   *
   * @return an array of graph URIs
   */
  public function queryOntologyGraphsFromStore() {
    $graph_uris = [];
    $query = "SELECT ?g WHERE { GRAPH ?g { ?ont a owl:Ontology} }";
    $result = $this->directQuery($query);
    foreach ($result as $obj) {
      $uri = $obj->g->getUri();
      $graph_uris[$uri] = $uri;
    }
    return $graph_uris;
  }

  /**
   *
   */
  public function getPropertiesFromStore($class = NULL, $class_after = NULL, $fast_mode = FALSE) {

    if ($fast_mode) {
      // The fast mode will only gather properties that are declared as
      // domain/range directly. This will only return an incomplete set of
      // properties unless we have resp. reasoning capabilities.
      $query = "SELECT DISTINCT ?property WHERE { \n";
      if (isset($class)) {
        $query .= "  GRAPH ?g3 { ?property rdfs:domain <$class>. }\n";
      }
      if (isset($class_after)) {
        $query .= "  GRAPH ?g4 { ?property rdfs:range <$class_after>. }\n";
      }
      $query .= "  { { GRAPH ?g1 { ?property a owl:ObjectProperty. } } UNION { GRAPH ?g2 { ?property a rdf:Property. } } }\n";
      $query .= "}";

    }
    else {
      // The complete mode makes more sophisticated queries that also return
      // properties that are declared domain/range indirectly, i.e. somewhere
      // appropriate within the class and property hierarchies.
      // This is way more inefficient.
      // We build up a default graph by using the FROM GRAPH clause.
      // Use all the graphs that contain ontology/tbox triples.
      // We have to use FROM GRAPH in complete mode as the
      // */+ modifiers will not work on multiple graphs if combined with a
      // GRAPH ?g {} statement. And we have to use GRAPH ?g {} for fuseki
      // support...
      // If there are no ontology graphs, we use the default graph.
      $ontology_graphs = $this->getOntologyGraphs();
      $from_graphs = " ";
      if (!empty($ontology_graphs)) {
        $from_graphs = "\nFROM <" . join(">\nFROM <", $ontology_graphs) . ">\n";
      }
      $query = "SELECT DISTINCT ?property{$from_graphs}WHERE {\n";

      if (isset($class)) {
        $query .=
          "  <$class> rdfs:subClassOf* ?d_def_class.\n"
          . "  {\n"
          . "    { ?d_def_prop rdfs:domain ?d_def_class. }\n"
          . "    UNION\n"
          . "    {\n"
          . "       ?d_def_prop owl:inverseOf ?inv.\n"
          . "       ?inv rdfs:range ?d_def_class.\n"
          . "    }\n"
          . "    UNION\n"
          . "    {\n"
          . "       ?d_def_prop rdfs:domain ?metaClass .\n"
          . "       ?metaClass owl:unionOf ?collection .\n"
          . "       ?collection rdf:rest*/rdf:first ?d_def_class . \n"
          . "    }\n"
          . "  }\n"
          . "  {\n"
          . "    { ?d_def_prop rdfs:subPropertyOf* ?property. }\n"
          . "    UNION\n"
          . "    {\n"
          . "      ?property rdfs:subPropertyOf+ ?d_def_prop.\n"
          . "      FILTER NOT EXISTS {\n"
          . "        ?mid_prop rdfs:subPropertyOf+ ?d_def_prop.\n"
          . "        ?property rdfs:subPropertyOf* ?mid_prop.\n"
          . "        {\n"
          . "          { ?mid_prop rdfs:domain ?any_domain. }\n"
          . "          UNION\n"
          . "          {\n"
          . "            ?mid_prop owl:inverseOf ?mid_inv.\n"
          . "            ?mid_inv rdfs:range ?any_range.\n"
          . "          }\n"
          . "        }\n"
          . "      }\n"
          . "    }\n"
          . "  }\n";
      }
      if (isset($class_after)) {
        $query .=
          "  <$class_after> rdfs:subClassOf* ?r_def_class.\n"
          . "  {\n"
          . "    { ?r_def_prop rdfs:range ?r_def_class. }\n"
          . "    UNION\n"
          . "    {\n"
          . "      ?r_def_prop owl:inverseOf ?inv.\n"
          . "      ?inv rdfs:domain ?inv.\n"
          . "    }\n"
          . "  }\n"
          . "  {\n"
          . "    { ?r_def_prop rdfs:subPropertyOf* ?property. }\n"
          . "    UNION\n"
          . "    {\n"
          . "      ?property rdfs:subPropertyOf+ ?r_def_prop.\n"
          . "      FILTER NOT EXISTS {\n"
          . "        ?mid_prop rdfs:subPropertyOf+ ?r_def_prop.\n"
          . "        ?property rdfs:subPropertyOf* ?mid_prop.\n"
          . "        {\n"
          . "          { ?mid_prop rdfs:range ?any_range. }\n"
          . "          UNION\n"
          . "          {\n"
          . "            ?mid_prop owl:inverseOf ?mid_inv.\n"
          . "            ?mid_inv rdfs:domain ?any_domain.\n"
          . "          }\n"
          . "        }\n"
          . "      }\n"
          . "    }\n"
          . "  }\n";
      }
      $query .= "  { { ?property a owl:ObjectProperty. } UNION { ?property a rdf:Property. } }\n";
      $query .= "}";
      /*      if (isset($class)) {
      $query .=
      "  GRAPH ?g8 { <$class> rdfs:subClassOf* ?d_def_class. }\n"
      ."  {\n"
      ."    { GRAPH ?g5 { ?d_def_prop rdfs:domain ?d_def_class.}}\n"
      ."    UNION\n"
      ."    {\n"
      ."       GRAPH ?g6 { ?d_def_prop owl:inverseOf ?inv. }\n"
      ."       GRAPH ?g7 { ?inv rdfs:range ?d_def_class. }\n"
      ."    }\n"
      ."  }\n"
      ."  {\n"
      ."    { GRAPH ?g9 { ?d_def_prop rdfs:subPropertyOf* ?property.}}\n"
      ."    UNION\n"
      ."    {\n"
      ."      GRAPH ?g10 { ?property rdfs:subPropertyOf+ ?d_def_prop. }\n"
      ."      FILTER NOT EXISTS {\n"
      ."        {\n"
      ."          GRAPH ?g11 { ?mid_prop rdfs:subPropertyOf+ ?d_def_prop. }\n"
      ."          GRAPH ?g12 { ?property rdfs:subPropertyOf* ?mid_prop. }\n"
      ."        }\n"
      ."        {\n"
      ."          { GRAPH ?g13 { ?mid_prop rdfs:domain ?any_domain.}}\n"
      ."          UNION\n"
      ."          {\n"
      ."            GRAPH ?g14 { ?mid_prop owl:inverseOf ?mid_inv. }\n"
      ."            GRAPH ?g15 { ?mid_inv rdfs:range ?any_range. }\n"
      ."          }\n"
      ."        }\n"
      ."      }\n"
      ."    }\n"
      ."  }\n";
      }
      if (isset($class_after)) {
      $query .=
      "  GRAPH ?g19 { <$class_after> rdfs:subClassOf* ?r_def_class.}\n"
      ."  {\n"
      ."    { GRAPH ?g16 { ?r_def_prop rdfs:range ?r_def_class.} }\n"
      ."    UNION\n"
      ."    {\n"
      ."      GRAPH ?g17 { ?r_def_prop owl:inverseOf ?inv. }\n"
      ."      GRAPH ?g18 { ?inv rdfs:domain ?inv. }\n"
      ."    }\n"
      ."  }\n"
      ."  {\n"
      ."    { GRAPH ?g20 { ?r_def_prop rdfs:subPropertyOf* ?property.} }\n"
      ."    UNION\n"
      ."    {\n"
      ."      GRAPH ?g21 { ?property rdfs:subPropertyOf+ ?r_def_prop. }\n"
      ."      FILTER NOT EXISTS {\n"
      ."        {\n"
      ."          GRAPH ?g22 { ?mid_prop rdfs:subPropertyOf+ ?r_def_prop. }\n"
      ."          GRAPH ?g23 { ?property rdfs:subPropertyOf* ?mid_prop. }\n"
      ."        }\n"
      ."        {\n"
      ."          { GRAPH ?g24 { ?mid_prop rdfs:range ?any_range. } }\n"
      ."          UNION\n"
      ."          {\n"
      ."            GRAPH ?g25 { ?mid_prop owl:inverseOf ?mid_inv. }\n"
      ."            GRAPH ?g26 { ?mid_inv rdfs:domain ?any_domain. }\n"
      ."          }\n"
      ."        }\n"
      ."      }\n"
      ."    }\n"
      ."  }\n";
      $query .= "  { { GRAPH ?g1 { ?property a owl:ObjectProperty. } } UNION { GRAPH ?g2 { ?property a rdf:Property. } } }\n";
      $query .= "}";
      }  */
    }

#    dpm($query, "query?");
    $result = $this->directQuery($query);
    $output = [];
    foreach ($result as $obj) {
      $prop = $obj->property->getUri();
      $output[$prop] = $prop;
    }

    return $output;
  }

  /**
   *
   */
  public function nextClasses($property = NULL, $property_after = NULL, $fast_mode = FALSE) {
    if (!isset($property) && !isset($property_after)) {
      return $this->getClasses();
    }
    elseif (isset($property) && $property[0] == '^') {
      if (isset($property_after)) {
        if ($property_after[0] == '^') {
          return $this->nextClasses(substr($property_after, 1), substr($property, 1), $fast_mode);
        }
        else {
          $classes1 = $this->nextClasses(NULL, $property_after, $fast_mode);
          $classes2 = $this->nextClasses(NULL, substr($property, 1), $fast_mode);
          return array_intersect($classes1, $classes2);
        }
      }
      // $property_after == NULL
      else {
        return $this->nextClasses(NULL, substr($property, 1), $fast_mode);
      }
    }
    elseif (isset($property_after) && $property_after[0] == '^') {
      // remember: $property[0] != '^' (otherwise we would be in branch above!)
      if (isset($property)) {
        $classes1 = $this->nextClasses($property, NULL, $fast_mode);
        $classes2 = $this->nextClasses(substr($property_after, 1), NULL, $fast_mode);
        return array_intersect($classes1, $classes2);
      }
      else {
        return $this->nextClasses(substr($property_after, 1), NULL, $fast_mode);
      }
    }
    else {
      // \Drupal::logger(__METHOD__)->debug('property: '.$property.', property_after: '.$property_after);
      $output = $this->getClassesFromCache($property, $property_after);
      // dpm($output, "output");.
      if ($output === FALSE) {
        // drupal_set_message('none in cache');.
        $output = $this->getClassesFromStore($property, $property_after, $fast_mode);
      }
      uksort($output, 'strnatcasecmp');
      return $output;
    }
  }

  /**
   *
   */
  protected function getClassesFromCache($property, $property_after = NULL) {

    /* cache version
    $dom_classes = array();
    $cid = 'wisski_reasoner_ranges';
    if ($cache = \Drupal::cache()->get($cid)) {
    $rng_classes = $cache->data[$property]?:array();
    } else return FALSE;
    $dom_classes = array();
    if (isset($property_after)) {
    $cid = 'wisski_reasoner_domains';
    if ($cache = \Drupal::cache()->get($cid)) {
    $dom_classes = $cache->data[$property_after]?:array();
    } else return FALSE;
    } else return $rng_classes;
    return array_intersect_key($rng_classes,$dom_classes);
     */
    // dpm(func_get_args()+array('result'=>$return),__FUNCTION__);
    // DB version.
    $rng_classes = $this->retrieve('ranges', 'class', 'property', $property);
    // dpm($rng_classes, "ranges");.
    if (isset($property_after)) {
      $dom_classes = $this->retrieve('domains', 'class', 'property', $property_after);
    }
    else {
      return $rng_classes;
    }
    return array_intersect_key($rng_classes, $dom_classes);
  }

  /**
   *
   */
  public function getClassesFromStore($property = NULL, $property_after = NULL, $fast_mode = FALSE) {

    $query = "SELECT DISTINCT ?class WHERE {  {"
      . "{ {?class a owl:Class. } UNION { ?class a rdfs:Class.} } ";
    if ($fast_mode) {
      if (isset($property)) {
        $query .= "<$property> rdfs:range ?class. ";
      }
      if (isset($property_after)) {
        $query .= "<$property_after> rdfs:domain ?class. ";
      }
    }
    else {
      if (isset($property)) {
        $query .= "<$property> rdfs:subPropertyOf* ?r_super_prop. "
          . "{ { ?r_super_prop rdfs:range ?r_super_class. } UNION "
          . "{ ?r_super_prop owl:inverseOf ?inv. ?inv rdfs:domain ?r_super_class. } UNION "
          . "{ ?r_super_prop rdfs:range ?metaClass . "
          . " ?metaClass owl:unionOf ?collection . "
          . " ?collection rdf:rest*/rdf:first ?r_super_class . } "
          . "} "
          . "FILTER NOT EXISTS { "
          . "?r_sub_prop rdfs:subPropertyOf+ ?r_super_prop. "
          . "<$property> rdfs:subPropertyOf* ?r_sub_prop. "
          . "?r_sub_prop rdfs:range ?r_any_class. "
          . "} "
          . "?class rdfs:subClassOf* ?r_super_class. ";
      }
      if (isset($property_after)) {
        $query .= "<$property_after> rdfs:subPropertyOf* ?d_super_prop. "
          . "{ { ?d_super_prop rdfs:domain ?d_super_class. } UNION "
          . "{ ?d_super_prop owl:inverseOf ?inv. ?inv rdfs:domain ?d_super_class. } UNION "
          . "{ ?d_super_prop rdfs:range ?metaClass . "
          . " ?metaClass owl:unionOf ?collection . "
          . " ?collection rdf:rest*/rdf:first ?d_super_class . } "
          . "} "
          . "FILTER NOT EXISTS { "
          . "?d_sub_prop rdfs:subPropertyOf+ ?d_super_prop. "
          . "<$property_after> rdfs:subPropertyOf* ?d_sub_prop. "
          . "?d_sub_prop rdfs:domain ?d_any_class. "
          . "} "
          . "?class rdfs:subClassOf* ?d_super_class. ";
      }
    }
    $query .= "} }";

    // drupal_set_message(serialize($query));
#    dpm($query, "query?");
    $result = $this->directQuery($query);

    if (count($result) == 0) {
      return [];
    }

    $output = [];
    foreach ($result as $obj) {
      $class = $obj->class->getUri();
      $output[$class] = $class;
    }
    natsort($output);
    return $output;

  }

  /*******************
   * End of BASIC Pathbuilder Support .***********************/

  /**
   * Copy from yaml-adapter - likes camels.
   */
  private $entity_info;

  /**
   * Load the image data for a given entity id.
   *
   * @return an array of values?
   */
  public function getImagesForEntityId($entityid, $bundleid, $do_sorting_by_weight = FALSE) {
    $pbs = $this->getPbsForThis();

    $entityid = $this->getDrupalId($entityid);

    $ret = [];

    foreach ($pbs as $pb) {
      // drupal_set_message("yay!" . $entityid . " and " . $bundleid);.
      $groups = $pb->getGroupsForBundle($bundleid);

      foreach ($groups as $group) {
        $paths = $pb->getImagePathIDsForGroup($group->id());

        // drupal_set_message("paths: " . serialize($paths));
        foreach ($paths as $pathid) {

          $path = WisskiPathEntity::load($pathid);

          // drupal_set_message(serialize($path));
          // drupal_set_message("thing: " . serialize($this->pathToReturnValue($path->getPathArray(), $path->getDatatypeProperty(), $entityid, 0, NULL, 0)));.
          // This has to be an absulte path - otherwise subgroup images won't load.
          $new_ret = $this->pathToReturnValue($path, $pb, $entityid, 0, NULL, FALSE);
          // If (!empty($new_ret)) dpm($pb->id().' '.$pathid.' '.$entitid,'News');.

                    if($do_sorting_by_weight == TRUE) {

            $allpbpaths = $pb->getPbPaths();
            $pbarray = $allpbpaths[$pathid];

#            dpm($pbarray);

            $fid_to_look_for = $pbarray['field'];

            $cached_field_values = \Drupal::database()->select('wisski_entity_field_properties', 'f')
              ->fields('f',array('ident','delta'));

            if(!empty($fid_to_look_for)) {
              $cached_field_values = $cached_field_values->condition('fid', $fid_to_look_for);
            }

//            $cached_field_values = $cached_field_values->condition('ident', $to_look_for)
            $cached_field_values = $cached_field_values->condition('eid', $entityid)
              ->execute()
              ->fetchAllAssoc('ident');

#            dpm($cached_field_values);
#            dpm($new_ret);

            if(empty($cached_field_values)) {
              $ret = array_merge($ret, $new_ret);
            } else {
              $ordered_ret = array();

              $not_ordered = array();

              foreach($new_ret as $one_ret) {
                if(isset($cached_field_values[$one_ret]) && isset($cached_field_values[$one_ret]->delta) ) {
                  $ordered_ret[$cached_field_values[$one_ret]->delta] = $one_ret;
                } else {
                  $not_ordered[] = $one_ret;
                }
              }

              // if we found things that don't have a delta we just append
              // them in the order we got them.
              if(!empty($not_ordered)) {
                foreach($not_ordered as $one_thing) {
                  $ordered_ret[] = $one_thing;
                }
              }

              // sort by keys
              ksort($ordered_ret);

#              dpm($ordered_ret);

              $ret = array_merge($ret, $ordered_ret);
            }

          } else {
            // easy case, no sorting
#          if (!empty($new_ret)) dpm($pb->id().' '.$pathid.' '.$entitid,'News');
            $ret = array_merge($ret, $new_ret);
          }


        }
      }
    }
    // drupal_set_message("returning: " . serialize($ret));
    // dpm($ret,__FUNCTION__);
    return $ret;
  }

  /**
   * @see getBundleIdsForUri()
   */
  public function getBundleIdsForEntityId($entityid) {

    if (is_numeric($entityid)) {
      $uri = $this->getUriForDrupalId($entityid, FALSE);
    }
    else {
      $uri = $entityid;
    }
    $url = parse_url($uri);

    if (!empty($url["scheme"])) {
      return $this->getBundleIdsForUri($uri);
    }
    else {
      // It is possible, that we got an entity URI instead of an entity ID here, so try that one first.
      $url = parse_url($entityid);
      if (!empty($url['scheme'])) {
        $uri = $entityid;
        return $this->getBundleIdsForUri($uri);
      }
    }

    // wsmlog(__METHOD__ . ": could not find URI for entityid '$entityid'", 'warning');.
    $this->messenger()->addError("Could not find URI for entityid '$entityid'");
    return [];

  }

  /**
   * Get the IDs of all bundles that can be used for the given instance.
   *
   * It returns only bundles defined via pathbuilders that are associated with
   * this engine/adapter.
   * The URI must be without prefix but with namespace. Prefixed or abbreviated
   * URIs will not be handled correctly! If you have an entity id or a
   * non-standardized URI you may want to use getBundleIdsForEntityId().
   *
   * @param uri a URI of the instance
   *
   * @return an array of bundle ids
   */
  public function getBundleIdsForUri($uri) {
    $pbs = $this->getPbsForThis();

    $query = "SELECT ?class WHERE { GRAPH ?g { <" . $uri . "> a ?class } }";

    $result = $this->directQuery($query);

    $out = [];
    foreach ($result as $thing) {
      $uri_to_find = $thing->class->getUri();

      $topbundles = [];
      $nontopbundles = [];

      foreach ($pbs as $pb) {
        $bundles = $pb->getAllBundleIdsAboutUri($uri_to_find);

        if (empty($bundles)) {
          continue;
        }

        [$tmptopbundles, $tmpnontopbundles] = $bundles;

        $topbundles = array_merge($topbundles, $tmptopbundles);
        $nontopbundles = array_merge($nontopbundles, $tmpnontopbundles);
      }
      // dpm($topbundles);
      foreach ($nontopbundles as $key => $value) {
        $out = array_merge([$key => $value], $out);
      }

      foreach ($topbundles as $key => $value) {
        $out = array_merge([$key => $value], $out);
      }

    }

    // dpm($out, "out");.
    return $out;

  }

  /**
   * Gets the array part to get from one subgroup to another.
   */
  public function getClearGroupArray($group, $pb) {
    // We have to modify the group-array in case of jumps
    // from one subgroup to another
    // if you have a groups with grouppaths:
    // g1: x0
    // g2: x0 y0 x1
    // g3: x0 y0 x1 y1 x2 y2 x3
    // then the way from g2 to g3 is x1 y1 x2 y2 x3
    // this should be calculated here.
    $patharraytoget = $group->getPathArray();
    $allpbpaths = $pb->getPbPaths();
    $pbarray = $allpbpaths[$group->id()];

    // Do some error handling.
    if (!$group->isGroup()) {
      $this->messenger()
        ->addError("getClearGroupArray called with something that is not a group: " . serialize($group));
      return;
    }

    // If we are a top group, won't do anything.
    if ($pbarray['parent'] > 0) {

      // First we have to calculate our own ClearPathArray.
      $clearGroupArray = $this->getClearPathArray($group, $pb);

      // Then we have to get our parents array.
      $pbparentarray = $allpbpaths[$pbarray['parent']];

      $parentpath = WisskiPathEntity::load($pbarray["parent"]);

      // If there is nothing, do nothing!
      // I am unsure if that ever could occur.
      if (empty($parentpath)) {
        return;
      }

      // -1 because we don't want to cut our own concept
      $parentcnt = count($parentpath->getPathArray()) - 1;

      // drupal_set_message("before cut: " . serialize($patharraytoget));
      for ($i = 0; $i < $parentcnt; $i++) {
        unset($patharraytoget[$i]);
      }

      // drupal_set_message("in between: " . serialize($patharraytoget));
      $patharraytoget = array_values($patharraytoget);

      // drupal_set_message("cga: " . serialize($clearGroupArray));
      $max = count($patharraytoget);

      // We have to cut away everything that is in $cleargrouparray
      // so we take the whole length and subtract that as a starting point
      // and go up from there.
      for ($i = (count($patharraytoget) - count($clearGroupArray) + 1); $i < $max; $i++) {
        unset($patharraytoget[$i]);
      }

      // drupal_set_message("after cut: " . serialize($patharraytoget));
      $patharraytoget = array_values($patharraytoget);

    }
    return $patharraytoget;
  }

  /**
   * This is Obsolete!
   *
   * Gets the common part of a group or path
   * that is clean from subgroup-fragments.
   */
  public function getClearPathArray($path, $pb) {
    // We have to modify the path-array in case of subgroups.
    // Usually if we have a subgroup path x0 y0 x1 we have to skip x0 y0 in
    // the paths of the group.
    if (!is_object($path) || !is_object($pb)) {
      $this->messenger()
        ->addError('getClearPathArray found no path or no pathbuilder. Error!');
      return [];
    }

    $patharraytoget = $path->getPathArray();
    $allpbpaths = $pb->getPbPaths();
    $pbarray = $allpbpaths[$path->id()];

    // dpm($pbarray, "pbarray!");
    // Is it in a group?
    if (!empty($pbarray['parent'])) {

      $pbparentarray = $allpbpaths[$pbarray['parent']];

      // How many path-parts are in the pb-parent?
      $parentpath = WisskiPathEntity::load($pbarray["parent"]);

      // If there is nothing, do nothing!
      // I am unsure if that ever could occur.
      if (empty($parentpath)) {
        return;
      }

      // We have to handle groups other than paths.
      if ($path->isGroup()) {
        // So this is a subgroup?
        // in this case we have to strip the path of the parent and
        // one object property from our path.
        $pathcnt = count($parentpath->getPathArray()) + 1;

        // Strip exactly that.
        for ($i = 0; $i < $pathcnt; $i++) {
          unset($patharraytoget[$i]);
        }

      }
      else {
        // This is no subgroup, it is a path
        // if(!empty($pbparentarray['parent'])) {
        // Only do something if it is a path in a subgroup, not in a main group
        // $parentparentpath = \Drupal\wisski_pathbuilder\Entity\WisskiPathEntity::load($pbparentarray["parent"]);.
        // In that case we have to remove the subgroup-part, however minus one, as it is the
        // $pathcnt = count($parentpath->getPathArray()) - count($this->getClearPathArray($parentpath, $pb));.
        // count($parentparentpath->getPathArray());
        $pathcnt = count($parentpath->getPathArray()) - 1;

        // dpm($pathcnt, "pathcnt");
        // dpm($parentpath->getPathArray(), "pa!");.
        for ($i = 0; $i < $pathcnt; $i++) {
          unset($patharraytoget[$i]);
        }
        // }
      }
    }

    // drupal_set_message("parent is: " . serialize($pbparentarray));
    // drupal_set_message("I am getting: " . serialize($patharraytoget));
    $patharraytoget = array_values($patharraytoget);

    return $patharraytoget;
  }

  /**
   * Gets the bundle and loads every individual in the TS
   * and returns an array of ids if there is something...
   */
  public function loadIndividualsForBundle($bundleid, $pathbuilder, $limit = NULL, $offset = NULL, $count = FALSE, $conditions = FALSE) {

    $conds = [];
    // See if we have any conditions.
    foreach ($conditions as $cond) {
      if ($cond["field"] != "bundle") {
        // Get pb entries.
        $pbentries = $pathbuilder->getPbEntriesForFid($cond["field"]);

        if (empty($pbentries)) {
          continue;
        }

        $path = WisskiPathEntity::load($pbentries['id']);

        if (empty($path)) {
          continue;
        }

        $conds[] = $path;
      }
    }

    // Build the query.
    if (!empty($count)) {
      $query = "SELECT (COUNT(?x0) as ?cnt) WHERE {";
    }
    else {
      $query = "SELECT ?x0 WHERE {";
    }

    if (empty($conds)) {

      // There should be someone asking for more than one...
      $groups = $pathbuilder->getGroupsForBundle($bundleid);

      // No group defined in this pb - return.
      if (empty($groups)) {
        if ($count) {
          return 0;
        }
        return [];
      }

      // For now simply take the first one
      // in future: iterate here!
      // @todo !
      $group = $groups[0];

      // Get the group
      // this does not work for subgroups! do it otherwise!
      // $grouppath = $group->getPathArray();
      // $grouppath = $this->getClearPathArray($group, $pathbuilder);.
      $grouppath = $pathbuilder->getRelativePath($group, FALSE);

      foreach ($grouppath as $key => $pathpart) {
        if ($key % 2 == 0) {
          $query .= " ?x" . $key . " a <" . $pathpart . "> . ";
        }
        else {
          $query .= " ?x" . ($key - 1) . " <" . $pathpart . "> ?x" . ($key + 1) . " . ";
        }
      }
    }
    else {
      foreach ($conds as $path) {
        $query .= $this->generateTriplesForPath($pathbuilder, $path, '', NULL, NULL, 0, 0, FALSE);
      }
    }

    $query .= "}";

    if (is_null($limit) == FALSE && is_null($offset) == FALSE && empty($count)) {
      $query .= " LIMIT $limit OFFSET $offset ";
    }

    // drupal_set_message("query: " . serialize($query) . " and " . microtime());
    // return;
    // dpm($query,__FUNCTION__.' '.$this->adapterId());
    // ask for the query.
    $result = $this->directQuery($query);

    $outarr = [];

    // For now simply take the first element
    // later on we need names here!
    foreach ($result as $thing) {

      // If it is a count query, return the integer.
      if (!empty($count)) {
        // dpm($thing,'Count Thing');.
        return $thing->cnt->getValue();
      }

      $uri = $thing->x0->dumpValue("text");

      // $uri = str_replace('/','\\',$uri);
      // This is no uri anymore - rename this variable.
      $uriname = $this->getDrupalId($uri);

      // Store the bundleid to the bundle-cache as it might be important
      // for subsequent queries.
      $pathbuilder->setBundleIdForEntityId($uriname, $bundleid);

      $outarr[$uriname] = [
        'eid' => $uriname,
        'bundle' => $bundleid,
        'name' => $uri,
      ];
    }
    // dpm($outarr, "outarr");
    // return;.
    if (empty($outarr) && $count) {
      return 0;
    }
    return $outarr;
  }

  /**
   *
   */
  public function loadEntity($id) {
    // Simply give back something without thinking about it.
    $out['eid'] = $id;

    return $out;
  }

  /**
   * This is deprecated and unfunctional.
   */
  public function loadMultipleEntities($ids = NULL) {
    $this->messenger()->addError("I may not be called: loadMultipleEntities. ");
    return;
  }

  /**
   * @inheritdoc
   */
  public function hasEntity($entity_id) {

    // dpm("has Entity?");.
    // dpm($this->getBaseFieldFromStoreForUri($uri, $entity_id));.
    $uri = $this->getUriForDrupalId($entity_id, FALSE);

    $out = NULL;

    if ($uri) {
      $out = $this->checkUriExists($uri);
    }

    return $out;
  }

  /**
   * The elemental data aggregation function
   * fetches the data for display purpose
   */
  public function pathToReturnValue($path, $pb, $eid = NULL, $position = 0, $main_property = NULL, $relative = TRUE, $language = LanguageInterface::LANGCODE_DEFAULT) {
    // drupal_set_message("I got: $eid " . serialize($path));
    // $tmpt1 = microtime(TRUE);
    // drupal_set_message("ptrv: " . microtime());
    // MyF: Instead of the LANGCODE_DEFAULT (x-default), we use the displayed language.
    $language = \Drupal::service('language_manager')
      ->getCurrentLanguage()
      ->getId();

    $languages = [];

    if ($language == "all") {
      $to_load = \Drupal::languageManager()->getLanguages();
      $languages = array_keys($to_load);

      // dpm($languages, "langs?");.
    }
    else {
      $languages = [$language];
    }

    if (empty($path)) {
      $this->messenger()->addError("No path supplied to ptr. This is evil.");
      return [];
    }

    if (!$path->isGroup()) {
      $primitive = $path->getDatatypeProperty();
    }
    else {
      $primitive = NULL;
    }

    $disamb = $path->getDisamb();

    // Also.
    if ($disamb > 0) {
      $disamb = ($disamb - 1) * 2;
    }
    else {
      $disamb = NULL;
    }

    $sparql = "SELECT DISTINCT ";

    if ($relative) {
      $starting_position = count($path->getPathArray()) - count($pb->getRelativePath($path));
    }
    else {
      $starting_position = $position;
    }

    // In case of disamb we contradict the theory below.
    // && $disamb === $starting_position) {.
    if (!is_null($disamb)) {
      $sparql .= "?x" . $disamb . " ";
    }

    // $starting_position+2 because we can omit x0 in this place - it will always be replaced
    // by the eid of this thing here.
    /*
    // We try to be more precise than this approach as it is rather costly...
    for($i = $starting_position+2; $i <= count($path->getPathArray()); $i+=2) {
    $sparql .= "?x" . $i . " ";
    }
     */

    // Get the queried one.
    $name = 'x' . (count($path->getPathArray()) - 1);

    // In case of primitives it is not the above one but "out".
    if (!empty($primitive) && $primitive != "empty") {
      $sparql .= "?out ";
    }
    else {
      // But in all other cases it is the above one.
      if ($name != "x" . $disamb) {
        $sparql .= "?" . $name . " ";
      }
    }

    $sparql .= "WHERE { ";

    // drupal_set_message("ptrv2: " . microtime());
    if (!empty($eid)) {
      // Rename to uri
      // $tmpt5 = microtime(TRUE);
      $eid = $this->getUriForDrupalId($eid, TRUE);
      // $tmpt6 = microtime(TRUE);
      // If the path is a group it has to be a subgroup and thus entity reference.
      if ($path->isGroup()) {

        // It is the same as field - so entity_reference is basic shit here
        //        $sparql .= $this->generateTriplesForPath($pb, $path, '', $eid, NULL, 0,  ($starting_position/2), FALSE, NULL, 'entity_reference', $relative);.
        // Here we can probably avoid the language tag...
        // by mark ;)
        $sparql .= $this->generateTriplesForPath($pb, $path, '', $eid, NULL, 0, ($starting_position / 2), FALSE, NULL, 'entity_reference', $relative, [], 0);

      }
      else {
        // $sparql .= $this->generateTriplesForPath($pb, $path, '', $eid, NULL, 0, ($starting_position/2), FALSE, NULL, 'field', $relative);
        $sparql .= $this->generateTriplesForPath($pb, $path, '', $eid, NULL, 0, ($starting_position / 2), FALSE, NULL, 'field', $relative, [], 0);
      }
      // $tmpt7 = microtime(TRUE);
    }
    else {
      $this->messenger()->addError("No EID for data. Error. ");
    }

    // drupal_set_message("ptrv3 " . microtime());
    // If disamb should be in the query we can bind it.
    if ($starting_position === $disamb) {
      // @todo /WARNING by Martin: I think this does not work! It should be VALUES {} instead!
      // afaik BIND only works for unbound variables.
      // I do not fix it now as i don't get the idea behind why this is done
      // here! When can starting and disamb position be the same and why and
      // what is expected to happen?
      $sparql .= " BIND( <" . $eid . "> AS ?x" . $disamb . ") . ";
    }

    $sparql .= " } ";

    // dpm($sparql, "spar-gel");
    // drupal_set_message(serialize($sparql) . " on " . serialize($this));
    // dpm(microtime(), "mic1");
    // $tmpt2 = microtime(TRUE);
    $result = $this->directQuery($sparql);
    // $tmpt3 = microtime(TRUE);
    // dpm(microtime(), "mic2");
    // drupal_set_message(serialize($result));
    // dpm($result, "res?");.
    $out = [];
    foreach ($result as $thing) {

      // If $thing is just a true or not true statement.
      // @todo by Martin: is this really working and if so in which case?
      // Also, what does true/false statement mean? we know that it's not an ASK query.
      if ($thing == new \StdClass()) {
        $this->messenger()->addError("This is ultra-evil!");
        continue;
      }

      // $name = 'x' . (count($patharray)-1);
      // $name = 'x' . (count($path->getPathArray())-1);
      if (!empty($primitive) && $primitive != "empty") {
        if (empty($main_property)) {

          // dpm("I have no main_property");.
          $my_language = $thing->out->getLang();

          // If we have no language, take the default one.
          //          if(empty($my_language))
          //            $my_language = LanguageInterface::LANGCODE_DEFAULT;.
          // Only add it if my language is the language provided to me
          //          if($my_language == $language)
          //            $out[] = $thing->out->getValue();
          // This probably breaks any case were we assumed we just get a value
          // I hope this are not too many ;D.
          // dpm("Danger Zone!!",.
          $out[] = $thing->out->getValue();
        }
        else {

          $my_language = $thing->out->getLang();

          // If we have no language, take the default one.
          if (empty($my_language)) {
            // $my_language = LanguageInterface::LANGCODE_DEFAULT;
            // MyF: if there is no language, we set the language to the current displayed language stored in $language (instead of setting it to LanguageInterface::LANGCODE_DEFAULT)
            $my_language = $language;
          }

          $outvalue = $thing->out->getValue();

          // Special case: DateTime... render this as normal value for now.
          if (is_a($outvalue, "DateTime")) {
            $outvalue = (string) $outvalue->format('Y-m-d\TH:i:s.u');
            ;
          }

          // if($path->isGroup() && $main_property == "target_id")
          // $outvalue = $this->getDrupalId($outvalue);
          // dpm($outvalue . " and " . serialize($disamb));
          if (is_null($disamb) == TRUE) {
            $out[] = [
              $main_property => $outvalue,
              "wisski_language" => $my_language,
            ];
          }
          else {
            $disambname = 'x' . $disamb;
            if (!isset($thing->{$disambname})) {
              $out[] = [
                $main_property => $outvalue,
                "wisski_language" => $my_language,
              ];
            }
            else {
              $out[] = [
                $main_property => $outvalue,
                'wisskiDisamb' => $thing->{$disambname}->dumpValue("text"),
                "wisski_language" => $my_language,
              ];
            }
          }
        }
      }
      else {

        // In this case we always take the language which is available...
        // $my_language = LanguageInterface::LANGCODE_DEFAULT;
        // MyF: in this case, we take the language which is displayed (stored in $language)
        // $my_language = LanguageInterface::LANGCODE_DEFAULT;.
        $my_language = $language;
        // dpm("(Sparl11EngineWithPB.php, line 1380) I changed my lang from x-default to " . $my_language);.
        if (empty($main_property)) {
          $out[] = $thing->{$name}->dumpValue("text");
        }
        else {
          $outvalue = $thing->{$name}->dumpValue("text");

          // if($path->isGroup() && $main_property == "target_id")
          // $outvalue = $this->getDrupalId($outvalue);
          if (is_null($disamb) == TRUE) {
            $out[] = [$main_property => $outvalue];
          }
          else {
            $disambname = 'x' . $disamb;
            if (!isset($thing->{$disambname})) {
              $out[] = [$main_property => $outvalue];
            }
            else {
              $out[] = [
                $main_property => $outvalue,
                'wisskiDisamb' => $thing->{$disambname}->dumpValue("text"),
              ];
            }
          }
        }
      }
    }
    // $tmpt4 = microtime(TRUE);
    // \Drupal::logger('WissKI Adapter ptrv')->debug($pb->id() . " " . $path->id() ."::". htmlentities(\Drupal\Core\Serialization\Yaml::encode( [$tmpt4-$tmpt1,$tmpt7-$tmpt6, $tmpt5-$tmpt1, $tmpt6-$tmpt5, $tmpt2-$tmpt7, $tmpt3-$tmpt2, $tmpt4-$tmpt3])));
    // dpm([$tmpt4-$tmpt1,$tmpt5-$tmpt1, $tmpt6-$tmpt5, $tmpt2-$tmpt6, $tmpt3-$tmpt2, $tmpt4-$tmpt3], $pb->id() . " " . $path->id());
    // dpm($out, "out?");.
    // By mark:
    // up to now out had the structure array( key => array( "value" => "some value that i've got from the Ts") )
    // in future we want that language dependent. This means we want:
    // array( "lang" = > array( key => array( "value" => "some value that i've got from the Ts") ) )
    return $out;

  }

  /**
   * @inheritdoc
   */
  public function loadFieldValues(array $entity_ids = NULL, array $field_ids = NULL, $bundleid_in = NULL, $language = LanguageInterface::LANGCODE_DEFAULT) {

    // Tricky thing here is that the entity_ids that are coming in typically
    // are somewhere from a store. In case of rdf it is easy - they are uris.
    // In case of csv or something it is more tricky. So I don't wan't to
    // simply go to the store and tell it "give me the bundle of this".
    // The field ids come in handy here - fields are typically attached
    // to a bundle anyway. so I just get the bundle from there. I think it is
    // rather stupid that this function does not load the field values per
    // bundle - it is implicitely anyway like that.
    //
    // so I ignore everything and just target the field_ids that are mapped to
    // paths in the pathbuilder.
    // drupal_set_message("I am asked for " . serialize($entity_ids) . " and fields: " . serialize($field_ids));
    $tsa = ['start' => microtime(TRUE)];
    $pb_ids = [];
    $pb_man = \Drupal::service('wisski_pathbuilder.manager');
    $bundle_infos = $pb_man->getPbsUsingBundle($bundleid_in);
    // dpm([$this->adapterId(), $bundle_infos]);.
    foreach ($bundle_infos as $bundle_info) {
      if ($bundle_info['adapter_id'] == $this->adapterId()) {
        $pb_id = $bundle_info['pb_id'];
        $pb_ids[$pb_id] = $pb_id;
      }
    }
    $pbs = WisskiPathbuilderEntity::loadMultiple($pb_ids);
    $tsa['pbs'] = join(' ', $pb_ids);

    $out = [];

    // Get the adapterid that was loaded
    // haha, this is the engine-id...
    // $adapterid = $this->getConfiguration()['id'];
    foreach ($pbs as $pb) {

      // If we find any data, we set this to true.
      $found_any_data = FALSE;

      foreach ($field_ids as $fkey => $fieldid) {
        // drupal_set_message("for field " . $fieldid . " with bundle " . $bundleid_in . " I've got " . serialize($this->loadPropertyValuesForField($fieldid, array(), $entity_ids, $bundleid_in, $language)));.
        $tmpc = microtime(TRUE);
        $got = $this->loadPropertyValuesForField($fieldid, [], $entity_ids, $bundleid_in);
        $tsa[$pb->id()][$fieldid] = (microtime(TRUE) - $tmpc);

        // drupal_set_message("I've got: " . serialize($got));
        if (empty($out)) {
          $out = $got;
        }
        else {
          foreach ($got as $eid => $value) {
            if (empty($out[$eid])) {
              $out[$eid] = $got[$eid];
            }
            else {
              $out[$eid] = array_merge($out[$eid], $got[$eid]);
            }
          }
        }

        // drupal_set_message("out after got: " . serialize($out));
      }

      // drupal_set_message("out is empty? " . serialize($out) . serialize(empty($out)));
      // @todo this is a hack.
      // if we did not find any data we unset this part so we don't return anything
      // however this might be evil in cases of edit or something...
      if (empty($out)) {
        return [];
      }
    }

    // drupal_set_message("I return: " . serialize($out));
    $tsa['ende'] = microtime(TRUE) - $tsa['start'];
    // \Drupal::logger('Sparql Adapter')->debug(str_replace("\n", '<br/>', htmlentities("loadFieldValues:\n".\Drupal\Core\Serialization\Yaml::encode($tsa))));
    return $out;

  }

  /**
   * @inheritdoc
   * The Yaml-Adapter cannot handle field properties, we insist on field values
   *   being the main property
   */
  public function loadPropertyValuesForField($field_id, array $property_ids, array $entity_ids = NULL, $bundleid_in = NULL) {

    // By MyF: We get all available languages here because we need them in the entity reference case in order to support multilanguage cases.
    $available_languages = \Drupal::languageManager()->getLanguages();
    $available_languages = array_keys($available_languages);
    // if(isset($language)){
    // dpm("High Voltage: " . $language . " is set in loadPropertyValuesForField() in Sparql11EngineWithPB.php and should not be set.");
    // }.
    // drupal_set_message("a1: " . microtime());
    // drupal_set_message("fun: " . serialize(func_get_args()));
    // drupal_set_message("2");
    //
    // drupal_set_message("muha: " . serialize($field_id));
    // ->getItemDefinition()->mainPropertyName();
    $field_storage_config = FieldStorageConfig::loadByName('wisski_individual', $field_id);
    // drupal_set_message("a11: " . microtime());
    // What does it mean if the field storage config is empty?
    // =>  it means it is a basic field.
    $out = [];

    if (empty($field_storage_config)) {
      // This is the case for base fields (fields that are defined directly in
      // the entity class)
      // We can handle base-fields in advance.
      if ($field_id == "bundle" || $field_id == "eid") {
        foreach ($entity_ids as $eid) {

          if ($field_id == "bundle" && !empty($bundleid_in)) {
            $out[$eid]["bundle"] = [$bundleid_in];
            continue;
          }
          // drupal_set_message("I am asked for fids: " . serialize($field_ids));
          if ($field_id == "eid") {
            $out[$eid][$field_id] = [$eid];
            continue;
          }

          // Bundle is a special case.
          // If we are asked for a bundle, we first look in the pb cache for the bundle
          // because it could have been set by
          // measures like navigate or something - so the entity is always displayed in
          // a correct manor.
          // If this is not set we just select the first bundle that might be appropriate.
          // We select this with the first field that is there. @TODO:
          // There might be a better solution to this.
          // e.g. knowing what bundle was used for this id etc...
          // however this would need more tables with mappings that will be slow in case
          // of a lot of data...
          if ($field_id == "bundle") {

            if (!empty($bundleid_in)) {
              $out[$eid]['bundle'] = [$bundleid_in];
              continue;
            }

            // Get all the bundles for the eid from us.
            $bundles = $this->getBundleIdsForEntityId($eid);

            if (!empty($bundles)) {
              // If there is only one, we take that one.
              // foreach($bundles as $bundle) {.
              $out[$eid]['bundle'] = array_values($bundles);
              // break;
              // }.
              continue;
            }
            else {
              // If there is none return NULL.
              $out[$eid]['bundle'] = NULL;
              continue;
            }
          }
        }
      }
      return $out;
    }

    if (!empty($field_storage_config)) {
      $main_property = $field_storage_config->getMainPropertyName();
      $target_type = $field_storage_config->getSetting("target_type");
    }

    if (!empty($field_id) && empty($bundleid_in)) {
      $this->messenger()
        ->addError("$field_id was queried but no bundle given.");
      return;
    }
    // drupal_set_message("a2: " . microtime());
    // Make an entity query for all relevant pbs with this adapter.
    $relevant_pb_ids = \Drupal::entityTypeManager()
      ->getStorage('wisski_pathbuilder')
      ->getQuery()
      ->condition('adapter', $this->adapterId())
      ->execute();

    // drupal_set_message("a3: " . microtime());
    // This approach will be not fast enough in the future...
    // the pbs have to have a better mapping of where and how to find fields.
    $pbs = WisskiPathbuilderEntity::loadMultiple($relevant_pb_ids);

    // drupal_set_message("a4: " . microtime());
    // What we loaded once we don't want to load twice.
    $adapter_cache = [];

    // Go through all relevant pbs.
    foreach ($pbs as $pb) {

      // Get the pbarray for this field.
      $pbarray = $pb->getPbEntriesForFid($field_id);

      // If there is no data about this path - how did we get here in the first place?
      // fields not in sync with pb?
      if (empty($pbarray["id"])) {
        continue;
      }

      $path = WisskiPathEntity::load($pbarray["id"]);

      // If there is no path we can skip that.
      if (empty($path)) {
        continue;
      }

      // If we find any data, we set this to true.
      $found_any_data = FALSE;

      foreach ($entity_ids as $eid) {
        // Every other field is an array, we guess
        // this might be wrong... cardinality?
        if (!isset($out[$eid][$field_id])) {
          $out[$eid][$field_id] = [];
        }

        // If this is question for a subgroup - handle it otherwise.
        if ($pbarray['parent'] > 0 && $path->isGroup()) {
          // @todo ueberarbeiten
          // drupal_set_message("danger zone!");
          $tmp = $this->pathToReturnValue($path, $pb, $eid, 0, $main_property);
          // $tmp = $this->pathToReturnValue($path, $pb, $eid, 0, $main_property, TRUE);
          foreach ($tmp as $key => $item) {
            $tmp[$key]["target_id"] = $this->getDrupalId($item["target_id"]);
          }

          // By mark
          // IF it is a group the language is probably the default language anyway?
          $tmp = [LanguageInterface::LANGCODE_DEFAULT => $tmp];

          $out[$eid][$field_id] = array_merge($out[$eid][$field_id], $tmp);
        }
        else {
          // It is a field?
          // get the parentid.
          $parid = $pbarray["parent"];

          // Get the parent (the group the path belongs to) to get the common group path.
          $par = WisskiPathEntity::load($parid);

          // If there is no parent it is a ungrouped path... who asks for this?
          if (empty($par)) {
            $this->messenger()
              ->addError("Path " . $path->getName() . " with id " . $path->id() . " has no parent.");
            continue;
          }

          // Get the clear path array.
          $clearPathArray = $pb->getRelativePath($path, FALSE);
          // $tmpt=microtime(true);
          // drupal_set_message("a1v1: " . microtime());
          $tmp = $this->pathToReturnValue($path, $pb, $eid, count($path->getPathArray()) - count($clearPathArray), $main_property);
          // By mark - for now we load all languages which means we provide "all" as a parameter.
          // dpm($tmp, "tmp?");.
          // $tmp = $this->pathToReturnValue($path, $pb, $eid, count($path->getPathArray()) - count($clearPathArray), $main_property, TRUE);
          // drupal_set_message("a1v2: " . microtime());
          // \Drupal::logger('WissKI Import lpvff')->debug((microtime(TRUE)-$tmpt). ": $field_id ".$path->id());
          if ($main_property == 'target_id') {

            $oldtmp = $tmp;

            // Special case for files - do not ask for a uri.
            if ($target_type == "file") {
              foreach ($tmp as $key => $item) {
                $tmp[$key]["target_id"] = $item["target_id"];
                $tmp[$key]["original_target_id"] = $item["target_id"];
              }
            }
            else {

              foreach ($tmp as $key => $item) {
                $tmp[$key]["original_target_id"] = $item["target_id"];
                $tmp[$key]["target_id"] = $this->getDrupalId(isset($item['wisskiDisamb']) ? $item["wisskiDisamb"] : $item["target_id"]);
              }
            }

          }

          // Merge it manually as recursive merge does not work properly in case of multi arrays.
          if ($main_property == 'target_id') {
            // dpm($tmp, "tmp");.
            foreach ($tmp as $key => $item) {
              $skip = FALSE;
              // Check if the value is already there...
              foreach ($out[$eid][$field_id] as $field) {
                // if($field["target_id"] == $item["target_id"]) {.
                if (isset($field[$key]) && $field[$key]["target_id"] == $item["target_id"]) {
                  $skip = TRUE;
                  break;
                }
              }

              // $tmp = array(LanguageInterface::LANGCODE_DEFAULT => $tmp);
              // By MyF: iterate through all available languages in order to make the entity reference case language dependent
              // without this change, the entity references were pulled out of the cache and this led to errors in the view (the field was not
              // displayed in languages other than the default language)
              /* foreach($available_languages as $al){
              // initialize if not existing
              // by MyF: ASSUMPTION: we delete the x-default language case since we do not need it anymore
              if(!isset($out[$eid][$field_id][$al])){
              $out[$eid][$field_id][$al] = array();
              }
              // if we don't skip, add it via array_merge...
              if(!$skip) {
              if(!isset($out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][$key])){
              //   $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][$key] = $item;
              $out[$eid][$field_id][$al][$key] = $item;
              } else {
              //   $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][] = $item;
              $out[$eid][$field_id][$al][] = $item;
              }
              }
              }*/

              // By MyF: Our intention was to make all entity references translatable but the code above is wrong
              // The problem is that we initialize all languages but we have to initialize only those which are translated
              // If we do it like that, translations in entity reference fields cause the source entity to exist in all languages although this is not correct;
              // This is due to the storage which checks in which languages the entity exists and then performs a wrong visualization
              // The further problem is that we do not have access to all translations of an entity since we do not load it here => we have to make this in the
              // WisskiStorage.php file since we load the entity there.
              if (!isset($out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT])) {
                $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT] = [];
              }

              // If we don't skip, add it via array_merge...
              if (!$skip) {
                if (!isset($out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][$key])) {
                  $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][$key] = $item;
                }
                else {
                  $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT][] = $item;
                }

                // By mark: we also add it in the correct language
                // if(isset($item['wisski_language']))
                // $out[$eid][$field_id][$item['wisski_language']][$key] = $item;.
                // $out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT] = array_merge($out[$eid][$field_id][LanguageInterface::LANGCODE_DEFAULT], $item);.
              }
            }
          }
          // "normal" behaviour
          else {
            // dpm($tmp, "merging with " . serialize($out[$eid][$field_id]));.
            $new_tmp = [];

            // By mark: Sort by language.
            foreach ($tmp as $key => $value) {
              $new_tmp[$value['wisski_language']][$key] = $value;
            }

            $out[$eid][$field_id] = array_merge($out[$eid][$field_id], $new_tmp);
          }
        }

        if (empty($out[$eid][$field_id])) {
          unset($out[$eid]);
        }
      }
    }

    // dpm($out, "out?");
    // drupal_set_message("a3: " . microtime());
    return $out;
  }

  /**
   *
   */
  public function getQueryObject(EntityTypeInterface $entity_type, $condition, array $namespaces) {
    // Do NOT copy this to parent, this is namespace dependent.
    return new Query($entity_type, $condition, $namespaces, $this);
  }

  /**
   *
   */
  public function deleteOldFieldValue($entity_id, $fieldid, $value, $pb, $count = 0, $mainprop = FALSE, $language = "und") {
    // drupal_set_message("entity_id: " . $entity_id . " field id: " . $fieldid . " value " . serialize($value));
    // Get the pb-entry for the field
    // this is a hack and will break if there are several for one field.
    $pbarray = $pb->getPbEntriesForFid($fieldid);

    $field_storage_config = FieldStorageConfig::loadByName('wisski_individual', $fieldid);

    // Store the target type to see if it references to a file for special handling.
    $target_type = NULL;

    if (!empty($field_storage_config)) {
      $target_type = $field_storage_config->getSetting("target_type");
    }

    // If there is absolutely nothing, we don't delete something.
    if (empty($pbarray)) {
      return;
    }

    if (!isset($pbarray['id'])) {
      $this->messenger()->addWarning . ("Danger zone: in PB " . $pb->getName() . " field $fieldid was queried with value $value in deleteOldFieldValue, but the path with array " . serialize($pbarray) . " has no id.");
      return;
    }

    // So what? delete nothing?!
    if (empty($value)) {
      return;
    }

    // Get path/field-related config
    // and do some checks to ensure that we are acting on a
    // well configured field.
    $path = WisskiPathEntity::load($pbarray['id']);
    if (empty($path)) {
      return;
    }

    // This is an important distinction till now!
    // @todo maybe we can combine reference delete and value delete
    // in case of file we don't need to have a disamb and it is not a real reference.
    // however we have to change the object.
    $is_reference = (($mainprop == 'target_id' && !empty($path->getDisamb())) || $path->isGroup() || ($pbarray['fieldtype'] == 'entity_reference'));
    // dpm($is_reference, "main");.
    // $image_value = NULL;
    // This is the special case for files... we have to adjust the object here.
    // the url of the file is directly used as value to have some more meaning in the
    // triple store than just an entity id
    // this might have sideeffects... we will see :)
    if ($target_type == "file" && $mainprop == 'target_id') {
      if (is_numeric($value)) {
        $value = $this->getUriForDrupalId($value, FALSE);
      }
    }

    // dpm($value, "value");
    // dpm($pbarray, "pbarr");
    // dpm($mainprop, "main");
    // dpm(serialize($is_reference), "is ref");.
    if ($is_reference) {
      // Delete a reference
      // this differs from normal field values as there is no literal
      // and the entity has to be matched to the uri.
      $subject_uri = $this->getUriForDrupalId($entity_id, FALSE);
      // dpm($subject_uri, "subj");.
      if (empty($subject_uri)) {
        // The adapter doesn't know of this entity. some other adapter needs
        // to handle it and we can skip it.
        return;
      }
      $subject_uris = [$subject_uri];

      // Value is the Drupal id of the referenced entity.
      $object_uri = $this->getUriForDrupalId($value, FALSE);
      // dpm($object_uri, "obj");.
      if (empty($object_uri)) {
        // The adapter doesn't know of this entity. some other adapter needs
        // to handle it and we can skip it.
        return;
      }

      $path_array = $path->getPathArray();

      if (count($path_array) < 3) {
        // This should never occur as it would mean that someone is deleting a
        // reference on a path with no triples!
        $this->messenger()
          ->addError("Bad path: trying to delete a ref with a too short path.");
        return;
      }
      elseif (count($path_array) == 3) {
        // We have the spacial case where subject and object uri are directly
        // linked in a triple <subj> <prop> <obj> / <obj> <inverse> <subj>.
        // So we know which triples to delete and can skip costly search for
        // the right triple.
        // Nothing to do!
      }
      else {
        // In all other cases we need to readjust the subject uri to cut the
        // right triples.
        $pathcnt = 0;
        $parent = WisskiPathEntity::load($pbarray['parent']);
        if (empty($parent)) {
          // Lonesome path!?
        }
        else {
          // We cannot use clearPathArray() here as path is a group and
          // the function would chop off too much.
          // @todo check if we can use the new *relative* methods.
          $parent_path_array = $parent->getPathArray();
          $pathcnt = (count($parent_path_array) + 1) / 2;

          // We need to take the start, too.
          $pathcnt = $pathcnt - 1;
        }

        // We have to set disamb manually to the last instance
        // otherwise generateTriplesForPath() won't produce right triples
        // By Mark:
        // In modern d9 drupal systems entity reference is a common thing to do
        // so don't queue up like that.
        // if($path->isGroup()) { // only do this for groups - for fields we have to handle this otherwise.
        if (empty($path->getDisamb())) {
          $disamb = (count($path_array) + 1) / 2;
        }
        else {
          $disamb = $path->getDisamb();
        }

        // The var that interests us is the one before disamb.
        // substract 2 as the disamb count starts from 1 whereas vars start from 0!
        // in W8, the x increases by 2!
        $subject_var = "x" . (($disamb - 2) * 2);

        // In this case the start of the path is the disamb and the relevant
        // information is already in $subject_uri... so we dont have to query!
        if ($pathcnt == ($disamb - 2)) {
          $subject_uris = [$subject_uri];
        }

        else {

          // Build up a select query that get us.
          $select = "SELECT DISTINCT ?$subject_var WHERE {";
          $select .= $this->generateTriplesForPath($pb, $path, "", $subject_uri, $object_uri, $disamb, $pathcnt, FALSE, NULL, 'entity_reference');
          $select .= "}";

          // dpm($pathcnt, "pathcnt?");
          // dpm($disamb, "disamb?");
          // dpm($subject_var, "subvar?");
          // dpm($select, "select");.
          $result = $this->directQuery($select);

          if ($result->numRows() == 0) {
            // There is no relation any more. has been deleted before!?
            return;
          }
          // ddl(array($disamb, $subject_var, $select,$result, $result->numRows()), 'delete disamb select');.
          // Reset subjects.
          $subject_uris = [];
          foreach ($result as $row) {
            $subject_uris[] = $row->{$subject_var}->getUri();
          }
        }
        // } else { // this is the case for the entity-reference fields that are not made by wisski
        //          $subject_uris = array($subject_uri);
        //        }
      }

      // $deltriples = $this->generateTriplesForPath($pb, $path, $object_uri, $subject_uri, NULL, $path->getDisamb(), $pathcnt-1, FALSE, NULL, 'normal');
      // If it is a entity reference we take that before the disamb!
      if (!$path->isGroup() && $is_reference && $path->getDisamb()) {
        $prop = $path_array[(($path->getDisamb() - 1) * 2) - 1];
      }
      // if($path->isGroup())
      else {
        $prop = $path_array[count($path_array) - 2];
      }

      $inverse = $this->getInverseProperty($prop);
      $delete = "DELETE DATA {\n";
      foreach ($subject_uris as $subject_uri) {
        $delete .= "  <$subject_uri> <$prop> <$object_uri> .\n";
        ## prevent "EasyRdf\Exception: HTTP request for SPARQL query failed: MALFORMED QUERY" error on empty inverse property
        if ($inverse !== '') {
          $delete .= "  <$object_uri> <$inverse> <$subject_uri> .\n";
	}
      }
      $delete .= ' }';
      // dpm($delete, "sparql");.
      $result = $this->directUpdate($delete);

    }
    // End reference branch.
    // No reference.
    else {

      $subject_uri = $this->getUriForDrupalId($entity_id, FALSE);
      $starting_position = $pb->getRelativeStartingPosition($path, TRUE);
      $clearPathArray = $pb->getRelativePath($path, TRUE);

      // Delete normal field value.
      $sparql = "SELECT DISTINCT ";

      for ($i = ($starting_position * 2); $i <= count($path->getPathArray()); $i += 2) {
        $sparql .= "?x" . $i . " ";
      }

      // I am unsure about this.
      $sparql .= "?out ";

      $sparql .= "WHERE { ";

      // I am unsure if this is correct
      // probably it needs to be relative - but I am unsure
      // $triples = $this->generateTriplesForPath($pb, $path, $value, $eid, NULL, 0, $diff, FALSE);
      // make a query without the value - this is necessary
      // because we have to think of the weight.
      // $triples = $this->generateTriplesForPath($pb, $path, '', $subject_uri, NULL, 0, $starting_position, FALSE);.
      $triples = $this->generateTriplesForPath($pb, $path, '', $subject_uri, NULL, 0, $starting_position, FALSE, '=', 'field', TRUE, [], 0, $language);

      $sparql .= $triples;

      $sparql .= " }";
      // dpm($sparql, "sparql");.
      $result = $this->directQuery($sparql);

      $outarray = [];
      // dpm($result, "result");
      // dpm($count, "count");.
      $loc_count = 0;
      $break = FALSE;

      $position = NULL;
      $the_thing = NULL;

      foreach ($result as $key => $thing) {

        $image_value = NULL;

        // dpm($target_type, "tt");
        // dpm($mainprop, "mp");
        // Special hack for images which might be received via uri.
        if ($target_type == "file" && $mainprop == 'target_id') {

          // dpm("hacking...");.
          $loc = NULL;
          $fid = \Drupal::service('entity_type.manager')
            ->getStorage('wisski_individual')
            ->getFileId($thing->out, $loc);
          // dpm($fid, "fid");.
          $public = \Drupal::service('entity_type.manager')
            ->getStorage('wisski_individual')
            ->getPublicUrlFromFileId($fid);

          $image_value = $public;

        }

        // dpm($image_value, "img");
        // dpm($value, "val");.
        // Easy case - it is at the "normal position".
        if ($key === $count && ($thing->out == $value || $image_value == $value)) {
          $the_thing = $thing;
          $position = $count;
          break;
        }

        // Not so easy case - it is somewhere else.
        if ($thing->out == $value) {
          $position = $key;
          $the_thing = $thing;
          if ($key >= $count) {
            break;
          }
        }
      }

      if (is_null($position)) {
        $this->messenger()->addError($this->t(
          "For path %name (%id): Could not find old value '@v' and thus could not delete it.",
          [
            '%name' => $path->getName(),
            '%id' => $path->id(),
            '@v' => $value,
          ]
        ));
        return;
      }

      /* We cannot use DELETE DATA as we do not know the graph(s) of the triple.
       * The code below would only delete the triple if it is in the default
       * graph. we cannot omit the graph as fuseki needs it.
       * so we instead make a DELETE WHERE update and leave the graph unspecified.
      // for fuseki we need graph
      $delete  = "DELETE DATA { GRAPH <".$this->getDefaultDataGraphUri()."> {";

      #      drupal_set_message("cpa: " . serialize($clearPathArray));

      // the datatype-property is not directly connected to the group-part
      if(count($clearPathArray) >= 3) {
      $prop = array_values($clearPathArray)[1];
      $inverse = $this->getInverseProperty($prop);

      $name = "x" . ($starting_position * 2 +2);

      $object_uri = $the_thing->{$name}->getUri();

      $delete .= "  <$subject_uri> <$prop> <$object_uri> .\n";
      $delete .= "  <$object_uri> <$inverse> <$subject_uri> .\n";

      }
      else {
      $primitive = $path->getDatatypeProperty();

      if(!empty($primitive)) {
      if(!empty($value)) {
      $value = $this->escapeSparqlLiteral($value);
      // Evil: no datatype or lang check!
      $delete .= "  <$subject_uri> <$primitive> '$value' .\n";
      #dpm([$primitive, $value, $fieldid,$delete], __METHOD__.__LINE__);
      }
      else {
      drupal_set_message($this->t(
      "Path %name (%id) has primitive but no value given.",
      array(
      '%name' => $path->getName(),
      '%id' => $path->id()
      )
      ),
      "error"
      );
      return;
      }
      }
      }

      $delete .= ' }}';
       */

      $delete_clause = "DELETE {\n  GRAPH ?g {\n";
      $where_clause = "WHERE {\n  GRAPH ?g {\n";

      // We have to distinguish between whether to delete a datatype prop
      // or an object prop.
      // first object prop.
      if (count($clearPathArray) >= 3) {
        // The datatype-property is not directly connected to the group-part.
        $prop = array_values($clearPathArray)[1];
        $inverse = $this->getInverseProperty($prop);
        $name = "x" . ($starting_position * 2 + 2);
        $object_uri = $the_thing->{$name}->getUri();
        $delete_clause .= "    <$subject_uri> <$prop> <$object_uri> .\n";
	## prevent "EasyRdf\Exception: HTTP request for SPARQL query failed: MALFORMED QUERY" error on empty inverse property
        if ($inverse !== '') {
          $delete_clause .= "    <$object_uri> <$inverse> <$subject_uri> .\n";
	}
        $where_clause .= "    {  <$subject_uri> <$prop> <$object_uri> . }\n";
	## prevent "EasyRdf\Exception: HTTP request for SPARQL query failed: MALFORMED QUERY" error on empty inverse property
        if ($inverse !== '') {
          $where_clause .= "    UNION\n";
          $where_clause .= "    {  <$object_uri> <$inverse> <$subject_uri> . }\n";
	}
      }
      // Now datatype prop.
      else {
        $primitive = $path->getDatatypeProperty();

        if (!empty($primitive)) {
          if (!empty($value)) {
            $escaped_value = $this->escapeSparqlLiteral($value);
            // Evil: no datatype or lang check!
            $delete_clause .= "    <$subject_uri> <$primitive> ?out .\n";
            $where_clause .= "    <$subject_uri> <$primitive> ?out .\n    FILTER (STR(?out) = '$escaped_value')\n";
          }
          else {
            $this->messenger()->addError($this->t(
              "Path %name (%id) has primitive but no value given.",
              [
                '%name' => $path->getName(),
                '%id' => $path->id(),
              ]
            ));
            return;
          }
        }
      }

      // Assemble the clauses.
      $delete = "$delete_clause  }\n}\n$where_clause  }\n}";
      // dpm(htmlentities($delete), "delete!");.
      $result = $this->directUpdate($delete);

    }

  }

  /**
   * Delete an entity.
   *
   * @param $entity
   *   an entity object
   *
   * @return True or false if it did not work
   */
  public function deleteEntity($entity) {
    // dpm("yay?");.
    $eid = $entity->id();

    if (empty($eid)) {
      $this->messenger()
        ->addError("This entity could not be deleted as it has no eid.");
      return;
    }

    $pbs = WisskiPathbuilderEntity::loadMultiple();

    // If there is an eid we try to get the entity URI form cache
    // if there is none $uri will be FALSE.
    $uri = $this->getUriForDrupalId($eid, FALSE);

    // $sparql = "DELETE { GRAPH <" . $this->getDefaultDataGraphUri() . "> { ?s ?p ?o } } " .
    // "WHERE { { GRAPH <" . $this->getDefaultDataGraphUri() . "> { ?s ?p ?o . FILTER ( <$uri> = ?s ) } } " .
    // "UNION { GRAPH <" . $this->getDefaultDataGraphUri() . "> { ?s ?p ?o . FILTER ( <$uri> = ?o ) } } }";
    // We can't use the default graph here as the uri may also occur in other graphs
    // by mark: This is inefficient. We do an or in the filter!
    // $sparql = "DELETE { GRAPH ?g { ?s ?p ?o } } " .
    // "WHERE { { GRAPH ?g { ?s ?p ?o . FILTER ( <$uri> = ?s ) } } " .
    // "UNION { GRAPH ?g { ?s ?p ?o . FILTER ( <$uri> = ?o ) } } }";
    // This does not work either:
    // $sparql = "DELETE { GRAPH ?g { ?s ?p ?o } } " .
    // "WHERE { { GRAPH ?g { ?s ?p ?o . FILTER ( <$uri> = ?s || <$uri> = ?o) } } }";
    // \Drupal::logger('WissKIsaveProcess')->debug('sparql deleting: ' . htmlentities($sparql));
    $sparql = "DELETE { GRAPH ?g { <$uri> ?p ?o } } " .
      "WHERE { { GRAPH ?g { <$uri> ?p ?o } } }";

    $result = $this->directUpdate($sparql);

    $sparql = "DELETE { GRAPH ?g { ?s ?p <$uri> } } " .
      "WHERE { { GRAPH ?g { ?s ?p <$uri> } } }";

    $result = $this->directUpdate($sparql);

    return $result;

  }

  /**
   * Create a new entity.
   *
   * @param $entity
   *   an entity object
   * @param $entity_id
   *   the eid to be set for the entity, if NULL and $entity
   *   dowes not have an eid, we will try to create one
   *
   * @return the Entity ID
   */
  public function createEntity($entity, $entity_id = NULL) {
    // dpm("create was called");
    // $uri = $this->getUri($this->getDefaultDataGraphUri());
    // dpm(func_get_args(),__FUNCTION__);
    // \Drupal::logger('WissKIsaveProcess')->debug(__METHOD__ . " with values: " . serialize(func_get_args()));
    $bundleid = $entity->bundle();

    $pbs = WisskiPathbuilderEntity::loadMultiple();

    $out = [];

    // Might be empty, but we can use it later.
    $eid = $entity->id() ?: $entity_id;
    $uri = NULL;
    // dpm($eid,$entity_id);
    // if there is an eid we try to get the entity URI form cache
    // if there is none $uri will be FALSE.
    if (!empty($eid)) {
      $uri = $this->getUriForDrupalId($eid, TRUE);
    }
    else {
      $uri = $this->getUri($this->getDefaultDataGraphUri());
    }
    // Get the adapterid that was loaded
    // haha, this is the engine-id...
    // $adapterid = $this->getConfiguration()['id'];
    // \Drupal::logger('WissKIsaveProcess')->debug(__METHOD__ . " with values: " . serialize(func_get_args()) . " gets id: " . $eid . " and uri: " . $uri);.
    foreach ($pbs as $pb) {
      // drupal_set_message("a2: " . microtime());
      // if we have no adapter for this pb it may go home.
      if (empty($pb->getAdapterId())) {
        continue;
      }

      $adapter = Adapter::load($pb->getAdapterId());

      // If we have not adapter, we may go home, too.
      if (empty($adapter)) {
        continue;
      }

      // If he didn't ask for us...
      if ($this->adapterId() !== $adapter->id()) {
        continue;
      }

      // dpm('I can create',$adapter->id());
      $groups = $pb->getGroupsForBundle($bundleid);

      // For now simply take the first one.
      if ($groups = current($groups)) {

        $triples = $this->generateTriplesForPath($pb, $groups, '', $uri, NULL, 0, ((count($groups->getPathArray()) - 1) / 2), TRUE, NULL, 'group_creation');
        // dpm(array('eid'=>$eid,'uri'=>$uri,'group'=>$groups->getPathArray()[0],'result'=>$triples),'generateTriplesForPath');
        $sparql = "INSERT DATA { GRAPH <" . $this->getDefaultDataGraphUri() . "> { " . $triples . " } } ";
        // \Drupal::logger('WissKIsaveProcess')->debug('sparql writing in create: ' . htmlentities($sparql));
        // dpm($sparql, "group creation");
        $result = $this->directUpdate($sparql);

        if (empty($uri)) {

          // First adapter to write will create a uri for an unknown entity.
          $uri = explode(" ", $triples, 2);

          $uri = substr($uri[0], 1, -1);
        }
      }
    }
    // dpm($groups, "bundle");.
    // $entity->set('id',$uri);.
    if (empty($eid)) {
      $eid = $this->getDrupalId($uri);
    }
    // dpm($eid,$adapter->id().' made ID');.
    $entity->set('eid', $eid);
    // "INSERT INTO { GRAPH <" . $this->getDefaultDataGraphUri() . "> { "
    return $eid;
  }

  /**
   *
   */
  public function getUri($prefix) {
    return uniqid($prefix);
  }

  /**
   * Generate the triple part for the statements (excluding any Select/Insert
   * or
   * whatever). This should be used for any pattern generation. Everything else
   * is evil.
   *
   * @param $pb
   *   a pathbuilder instance
   * @param $path
   *   the path as a path object of which the triple parts should be
   *   generated. May also be a group.
   * @param $primitiveValue
   *   The primitive data value that should be stored or
   *   asked for in the query.
   * @param $subject_in
   *   If there should be any subject on a certain position
   *   this could be encoded by using $subject_in and the
   *              $startingposition parameter.
   * @param $object_in
   *   If there should be any object. The position of the
   *   object
   *              may be encoded in the disambposition.
   * @param $disambposition
   *   The position in the path where the object or the
   *   general disambiguation of this path lies. 0 means no disamb,
   *              1 means disamb on the first concept, 2 on the second concept
   *              and so on.
   * @param $startingposition
   *   From where on the path should be generated in
   *   means of concepts from the beginning (warning: not like disamb! 0 means
   *   start here!).
   * @param $write
   *   Is this a write or a read-request?
   * @param $op
   *   How should it be compared to other data
   * @param $mode
   *   defaults to 'field' - but may be 'group' or
   *   'entity_reference' in special cases
   * @param $relative
   *   should it be relative to the other groups?
   * @param $variables
   *   the variable of index i will be set to the value of key
   *   i. The variable ?out will be set to the key "out".
   * @param $numbering
   *   the initial numbering for the results - typically
   *   starting with 0. But can be modified by this.
   */
  public function generateTriplesForPath($pb, $path, $primitiveValue = "", $subject_in = NULL, $object_in = NULL, $disambposition = 0, $startingposition = 0, $write = FALSE, $op = '=', $mode = 'field', $relative = TRUE, $variable_prefixes = [], $numbering = 0, $language = "und") {
    // dpm($language, "called with language!");
    // drupal_set_message("gt1: " . microtime());
    // \Drupal::logger('WissKIsaveProcess')->debug('generate: ' . serialize(func_get_args()));
    // if($mode == 'entity_reference')
    // dpm(func_get_args(), "fun");
    // The query construction parameter.
    $query = "";
    // If we disamb on ourself, return.
    if ($disambposition == 0 && !empty($object_in)) {
      return "";
    }

    $transitive_modificator = "";
    // Is the path transitive?
    if (!empty($path->getTransitive())) {
      $transitive_modificator = "+";
    }

    // We get the sub-section for this path.
    $clearPathArray = [];
    if ($relative) {
      // In case of group creations we just need the " bla1 a type " triple.
      if ($mode == 'group_creation') {
        $clearPathArray = $pb->getRelativePath($path, FALSE);
      }
      // In any other case we need the relative path.
      else {
        $clearPathArray = $pb->getRelativePath($path);
      }
    }
    // Except some special cases.
    else {
      $clearPathArray = $path->getPathArray();
    }

    // The RelativePath will be keyed like the normal path array
    // meaning that it will not necessarily start at 0.
    // \Drupal::logger('WissKIsaveProcess')->debug('countdiff ' . $countdiff . ' cpa ' . serialize($clearPathArray) . ' generate ' . serialize(func_get_args()));
    // Old uri pointer.
    $olduri = NULL;
    $oldvar = NULL;

    // If the old uri is empty we assume there is no uri and we have to
    // generate one in write mode. In ask mode we make variable-questions.
    // Get the default datagraphuri.
    $datagraphuri = $this->getDefaultDataGraphUri();

    $first = TRUE;

    // Iterate through the given path array.
    $localkey = 0;
    foreach ($clearPathArray as $key => $value) {
      $localkey = $key;

      if ($first) {
        if ($key > ($startingposition * 2) || ($startingposition * 2) > ($key + count($clearPathArray))) {
          // dpm($key, "key");
          // dpm($startingposition, "starting");
          // dpm($clearPathArray, "cpa");.
          $this->messenger()
            ->addError("Starting Position is set to a wrong value: '$startingposition'. See reports for details");
          if (WISSKI_DEVEL) {
            \Drupal::logger('WissKIsaveProcess')
              ->debug('ERROR: ' . serialize($clearPathArray) . ' generate ' . serialize(func_get_args()));
          }
          if (WISSKI_DEVEL) {
            \Drupal::logger('WissKIsaveProcess')
              ->debug('ERROR: ' . serialize(debug_backtrace()[1]['function']) . ' and ' . serialize(debug_backtrace()[2]['function']));
          }
        }

        if ($disambposition > 0 && !empty($object_in)) {
          if ($key > (($disambposition - 1) * 2) || (($disambposition - 1) * 2) > ($key + count($clearPathArray))) {
            $this->messenger()
              ->addError("Disambposition is set to a wrong value: '$disambposition'. See reports for details.");
            if (WISSKI_DEVEL) {
              \Drupal::logger('WissKIsaveProcess')
                ->debug('ERROR: ' . serialize($clearPathArray) . ' generate ' . serialize(func_get_args()));
            }
            if (WISSKI_DEVEL) {
              \Drupal::logger('WissKIsaveProcess')
                ->debug('ERROR: ' . serialize(debug_backtrace()[1]['function']) . ' and ' . serialize(debug_backtrace()[2]['function']));
            }
          }
        }
      }

      $first = FALSE;

      // Skip anything that is smaller than $startingposition.
      if ($key < ($startingposition * 2)) {
        continue;
      }

      // Basic initialisation.
      $uri = NULL;

      // Basic initialisation for all queries.
      $localvar = "?" . ($variable_prefixes[$key] ?? "x" . ($numbering + $key));
      if (empty($oldvar)) {
        // This is a hack but i don't get the if's below
        // and when there should be set $oldvar.
        // @todo fix this!
        $oldvar = "?" . ($variable_prefixes[$key] ?? "x" . ($numbering + $key));
      }
      $graphvar = "?g_" . ($variable_prefixes[$key] ?? "x" . ($numbering + $key));

      if ($key % 2 == 0) {
        // If it is the first element and we have a subject_in
        // then we have to replace the first element with subject_in
        // and typically we don't do a type triple. So we skip the rest.
        if ($key == ($startingposition * 2) && !empty($subject_in)) {
          $olduri = $subject_in;

          if (!$write) {
            $query .= "GRAPH $graphvar { <$olduri> a <$value> } . ";
          }
          else {
            $query .= " <$olduri> a <$value> . ";
          }

          continue;
        }

        // If the key is the disambpos
        // and we have an object.
        if ($key == (($disambposition - 1) * 2) && !empty($object_in)) {
          $uri = $object_in;
          if ($write) {
            // We also write the class in case the object is already known to
            // the system but not to this adapter and there are no further
            // data values to be written in this adapter. in this case we would
            // end up with the uri correctly written to this adapter but the
            // reading query needs to check the class as well which wouldn't be
            // there, then.
            $query .= "<$uri> a <$value> . ";
          }
        }
        else {

          // If it is not the disamb-case we add type-triples.
          if ($write) {
            // Generate a new uri.
            $uri = $this->getUri($datagraphuri);
            $query .= "<$uri> a <$value> . ";
          }
          else {
            $query .= "GRAPH $graphvar { $localvar a <$value> } . ";
          }
        }
        // drupal_set_message("gttt?: " . microtime());
        // Magic function
        // this if writes the triples from one x to another (x_1 y_1 x_n+1)
        // for x0 $prop is not set and thus it does not match!
        //
        // for $prop we recognize a leading '^' for inverting the property
        // direction, see '^' as sparql property chain operator.
        if ($key > 0 && !empty($prop)) {

          if ($write) {

            if ($prop[0] == '^') {
              // We cannot use the '^' for sparql update.
              $prop = substr($prop, 1);
              $query .= "<$uri> <$prop> <$olduri> . ";
            }
            else {
              $query .= "<$olduri> <$prop> <$uri> . ";
            }

          }
          else {

            $inv_sign = '';
            if ($prop[0] == '^') {
              $inv_sign = '^';
              $prop = substr($prop, 1);
            }

            $query .= "GRAPH {$graphvar}_1 { ";
            $inverse = $this->getInverseProperty($prop);
            // drupal_set_message("inverse?: " . microtime());
            // If there is not an inverse, don't do any unions.
            if (empty($inverse)) {
              if (!empty($olduri)) {
                $query .= "<$olduri> ";
              }
              else {
                $query .= "$oldvar ";
              }

              $query .= "$inv_sign<$prop>$transitive_modificator ";

              if (!empty($uri)) {
                $query .= "<$uri> . ";
              }
              else {
                $query .= "$localvar . ";
              }
            }
            // If there is an inverse, make a union.
            else {
              $query .= "{ { ";
              // Forward query part.
              if (!empty($olduri)) {
                $query .= "<$olduri> ";
              }
              else {
                $query .= "$oldvar ";
              }

              $query .= "$inv_sign<$prop>$transitive_modificator ";

              if (!empty($uri)) {
                $query .= "<$uri> . ";
              }
              else {
                $query .= "$localvar . ";
              }

              $query .= " } UNION { ";

              // Backward query part.
              if (!empty($uri)) {
                $query .= "<$uri> ";
              }
              else {
                $query .= "$localvar ";
              }

              $query .= "$inv_sign<$inverse>$transitive_modificator ";

              if (!empty($olduri)) {
                $query .= "<$olduri> . ";
              }
              else {
                $query .= "$oldvar . ";
              }

              $query .= " } } . ";
            }

            $query .= " } . ";
          }
        }

        // If this is the disamb, we may break.
        if ($key == (($disambposition - 1) * 2) && !empty($object_in)) {
          break;
        }

        $olduri = $uri;
        $oldvar = $localvar;
      }
      else {
        $prop = $value;
      }
    }
    // drupal_set_message("gt2: " . microtime());
    // \Drupal::logger('testung')->debug($path->getID() . ":".htmlentities($query));
    // Get the primitive for this path if any.
    $primitive = $path->getDatatypeProperty();

    $pb_path_info = $pb->getPbPath($path->id());
    $has_primitive = !empty($primitive) && $primitive != "empty";
    // All paths should have a primitive. only this cases usually have no
    // primitive:
    // - a group
    // - a path belonging to an entity reference field
    // - a path that has no corresp field (depending on use it may or may not
    //   have a primitive)
    $should_have_primitive =
      !$path->isGroup()
      && $pb_path_info['fieldtype'] != 'entity_reference'
      && $pb_path_info['field'] != $pb::CONNECT_NO_FIELD;

    if (!$has_primitive && $should_have_primitive) {
      $this->messenger()
        ->addError("There is no primitive Datatype for Path " . $path->getName());
    }
    // If write context and there is an object, we don't attach the primitive
    // also if we create a group.
    elseif (($write && !empty($object_in) && !empty($disambposition)) || $mode == 'group_creation' || $mode == 'entity_reference') {
      // Do nothing!
    }
    elseif ($has_primitive) {
      $outvar = "?" . ($variable_prefixes["out"] ?? "out");
      $outgraph = "?g_" . ($variable_prefixes["out"] ?? "out");

      if (!$write) {
        $query .= "GRAPH $outgraph { ";
      }
      else {
        $query .= "";
      }

      if (!empty($olduri)) {
        $query .= "<$olduri> ";
      }
      else {
        // If we initialized with a nearly empty path oldvar is empty.
        // in this case we assume x at the startingposition.
        if (empty($oldvar)) {
          $query .= "?" . ($variable_prefixes[$localkey] ?? "x" . $startingposition);
        }
        else {
          $query .= "$oldvar ";
        }
      }

      $query .= "<$primitive> ";

      // dpm($query, "before?");.
      if (!empty($primitiveValue)) {
        // dpm($primitiveValue, "prim");.
        if ($write) {
          // We have to escape it otherwise the sparql query may break.
          $primitiveValue = $this->escapeSparqlLiteral($primitiveValue);

          // By mark: here we add the language tag for translatability
          // $query .= "\"$primitiveValue\"";.
          if ($language != "und") {
            $query .= "\"$primitiveValue\"@$language";
          }
          else {
            $query .= "\"$primitiveValue\"";
          }
        }
        else {

          /* putting there the literal directly is not a good idea as
          there may be problems with matching lang and datatype
          if($op == '=')
          $query .= "'" . $primitiveValue . "' . ";
          else {
          Instead we compare it to the STR()-value

           */

          $regex = FALSE;
          $negate = FALSE;
          $safe = FALSE;
          $cast_outvar = "STR($outvar)";

          if ($op == "STARTS") {
            $op = "STARTS_WITH";
          }

          if ($op == "ENDS") {
            $op = "ENDS_WITH";
          }

          if ($op == '<>') {
            $op = '!=';
          }
          elseif ($op == 'EMPTY' || $op == 'NOT_EMPTY') {
            $regex = FALSE;
            $safe = TRUE;
          }
          elseif ($op == 'STARTS_WITH' || $op == 'starts_with') {
            $regex = FALSE;
            $safe = TRUE;
            // We don't do this here anymore, but do it with strStarts below
            // this should be faster on most triplestores than REGEX.
            // $primitiveValue = '^' . $this->escapeSparqlRegex($primitiveValue, TRUE);.
          }
          elseif ($op == 'NOT_STARTS_WITH' || $op == 'not_starts_with') {
            $regex = TRUE;
            $safe = TRUE;
            $negate = TRUE;
          }
          elseif ($op == 'ENDS_WITH' || $op == 'ends_with') {
            $regex = TRUE;
            $safe = TRUE;
            // $primitiveValue = $this->escapeSparqlRegex($primitiveValue, TRUE) . '$';
          }
          elseif ($op == 'NOT_ENDS_WITH' || $op == 'not_ends_with') {
            $regex = TRUE;
            $safe = TRUE;
            $negate = TRUE;
          }
          elseif ($op == 'REGEXP') {
            $regex = TRUE;
            $safe = TRUE;
          }
          elseif ($op == 'CONTAINS' || $op == 'contains' || $op == 'NOT') {
            $regex = TRUE;
            $safe = TRUE;

            // dpm($op, "op");.
            if ($op == "NOT") {
              $negate = TRUE;
              $op = "CONTAINS";
            }

            // dpm($primitiveValue, "prim");.
            $primitiveValue = $this->escapeSparqlRegex($primitiveValue, TRUE);
            // dpm($primitiveValue, "prim");.
          }
          elseif ($op == 'BETWEEN' || $op == "NOT BETWEEN") {
            $regex = FALSE;
            $negate = ($op == 'NOT BETWEEN');
            $safe = TRUE;
            $values = is_array($primitiveValue) ? $primitiveValue : explode(";", $primitiveValue);
            $primitiveValue = join(';', $values);
          }
          elseif ($op == 'IN' || $op == 'NOT IN' || $op == 'in' || $op == 'not in' || $op == 'not_in') {
            $regex = FALSE;
            $negate = ($op == 'NOT IN' || $op == 'not in');
            $safe = TRUE;
            $values = is_array($primitiveValue) ? $primitiveValue : explode(",", $primitiveValue);
            $primitiveValue = '"' . join('","', $values) . '"';
          }
          elseif ($op == 'IN_REGEX' || $op == 'NOT IN_REGEX') {
            $regex = TRUE;
            $negate = ($op == 'NOT IN' );
            $safe = TRUE;
            $values = is_array($primitiveValue) ? $primitiveValue : explode(",", $primitiveValue);
            foreach ($values as &$v) {
              $v = $this->escapeSparqlRegex($v, TRUE);
            }
            $primitiveValue = join('|', $values);
          }
          else {
            if (strtoupper($op) == "LONGERTHAN" || strtoupper($op) == "SHORTERTHAN") {
              $regex = FALSE;
              $safe = TRUE;
            }
          }
          // dpm($primitiveValue, "prim");.
          if (!$safe) {
            if (is_numeric($primitiveValue)) {
              // dpm("yo!");
              // $escapedValue = intval($primitiveValue);
              $escapedValue = $primitiveValue;
              $cast_outvar = "xsd:decimal($outvar)";
            }
            else {
              // dpm("else?");
              // $escapedValue = '"' . $this->escapeSparqlLiteral($primitiveValue) . '"';.
              // $query .= "\"$primitiveValue\"@$language";.
              if ($language != "und") {
                $escapedValue = '"' . $this->escapeSparqlLiteral($primitiveValue) . '"@' . $language;
              }
              else {
                // $query .= "\"$primitiveValue\"";
                // $escapedValue = '"' . $this->escapeSparqlLiteral($primitiveValue) . '"';
                if ($op == "=") {
                  $escapedValue = '?' . $path->id() . ($numbering + $key) . ' . FILTER ( STR(?' . $path->id() . ($numbering + $key) . ') = "' . $this->escapeSparqlLiteral($primitiveValue) . '" )';
                }
                else {
                  $escapedValue = '"' . $this->escapeSparqlLiteral($primitiveValue) . '"';
                }
                // $escapedValue = '?primtemp . FILTER ( STR(?primtemp) = "' . $this->escapeSparqlLiteral($primitiveValue) . '" )';
              }
            }
          }
          else {
            // dpm("else!");
            // $escapedValue = '"' . $primitiveValue . '"';.
            if ($language != "und") {
              $escapedValue = '"' . $primitiveValue . '"@' . $language;
            }
            // $query .= "\"$primitiveValue\"@$language";
            else {
              if ($op == "EQUALS" || $op == "equals" || $op == "LIKE" || $op == "=") {
                $escapedValue = '?' . $path->id() . ($numbering + $key) . ' . FILTER ( STR(?' . $path->id() . ($numbering + $key) . ') = "' . $primitiveValue . '" )';
              }
              else {
                $escapedValue = '"' . $primitiveValue . '"';
              }
              // $query .= "\"$primitiveValue\"";
            }
          }

          // dpm($query, "in between");.
          // dpm($escapedValue, "esc");.
          if ($op == 'BETWEEN' || $op == 'NOT BETWEEN') {
            [
              $val_min,
              $val_max,
            ] = is_array($primitiveValue) ? $primitiveValue : explode(";", $primitiveValue, 2);
            if (is_numeric($val_min) && is_numeric($val_max)) {
              $val_min = intval($val_min);
              $val_max = intval($val_max);
              $cast_outvar = "xsd:decimal($outvar)";
            }
            else {
              $val_min = $this->escapeSparqlLiteral($val_min);
              $val_max = $this->escapeSparqlLiteral($val_max);
            }
            $filter = "($cast_outvar >= $val_min && $cast_outvar <= $val_max)";
          }
          else {
            if ($op == "STARTS_WITH" || $op == 'starts_with' || $op == "NOT_STARTS_WITH") {
              $filter = "strStarts($cast_outvar, $escapedValue)";
            }
            else {
              if ($op == "ENDS_WITH" || $op == 'ends_with' || $op == "NOT_ENDS_WITH") {
                $filter = "strEnds($cast_outvar, $escapedValue)";
              }
              /* This is wrong here...
              else if($op == "EMPTY") {
              $filter = "!bound(" . $outvar . ")";
              } else if($op == "NOT_EMPTY") {
              $filter = "bound(" . $outvar . ")";
              } */
              else {
                if (strtoupper($op) == "LONGERTHAN" || strtoupper($op) == "SHORTERTHAN") {
                  $filter = "strlen($outvar) ";
                  if (strtoupper($op) == "LONGERTHAN") {
                    $filter .= ">";
                  }
                  else {
                    $filter .= "<";
                  }
                  $filter .= " $primitiveValue";
                }
                elseif ($regex) {
                  // We have to use STR() otherwise we may get into trouble with
                  // datatype and lang comparisons.
                  $filter = "REGEX($cast_outvar, " . $escapedValue . ', "i" )';
                }
                else {
                  // We have to use STR() otherwise we may get into trouble with
                  // datatype and lang comparisons.
                  $filter = "$cast_outvar " . $op . ' ' . $escapedValue;
                }
              }
            }
          }

          // If ($negate && $op != "CONTAINS") {
          //            $filter = "NOT( $filter )";
          //          } else.
          // && $op == "CONTAINS") {.
          if ($negate) {
            $filter = "!" . $filter;
          }

          // Speed up in case of equivalence.
          if ($op == "=") {
            if (is_numeric($primitiveValue)) {
              $escapedValue = '?' . $path->id() . ($numbering + $key) . ' . FILTER ( STR(?' . $path->id() . ($numbering + $key) . ') = "' . $escapedValue . '" )';
            } //"'" . $escapedValue . "'";
            $query .= " " . $escapedValue . " . ";
          }
          elseif ($op == "EMPTY" || $op == "NOT EMPTY") {
            $query .= " $outvar . ";
          }
          elseif ($op == "IN" || $op == "NOT IN"){
            $query .= " $outvar . FILTER (STR($outvar) $op ($primitiveValue) )";
          }
          else {
            $query .= " $outvar . FILTER( $filter ) . ";
          }
        }
      }
      else {
        $query .= " $outvar . ";
      }
      if (!$write) {
        $query .= " } . ";
      }
    }

    // dpm($query, "after?");
    // \Drupal::logger('WissKIsaveProcess')->debug('erg generate: ' . htmlentities($query));
    // if($mode == 'entity_reference')
    // \Drupal::logger('WissKIsaveProcess')->debug('erg generate: ' . htmlentities($query));
    // dpm($query, "query");
    // \Drupal::logger('Sparql Adapter gtfp')->debug(str_replace("\n", '<br/>', htmlentities($path->id().":\n".\Drupal\Core\Serialization\Yaml::encode($tsa))));
    // In case of empty it must be optional or we never will get something, because the path may not be there and be not there at the same time.
    if ($op == "EMPTY") {
      // This is the case if it is an entity reference for example.
      if (empty($outvar)) {
        $outvar = "?" . ($variable_prefixes[$localkey] ?? "x" . $startingposition);
      }
      // dpm($outvar, "changed it to:");
      // $query = " OPTIONAL { " . $query . " } . FILTER(!bound($outvar)) . ";
      // Do some kind of rewriting
      // the problem is if we do everything optional
      // we also dont get back ?x0 - which is probably bad
      // we have to give the triplestore SOME anchorpoint
      // so lets ask for x0!
      $query = substr($query, 0, strpos($query, "} . ") + 3) . " OPTIONAL { " . substr($query, strpos($query, "} . ") + 3) . " } . FILTER(!bound($outvar)) . ";
    }
    // dpm($query, "query");
    // dpm($query, "gt3: " . microtime());
    return $query;
  }

  /**
   *
   */
  public function addNewFieldValue($entity_id, $fieldid, $value, $pb, $mainprop = FALSE, $language = "und") {

    // dpm($language, "I get?");.
    // Compatibility purpose.
    // $entity_id = $entity->id();
    // dpm(serialize($entity), "yay?");
    // dpm(serialize($value), "value?");
    // dpm(serialize($fieldid), "fieldid?");.
    // drupal_set_message(" addNewFieldValue I get: " . $entity_id.  " with fid " . $fieldid . " and value " . $value . ' for pb ' . $pb->id() . ' er ' . serialize($value_is_entity_ref));
    // drupal_set_message(serialize($this->getUri("smthg")));
    $datagraphuri = $this->getDefaultDataGraphUri();

    $pbarray = $pb->getPbEntriesForFid($fieldid);

    // drupal_set_message("I fetched path: " . serialize($pbarray));
    $path = WisskiPathEntity::load($pbarray['id']);

    if (empty($path)) {
      return;
    }

    $field_storage_config = FieldStorageConfig::loadByName('wisski_individual', $fieldid);

    $target_type = NULL;

    if (!empty($field_storage_config)) {
      $target_type = $field_storage_config->getSetting("target_type");
    }
    // $autocomplete_title_pattern_enabled = $field_storage_config->getSetting("switch_value");
    // autocomplete_title_pattern_enabled is a per-field setting that enables
    // rendering title patterns in the autocomplete widget.
    // This setting changes the default behavior for saving from taking the
    // direct value of the field to the entity_reference mode which includes
    // parsing of value for brackets and extracting the value of the entity in
    // brackets.
    //
    // e.g. extracting the value 64 from the string "Archeological Collection (64)".
    $autocomplete_title_pattern_enabled = FALSE;

    // By Mark: I don't know why this was evaluated here. I don't think that $pbarray ever
    // had the information that is assumed here. I think this is deprecated code...
    //if (array_key_exists('autocomplete_title_pattern_enabled', $pbarray)) {
    //  $autocomplete_title_pattern_enabled = $pbarray['autocomplete_title_pattern_enabled'] == TRUE;
      // The "== TRUE" casts it into a boolean!
    //}

    // check if the autocomplete module exists
    $moduleHandler = \Drupal::service('module_handler');
    if ($moduleHandler->moduleExists('wisski_autocomplete')) {

      // if it exists check if we want the special title pattern behaviour.
      $config = \Drupal::service('config.factory')->getEditable("wisski.autocomplete");

      // get all settings
      $settings = $config->get($fieldid);

      // if this setting is not set, don't do it!
      if(!empty($settings) && isset($settings['useTitlePattern']) & !empty($settings['useTitlePattern']))
        $autocomplete_title_pattern_enabled = TRUE;
    }


    // We distinguish two modes of how to interpret the value:
    // entity ref: the value is an entity id that shall be linked to
    // normal: the value is a literal and may be disambiguated.
    $is_entity_ref = (($mainprop == 'target_id' && ($path->isGroup() || $pb->getPbEntriesForFid($fieldid)['fieldtype'] == 'entity_reference')));

    // Special case for files:
    if ($target_type == "file" && $mainprop == 'target_id') {
      if (is_numeric($value)) {
        $value = $this->getUriForDrupalId($value, TRUE);
      }
      // Else { // it might be that there are spaces in file uris. These are bad for TS-queries.
      // $strrpos = strrpos($value, '/');
      // if($strrpos) { // only act of there is a / in it.
      // $value = substr($value, 0, $strrpos) . rawurlencode(substr($value, $strrpos));
      // }
      // }.
    }

    // In case of no entity-reference we do not search because we
    // already get what we want!
    if ($path->getDisamb() && !$is_entity_ref) {
      $sparql = "SELECT ?x" . (($path->getDisamb() - 1) * 2) . " WHERE { ";

      // Starting position one before disamb because disamb counts the number of concepts, startin position however starts from zero.
      $sparql .= $this->generateTriplesForPath($pb, $path, $value, NULL, NULL, NULL, $path->getDisamb() - 1, FALSE, '=', 'field', TRUE, [], 0, $language);

      $sparql .= " }";
      // dpm($sparql, "disamb?");
      // drupal_set_message("spq: " . ($sparql));
      // dpm($path, "path");.
      $disambresult = $this->directQuery($sparql);
      // dpm("disamb query one: " . serialize($disambresult));
      // dpm(array($sparql, $disambresult), __METHOD__ . " disamb query1");.
      if (!empty($disambresult) && isset($disambresult[0]) && !empty($disambresult[0])) {
        // dpm("it is not empty!");.
        $disambresult = $disambresult[0];
        // This fails due to some bug?
        // $disambresult = current($disambresult);
      }
      else {
        // It might be that we have some legacy data and the language is not
        // set in the ts... so we have to do a search without language tag, too.
        $sparql = "SELECT ?x" . (($path->getDisamb() - 1) * 2) . " WHERE { ";
        $sparql .= $this->generateTriplesForPath($pb, $path, $value, NULL, NULL, NULL, $path->getDisamb() - 1, FALSE, '=', 'field', TRUE, [], 0);
        $sparql .= " }";

        $disambresult = $this->directQuery($sparql);
        if (!empty($disambresult) && isset($disambresult[0]) && !empty($disambresult[0])) {
          $disambresult = $disambresult[0];
        }
        // dpm(array($sparql, $disambresult), __METHOD__ . " disamb query2");.
      }
    }

    // Rename to uri.
    $subject_uri = $this->getUriForDrupalId($entity_id, TRUE);

    // extract value in autocomplete_title_pattern_enabled case:
    // - if user picks existing value in autocomplete, the entity id has to be removed
    // - if user enters new value into the field, the value *must not* be changed
    // (the pattern seems to be the only indicator for which case holds -- this means
    //  that a new value ending in a space followed by a number in braces cannot be
    //  inserted when using the "autocomplete enabled" widget!)
    $autocomplete_use_existing_value = FALSE;
    if ($autocomplete_title_pattern_enabled && preg_match("/ \(\d+\)$/", $value)) {
      $value = substr($value, strrpos($value, '(', -1) + 1, (strrpos($value, ')', -1) - strrpos($value, '(', -0) - 1));
      $autocomplete_use_existing_value = TRUE;
    }

    $sparql = "INSERT DATA { GRAPH <" . $datagraphuri . "> { ";

    // 1.) A -> B -> C -> D -> E (l: 9) and 2.) C -> D -> E (l: 5) is the relative, then
    // 1 - 2 is 4 / 2 is 2 - which already is the starting point.
    $start = ((count($path->getPathArray()) - (count($pb->getRelativePath($path)))) / 2);

    // if we have entity reference *or* {we use the autocomplete widget *and* the user has picked an existing value}
    if ($is_entity_ref || $autocomplete_use_existing_value === TRUE) {

      // If it is a group - we take the whole group path as disamb pos.
      if ($path->isGroup()) {
        // $sparql .= $this->generateTriplesForPath($pb, $path, "", $subject_uri, $this->getUriForDrupalId($value, TRUE), (count($path->getPathArray())+1)/2, $start, TRUE, '', 'entity_reference' );
        $sparql .= $this->generateTriplesForPath($pb, $path, "", $subject_uri, $this->getUriForDrupalId($value, TRUE), (count($path->getPathArray()) + 1) / 2, $start, TRUE, '', 'entity_reference', TRUE, [], 0, $language);
      }
      // If it is a field it has a disamb pos!
      else {
        // $sparql .= $this->generateTriplesForPath($pb, $path, "", $subject_uri, $this->getUriForDrupalId($value, TRUE), $path->getDisamb(), $start, TRUE, '', 'entity_reference');.
        $sparql .= $this->generateTriplesForPath($pb, $path, "", $subject_uri, $this->getUriForDrupalId($value, TRUE), $path->getDisamb(), $start, TRUE, '', 'entity_reference', TRUE, [], 0, $language);
      }
    }
    else {
      // $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, NULL, NULL, $start, TRUE);
      if (empty($path->getDisamb())) {
        $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, NULL, NULL, $start, TRUE, '=', 'field', TRUE, [], 0, $language);
      }
      else {
        // drupal_set_message("disamb: " . serialize($disambresult) . " miau " . $path->getDisamb());
        if (empty($disambresult) || empty($disambresult->{"x" . ($path->getDisamb() - 1) * 2})) {

          // Hier rein.
          $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, NULL, NULL, $start, TRUE, '=', 'field', TRUE, [], 0, $language);

        }

        // $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, NULL, NULL, $start, TRUE);
        else {
          // We may not set a value here - because we have a disamb result!
          //          $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, $disambresult->{"x" . ($path->getDisamb()-1)*2}->dumpValue("text"), $path->getDisamb(), $start, TRUE);.
          $sparql .= $this->generateTriplesForPath($pb, $path, $value, $subject_uri, $disambresult->{"x" . ($path->getDisamb() - 1) * 2}->dumpValue("text"), $path->getDisamb(), $start, TRUE, '=', 'field', TRUE, [], 0, $language);

        }
      }
    }
    $sparql .= " } } ";
    // \Drupal::logger('WissKIsaveProcess')->debug('sparql writing in add: ' . htmlentities($sparql));
    // dpm($sparql, __METHOD__ . " sparql");
    $result = $this->directUpdate($sparql);

    // drupal_set_message("I add field $field from entity $entity_id that currently has the value $value");.
  }

  /**
   *
   */
  public function writeFieldValues($entity, array $field_values, $pathbuilder, $bundle_id = NULL, $old_values = [], $force_new = FALSE, $initial_write = FALSE, $language = "und") {

    // Get back the entity id for compatibility purpose.
    $entity_id = $entity->id();

    // dpm($field_values);
    // dpm(serialize($entity->langcode->getValue()), "le langue dü entitü");
    // dpm(serialize($entity->getTranslationLanguages()), "trans`?");
    //
    // dpm(serialize($entity), "se ente?");.
    // dpm($language, "I save for language");.
    // dpm($field_values, "I get values?");.
    // dpm($old_values, "ov");
    // dpm($entity_id, "I am saving entity_id");
    // drupal_set_message(serialize("Hallo welt!") . serialize($entity_id) . " " . serialize($field_values) . ' ' . serialize($bundle));
    // dpm(func_get_args(), __METHOD__);
    // \Drupal::logger('WissKIsaveProcess')->debug(__METHOD__ . " with values: " . serialize(func_get_args()));
    // Tricky thing here is that the entity_ids that are coming in typically
    // are somewhere from a store. In case of rdf it is easy - they are uris.
    // In case of csv or something it is more tricky. So I don't wan't to
    // simply go to the store and tell it "give me the bundle of this".
    // The field ids come in handy here - fields are typically attached
    // to a bundle anyway. so I just get the bundle from there. I think it is
    // rather stupid that this function does not load the field values per
    // bundle - it is implicitely anyway like that.
    //
    // so I ignore everything and just target the field_ids that are mapped to
    // paths in the pathbuilder.
    // $tmpt = microtime(TRUE);
    $out = [];

    // Return $out;
    // Here we should check if we really know the entity by asking the TS for it.
    // this would speed everything up largely, I think.
    // by mark: additionally the below code always is true.
    // this results in default values of entities not getting initiated. I added
    // another parameter ($initial_write) for that which simply
    // ignores the old field values and forces a write.
    // @todo by mark: This should be checked inside of the entity itself.
    // By Mark:
    // I rewrite this code here as the entity interface has severely changed
    // and this logic does not follow the new procedures.
    // Nowadays we can check everything from the entity directly as it is a
    // regular content entity.
    //
    // I think this code never got used. So we can remove it anyway.
    // $init_entity = $this->loadEntity($entity_id);
    // dpm(serialize($init_entity->getTranslationLanguages()), "trans2?");
    // $init_entity = $this->hasEntity($entity_id);
    // \Drupal::logger('WissKI Import tmpc')->debug("l:".(microtime(TRUE)-$tmpt));
    // dpm(serialize($init_entity), "init");
    // dpm($force_new, "force!");.
    // dpm($init_entity->getTranslationLanguages(), "langs?");.
    // If there is nothing, continue.
    // by mark: currently the storage calls the createEntity. So this never may be used.
    // simply don't worry about it.
    // By Mark on code-rewrite:
    // this never happens.
    // so we get rid of it.
    // if (empty($init_entity)) {
    // //   dpm('empty entity',__FUNCTION__);
    //  If ($force_new) {
    //    $entity = new WisskiEntity(array('eid' => $entity_id,'bundle' => $bundle_id),'wisski_individual',$bundle_id);
    //    $this->createEntity($entity,$entity_id);
    //  } else return;
    // }
    // \Drupal::logger('WissKI Import tmpc')->debug("c:".(microtime(TRUE)-$tmpt));
    // By Mark on code-rewrite:
    // this never happens either...
    // if(empty($entity) && !empty($init_entity))
    //  $entity = $init_entity;.
    $components = [];

    // dpm(serialize($entity), "ente?");
    // dpm(serialize($entity->getOriginalValues()), "ori?");.
    // By mark on code-rewrite
    // throw out the init_entity as it always was initialized...
    // if (!isset($old_values) && !empty($init_entity)) {.
    if (!isset($old_values)) {
      // It would be better to gather this information from the form and not from the ts
      // there might have been somebody saving in between...
      // @todo !!!
      $ofv = \Drupal::service('entity_field.manager')
        ->getFieldDefinitions('wisski_individual', $bundle_id);

      // Load the view ids that are available for this.
      $view_ids = \Drupal::entityQuery('entity_view_display')
        ->accessCheck(TRUE)
        ->condition('id', 'wisski_individual.' . $bundle_id . '.', 'STARTS_WITH')
        ->execute();

      // As there might be several, load all.
      $settings = \Drupal::entityTypeManager()
        ->getStorage('entity_form_display')
        ->loadMultiple($view_ids);

      // We dont really know which one to take -.-
      // By Mark: We need to do more research on this!
      if (!empty($settings)) {
        $settings = current($settings);
      }
      else {
        $settings = NULL;
      }

      // If there is a setting.
      if ($settings) {
        // Load the components.
        $components = $settings->getComponents();

        // And throw away these that are not in there.
        // these fields are disabled and such should not be saved etc.
        // otherwise we might overwrite something we do not want to overwrite.
        foreach ($ofv as $key => $field_defi) {
          if (!isset($components[$key])) {
            unset($ofv[$key]);
            // dpm(serialize($components), "set?");.
          }
        }
      }
      // dpm(serialize(), "old field values");
      // dpm(\Drupal::entityManager()->
      // $view_ids = \Drupal::entityQuery('entity_view_display')
      // ->condition('id', 'wisski_individual.' . $bundle_id . '.', 'STARTS_WITH')
      // ->execute();
      //
      // // is there a view display for it?
      // $entity_view_displays = \Drupal::entityManager()->getStorage('entity_view_display')->loadMultiple($view_ids);
      //
      // // did we get something?
      // if(!empty($entity_view_displays))
      // $entity_view_display = current($entity_view_displays);
      // else
      // $entity_view_display = NULL;
      // dpm($entity_view_displays->getComponent('field_name'), "miau");
      //
      // foreach($ofv as $key => $field_defi) {
      // dpm(serialize($entity_view_display->getComponent($key)), "settings or $key");
      // }
      // dpm(array_keys($ofv), "ak");
      // dpm(serialize($ofv), "ofv!");.
      $old_values = $this->loadFieldValues([$entity_id], array_keys($ofv), $bundle_id);
      // dpm($old_values, "got ov?");.
      if (!empty($old_values)) {
        $old_values = $old_values[$entity_id];
      }

      // We have one big problem: if there are fields that are for display purpose only
      // like coordinates for a place that get filled automatically
      // so we just delete what we see and nothing more!
    }
    else {
      // We are in edit mode and unfortunatelly old_values is not filled accordingly with language-tags.
      // so we do this here.
      $new_structure = [];
      foreach ($old_values as $old_key => $old_value) {
        $new_structure[$old_key][$language] = $old_value;
      }

      $old_values = $new_structure;
    }

    $old_languages = [];
    foreach ($old_values as $old_entityid => $old_language_values) {
      // dpm($old_field_values, "??");.
      $lang_keys = array_keys($old_language_values);
      if (!empty($lang_keys)) {
        $old_languages = array_merge($old_languages, $lang_keys);
      }
    }

    $curr_languages = array_keys($entity->getTranslationLanguages());
    // dpm($curr_languages, "curr?");
    // dpm($old_languages, "old?");.
    $deleted_languages = array_diff($old_languages, $curr_languages);

    // Ignore x-default, because we dont want to delete this.
    foreach ($deleted_languages as $del_lang_key => $del_lang_val) {
      if ($del_lang_val == LanguageInterface::LANGCODE_DEFAULT) {
        unset($deleted_languages[$del_lang_key]);
      }
    }

    // If we have something here we have to delete that language
    // unfortunatelly this is a non trivial thing on a triple store
    // and you will burn in hell if you try to ;D.
    foreach ($old_values as $old_key => $old_value) {
      // This is just copy pasted from below. I guess you can do it better
      // but I wait until the selphie does change this here ;D.
      foreach ($deleted_languages as $del_lang) {

        $entity->removeTranslation($del_lang);

        // dpm($entity->label(), "label");
        // dpm(serialize($entity), "ser?");.
        if (!isset($old_value[$del_lang]) || empty($old_value[$del_lang])) {
          continue;
        }

        $old_value = $old_value[$del_lang];

        // In case there is no main prop it is typically value.
        if (isset($old_value['main_property'])) {
          $mainprop = $old_value['main_property'];
        }
        else {
          $mainprop = "value";
        }

        // dpm("we have to delete " . serialize($old_value));
        foreach ($old_value as $key => $val) {

          // Main prop?
          if (!is_array($val)) {
            continue;
          }

          // Empty value?
          if (empty($val[$mainprop])) {
            continue;
          }

          $this->deleteOldFieldValue($entity_id, $old_key, $val[$mainprop], $pathbuilder, $key, $mainprop, $del_lang);
        }
      }
    }

    // dpm(serialize($deleted_languages), "del?");.
    if (!empty($deleted_languages)) {
      // dpm("Im here!");.
      foreach ($deleted_languages as $deleted_language) {
        $cached_field_values = \Drupal::database()
          ->delete('wisski_entity_field_properties')
          // ->fields('f',array('fid', 'ident','delta','properties','lang'))
          ->condition('eid', $entity->id())
          ->condition('bid', $entity->bundle())
          ->condition('lang', $deleted_language)
          // ->condition('fid',$field_name)
          ->execute();
        // ->fetchAll();
        //      dpm(serialize($cached_field_values), "cache?");
      }
    }

    // dpm($deleted_languages, "was deleted");.
    // \Drupal::logger('WissKI Import tmpc')->debug("lf:".(microtime(TRUE)-$tmpt));
    // drupal_set_message("the old values were: " . serialize($old_values));
    // dpm($old_values,'old values');
    // dpm($field_values,'new values');
    // dpm($initial_write, "init");
    // dpm($language, "lang?");.
    // In case of an initial write we forget the old values.
    if ($initial_write) {
      $old_values = [];
    }

    // If there are fields in the old_values that were deleted in the current
    // version we have to get rid of these.
    // also if you delete some string completely it might be
    // that the key is not set in the new values anymore.
    // Check if we have to delete some values
    // we go thru the old values and search for an equal value in the new
    // values array
    // as we do this we also keep track of values that haven't changed so that we
    // do not have to write them again.
    // This case only fires if field_values[$old_key] is not set anymore
    // at all - so just if the field is deleted completely!
    foreach ($old_values as $old_key => $old_value) {

      // By Mark: unfortunatelly entity reference is in x-default
      // so we have to see what language we have here...
      $this_language = $language;
      if (!isset($old_value[$language]) && isset($old_value[LanguageInterface::LANGCODE_DEFAULT])) {
        $this_language = LanguageInterface::LANGCODE_DEFAULT;
      }

      if (!isset($old_value[$this_language])) {
        continue;
      }

      $old_value = $old_value[$this_language];
      // dpm("deleting key $old_key with value " . serialize($old_value) . " from values " . serialize($field_values));
      if (!isset($field_values[$old_key])) {

        // In case there is no main prop it is typically value.
        if (isset($old_value['main_property'])) {
          $mainprop = $old_value['main_property'];
        }
        else {
          $mainprop = "value";

          // Check if the field is an entity reference in the field definitions.
          // In this case mainprop is the target_id.
          /**  @var \Drupal\field\Entity\FieldConfig[] */
          $field_definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions('wisski_individual', $bundle_id);
          // Sanity check that the field really exists in the definitions.
          if (in_array($old_key, array_keys($field_definitions))) {
            $field_definition = $field_definitions[$old_key];
            $fieldType = $field_definition->getType();

            if ($fieldType == "entity_reference") {
              $mainprop = "target_id";
            }
          }
        }

        foreach ($old_value as $key => $val) {

          // Main prop?
          if (!is_array($val)) {
            continue;
          }

          // Empty value?
          if (empty($val[$mainprop])) {
            continue;
          }

          // If not its a value...
          // dpm("I delete from " . $entity_id . " field " . $old_key . " value " . $val[$mainprop] . " key " . $key);.
          $this->deleteOldFieldValue($entity_id, $old_key, $val[$mainprop], $pathbuilder, $key, $mainprop, $this_language);
        }
      }
    }
    // \Drupal::logger('WissKI Import tmpc')->debug("df:".(microtime(TRUE)-$tmpt));
    // dpm($old_values, "old values");.
    // Combined go through the new fields.
    foreach ($field_values as $field_id => $field_items) {

      // By Mark:
      // If this field is not enabled in the edit form
      // then we probably dont want to write any data to it
      // --!! THIS IS A GUESS !!--
      // example is posse: viewfield for coordinates in
      // places sub form without grouping -> it is for display
      // purpose only. If we enable this here regularly data
      // gets stored that should not be stored.
      if (!empty($components) && !isset($components[$field_id])) {
        continue;
      }

      // dpm("I try to add data to field $field_id with items: " . serialize($field_items));
      $path = $pathbuilder->getPbEntriesForFid($field_id);
      // drupal_set_message("found path: " . serialize($path). " " . microtime());
      // dpm(LanguageInterface::LANGCODE_DEFAULT, "lang?");.
      $old_language = $language;

      $old_value = [];
      if (isset($old_values[$field_id][$language])) {
        $old_value = $old_values[$field_id][$language];
        $old_language = $language;
      }
      else {
        if (isset($old_values[$field_id][LanguageInterface::LANGCODE_DEFAULT])) {
          $old_value = $old_values[$field_id][LanguageInterface::LANGCODE_DEFAULT];
          $old_language = LanguageInterface::LANGCODE_DEFAULT;
        }
      }

      // dpm($old_values[$field_id][LanguageInterface::LANGCODE_DEFAULT], "ould be?");.
      if (empty($path)) {
        // drupal_set_message("I leave here: $field_id " . microtime());
        continue;
      }

      // drupal_set_message("I am still here: $field_id");.
      $mainprop = $field_items['main_property'];

      unset($field_items['main_property']);

      $write_values = $field_items;

      // dpm("write values still is: " . serialize($write_values));
      // @todo $val is not set: iterate over fieldvalue!
      // if there are old values.
      if (!empty($old_value)) {
        // We might want to delete some.
        $delete_values = $old_value;

        // drupal_set_message("del: " . serialize($delete_values));
        // If it is not an array there are no values, so we can savely stop.
        if (!is_array($old_value)) {
          $delete_values = [$mainprop => $old_value];
          // $old_value contains the value directly
          foreach ($field_items as $key => $new_item) {
            // Empty field item due to cardinality, see else branch.
            if (empty($new_item)) {
              unset($write_values[$key]);
              continue;
            }
            // dpm("old value is: " . serialize($old_value) . " new is: " . $new_item[$mainprop]);
            // If the old value is somwhere in the new item.
            if ($old_value == $new_item[$mainprop]) {
              // We unset the write value at this key because this doesn't have to be written.
              unset($write_values[$key]);
              // We reset the things we need to delete.
              $delete_values = [];
            }
          }
        }
        else {
          // $old_value is an array of arrays resembling field list items and
          // containing field property => value pairs
          // dpm($old_value, "old value is:");
          // dpm($write_values, "new are?");
          foreach ($old_value as $old_key => $old_item) {

            if (!is_array($old_item) || empty($old_item)) {
              // This may be the case if
              // - it contains key "main_property"... (not an array)
              // - it is an empty field that is there because of the
              // field's cardinality (empty)
              unset($delete_values[$old_key]);
              continue;
            }
            // Also if it does not have our language, we can simply skip it.
            // MyF: here we also have to check whether a language is set; because in case of entity references there exist no wisski_language.
            // In such cases, we may not do an unset because otherwise in case of changes (i.e. deleting a targeting entity reference), the old
            // values would not be deleted properly in the code further down (Mostly fixed for the usage of the entity reference tree widget)
            if (isset($old_item['wisski_language']) && $old_item['wisski_language'] != $language) {
              unset($delete_values[$old_key]);
              continue;
            }

            $maincont = FALSE;

            foreach ($write_values as $key => $new_item) {
              if (empty($new_item)) {
                unset($write_values[$key]);
                // Empty field item due to cardinality.
                continue;
              }
              if ($old_item[$mainprop] == $new_item[$mainprop]) {
                // If we find the item in the old values we don't have to write it.
                unset($write_values[$key]);
                // And we don't have to delete it.
                unset($delete_values[$old_key]);

                $maincont = TRUE;
                break;
              }
            }

            // If we found something we continue in the old values.
            if ($maincont) {
              continue;
            }
          }
        }

        // dpm($delete_values, "we have to delete");.
        if (!empty($delete_values)) {
          foreach ($delete_values as $key => $val) {
            // dpm("I1 delete from " . $entity_id . " field " . $old_key . " value " . $val[$mainprop] . " key " . $key);.
            $this->deleteOldFieldValue($entity_id, $field_id, $val[$mainprop], $pathbuilder, $key, $mainprop, $old_language);
          }
        }
      }

      // dpm($write_values, "we have to write");
      // Now we write all the new values.
      // @todo it seems like there is a duplicate write in case of image files..
      // probably due to the fact that they are not found as old value because the URL is stored.
      // it does not hurt currently, but it is a performance sink.
      foreach ($write_values as $new_item) {
        // dpm($mainprop, "mainprop");
        // dpm($language, "I give it to add New Field Value");.
        $this->addNewFieldValue($entity_id, $field_id, $new_item[$mainprop], $pathbuilder, $mainprop, $language);
      }

    }
    // \Drupal::logger('WissKI Import tmpc')->debug("da:".(microtime(TRUE)-$tmpt));
    // drupal_set_message("out: " . serialize($out));
    // dpm(serialize($out), "out?");.
    return $out;

  }

  /**
   * -------------------------------- Ontologie thingies ----------------------
   */
  public function addOntologies($iri = NULL) {

    if (empty($iri)) {
      // Load all ontologies.
      $query = "SELECT ?ont WHERE { GRAPH ?g { ?ont a owl:Ontology} }";
      $result = $this->directQuery($query);
      foreach ($result as $obj) {
        $this->addOntologies(strval($obj->ont));
      }
      return;
    }

    // Check if the Ontology is already there.
    $result = $this->directQuery("ASK { GRAPH ?g { <$iri> a owl:Ontology } }");

    // If we get here we may load the ontology.
    $query = "LOAD <$iri> INTO GRAPH <$iri>";
    // dpm($query, "query");.
    $result = $this->directUpdate($query);

    $this->messenger()
      ->addStatus("Successfully loaded $iri into the Triplestore.");
    \Drupal::logger('WissKI Ontology')->info(
      'Adapter {a}: Successfully loaded ontology <{iri}>.',
      [
        'a' => $this->adapterId(),
        'iri' => $iri,
      ]
    );

    // Look for imported ontologies.
    $query = "SELECT DISTINCT ?ont FROM <$iri> WHERE { ?s a owl:Ontology . ?s owl:imports ?ont . }";
    $results = $this->directQuery($query);

    foreach ($results as $to_load) {
      $this->addOntologies(strval($to_load->ont));
    }

    // Add namespaces to table.
    // @todo curently all namespaces from all adapters are stored in a single
    // table and adapters may override existing ones from themselves or from
    // other adapters!
    // We add the namespaces AFTER we loaded the imported ontologies so that
    // the importing ontology's namespaces win over the ones in the imported
    // ontologies.
    [$default, $namespaces] = $this->getNamespacesFromDocument($iri);
    if (!empty($namespaces)) {
      foreach ($namespaces as $key => $value) {
        $this->putNamespace($key, $value);
      }
      \Drupal::logger('WissKI Ontology')->info(
        'Adapter {a}: registered the following namespaces and prefixes: {n} Default is {b}',
        [
          'a' => $this->adapterId(),
          'n' => array_reduce(array_keys($namespaces), function ($carry, $k) use ($namespaces) {
            return "$carry$k: <$namespaces[$k]>. ";
          }, ''),
          'b' => empty($base) ? 'not set' : "<$base>",
        ]
      );
      // @todo default is currently not stored. Do we need to store it?
      // it is no proper namespace prefix.
      // @todo check if it is already in the ontology.
      // @todo why declare these here? we never use them!
      // global $base_url;
      // $this->putNamespace("local", $base_url . '/');
      // $this->putNamespace("data", $base_url . '/inst/');
    }
    else {
      \Drupal::logger('WissKI Ontology')
        ->info('Adapter {a}: no namespaces registered', ['a' => $this->adapterId()]);
    }

    // Return the result of the loading of this ontology.
    return $result;

  }

  /**
   * Tries to parse RDF namespace declarations in a given document.
   *
   * @param iri the IRI of the document
   *
   * @return an array where the first element is the default prefix URI and the
   *   second is an array of prefix-namespace pairs. If the document
   *         cannot be parsed, an array(FALSE, FALSE) is returned.
   */
  public function getNamespacesFromDocument($iri) {
    $file = file_get_contents($iri);
    $format = EasyRdf_Format::guessFormat($file, $iri);
    // Unfortunately EasyRdf does not provide any API to get the declared
    // namespaces although in general its Parsers do handle them.
    // Therefore we have to provide our own namespace parsers.
    // atm, we only supprt rdfxml.
    if (stripos($format->getName(), 'xml') !== FALSE) {
      // This is a quick and dirty parse of an rdfxml file.
      // search for xmlns:xyz pattern inside the rdf:RDF tag.
      if (preg_match('/<(?:\w+:)?RDF[^>]*>/i', $file, $rdf_tag)) {
        preg_match_all('/xmlns[^=]*=(?:"[^"]*"|\'[^\']*\')/i', $rdf_tag[0], $nsarray);
        // Go thru the matches and collect the default and prefix-namespace pairs.
        $namespaces = [];
        $default = NULL;
        foreach ($nsarray[0] as $ns_decl) {
          [
            $front,
            $back,
          // Remove the leading xmlns.
          ] = explode("=", substr($ns_decl, 5));
          $front = trim($front);
          // Chop the "/'.
          $value = substr($back, 1, -1);
          if (empty($front)) {
            // If front is empty it is the default namespace.
            $default = $value;
          }
          elseif ($front[0] == ':') {
            // A named prefix must start with a :
            // Chop the leading :
            $prefix = substr($front, 1);
            $namespaces[$prefix] = $value;
          }
          // else:
          // it's not a namespace declaration but an attribute that starts
          // with xmlns. we must ignore it.
        }
        return [$default, $namespaces];
      }
      // else: if we cannot match an RDF tag it's no / an unknown RDF document.
    }
    // No parser available for this file format.
    return [FALSE, FALSE];
  }

  /**
   *
   */
  public function getOntologies($graph = NULL) {
    // Get ontology and version uri.
    if (!empty($graph)) {
      $query = "SELECT DISTINCT ?ont ?iri ?ver FROM $graph WHERE {\n"
        . "  ?ont a owl:Ontology .\n"
        . "  OPTIONAL { ?ont owl:ontologyIRI ?iri. } OPTIONAL { ?ont owl:versionIRI ?ver . }\n"
        . "}";
    }
    else {
      $query = "SELECT DISTINCT ?ont (COALESCE(?niri, 'none') as ?iri) (COALESCE(?nver, 'none') as ?ver) ?graph WHERE {\n"
        . "  GRAPH ?graph { ?ont a owl:Ontology } .\n"
        . "  OPTIONAL { ?ont owl:ontologyIRI ?niri. } OPTIONAL { ?ont owl:versionIRI ?nver . }\n"
        . "}";
    }
    $results = $this->directQuery($query);
    return $results;
  }

  /**
   *
   */
  public function deleteOntology($graph, $type = "graph") {

    // Get ontology and version uri.
    if ($type == "graph") {
      $query = "WITH <$graph> DELETE { ?s ?p ?o } WHERE { ?s ?p ?o }";
    }
    else {
      $query = "DELETE { ?s ?p ?o } WHERE { ?s ?p ?o . FILTER ( STRSTARTS(STR(?s), '$graph')) }";
    }

    $results = $this->directUpdate($query);

    /* if (!$ok) {
    // some useful error message :P~
    drupal_set_message('some error encountered:' . serialize($results), 'error');
    }
     */
    return $results;
  }

  /**
   * Insert a namespace mapping into the database.
   * If the mapping already exists, attempt to be smart by renaming existing
   * abbreviations.
   *
   * @param string $short_name
   *   The prefix abbreviation.
   * @param string $long_name
   *   The namespace IRI.
   *
   * @return void
   */
  private function putNamespace($short_name, $long_name) {

    // First check if a mapping for short_name already exists.
    // consider three cases (3. with two sub cases):
    // 1. no mapping exists => insert the new one
    // 2. exactly one mapping exists in the database
    //    2a. mapping exists AND has an identical long_name => do nothing
    //   2b. mapping exists AND has a different long_name => "smart" rename of the short_name
    // 3. more than one mapping exists in the database (SHOULD NOT HAPPEN, but MIGHT)
    //    treat the same way as 2b (shouldn't happen!)
    $result = \Drupal::database()
      ->select('wisski_core_ontology_namespaces', 'ns')
      ->fields('ns', ['short_name', 'long_name'])
      ->condition('ns.short_name', $short_name, '=')
      ->execute()
      ->fetchAll();

    // Case 1: no mapping exists => insert a new one!
    if (empty($result)) {
      \Drupal::database()->insert('wisski_core_ontology_namespaces')
        ->fields(['short_name' => $short_name, 'long_name' => $long_name])
        ->execute();
      return;
    }

    // Case 2a: Exactly one mapping exists AND has an identical long_name.$_COOKIE
    // For example:
    //
    // Database contains ecrm:http://erlangen-crm.org/200717/
    // $this->putNamespace("ecrm", "http://erlangen-crm.org/200717/");
    if (count($result) == 1 && ($result[0]->long_name == $long_name)) {
      return;
    }

    if (count($result) > 1) {
      // @todo tell the user about case 3, but don't do anything
      // Info "Danger Zone: More than one instance of $short_name detected in namespace list".
    }

    // Case 2b: smart rename
    // For example:
    //
    // Database contains ecrm:http://erlangen-crm.org/200717/
    // $this->putNamespace("ecrm", "http://erlangen-crm.org/211015/");
    // Search the database for all names that start with "{$short_name}_"
    // then generate a new name that is different from the existing ones.
    $short_name_prefix = str_replace(['\\', '_', '%'], [
      '\\\\',
      '\\_',
      '\\%',
    ], $short_name);
    $results = \Drupal::database()
      ->select('wisski_core_ontology_namespaces', 'ns')
      ->condition('ns.short_name', $short_name_prefix . '%', 'LIKE')
      ->fields('ns', ['short_name', 'long_name'])
      ->orderBy('ns.long_name', 'DESC')
      ->execute()
      ->fetchAll();

    $results = json_decode(json_encode($results), TRUE);

    // Get all long names and sort only by them
    // smaller numbers will be at the beginning of the array.
    $all_longnames = [];
    foreach ($results as $tmp) {
      $all_longnames[] = $tmp['long_name'];
    }
    $all_longnames[] = $long_name;

    sort($all_longnames, SORT_STRING);

    // Delete all entries from the data table which refer to the long names in
    // $all_longnames (= sorted long name array)
    // and insert them again with the appropriate name+index
    // the first entry in the sorted array gets the short name without an index
    // the others are enumerated beginning with 1.
    $counter = 0;
    $new_name = $short_name;
    foreach ($all_longnames as $tmp_longname) {
      \Drupal::database()->delete('wisski_core_ontology_namespaces')
        ->condition('long_name', $tmp_longname)
        ->execute();
      if ($counter > 0) {
        $new_name = $short_name . "_" . $counter;
      }
      \Drupal::database()->insert('wisski_core_ontology_namespaces')
        ->fields(['short_name' => $new_name, 'long_name' => $tmp_longname])
        ->execute();
      $counter++;
    }
  }

  /**
   * This should be made global as it actually stores the namespaces globally.
   */
  public function getNamespaces() {
    $ns = [];
    // @todo Drupal Rector Notice: Please delete the following comment after you've made any necessary changes.
    // You will need to use `\Drupal\core\Database\Database::getConnection()` if you do not yet have access to the container here.
    $db_spaces = \Drupal::database()
      ->select('wisski_core_ontology_namespaces', 'ns')
      ->fields('ns')
      ->execute()
      ->fetchAllAssoc('short_name');
    foreach ($db_spaces as $space) {
      $ns[$space->short_name] = $space->long_name;
    }
    return $ns;
  }

  private $super_properties = [];

  private $clean_super_properties = [];

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    // $cids = array(
    // 'properties',
    // 'sub_properties',
    // 'super_properties',
    // 'inverse_properties',
    // 'sub_classes',
    // 'super_classes',
    // 'domains',
    // 'reverse_domains',
    // 'ranges',
    // 'reverse_ranges',
    // );
    // $results = array();
    // foreach ($cids as $cid) {
    // if ($cache = \Drupal::cache()->get('wisski_reasoner_'.$cid)) {
    // $results[$cid] = $cache->data;
    // }
    // }
    // dpm($results,'Results');
    $in_cache = $this->isCacheSet();

    $form = parent::buildConfigurationForm($form, $form_state);

    $button_label = $this->t('Start Reasoning');
    $emphasized = $this->t('This will take several minutes.');

    $always_reason = \Drupal::state()->get('wisski_always_reason');

    if (isset($always_reason[$this->adapterId()])) {
      $always_reason = $always_reason[$this->adapterId()];
    }
    else {
      $always_reason = TRUE;
    }

    $form['properties']['allow_inverse_property_pattern'] = [
      '#type' => 'checkbox',
      '#title' => 'Inverse property selection',
      '#default_value' => $this->allow_inverse_property_pattern,
      '#return_value' => TRUE,
      '#description' => 'Allows selecting properties in inverse direction in pathbuilder. These properties are marked with a leading "^". E.g. for "^ex:prop1", the triple x2 ex:prop x1 must hold instead of x1 ex:prop1 x2.',
    ];

    $form['reasoner'] = [
      '#type' => 'details',
      '#title' => $this->t('Compute Type and Property Hierarchy and Domains and Ranges'),
      '#prefix' => '<div id="wisski-reasoner-block">',
      '#suffix' => '</div>',
      'description' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Read carefully'),
        'description_start' => ['#markup' => $this->t("Clicking the %label button will initiate a set of complex SPARQL queries computing", ['%label' => $button_label])],
        'description_list' => [
          '#theme' => 'item_list',
          '#items' => [
            $this->t("the class hierarchy"),
            $this->t("the property hierarchy"),
            $this->t("the domains of all properties"),
            $this->t("the ranges of all properties"),
          ],
        ],
        'description_end' => [
          '#markup' => $this->t(
            "in the specified triple store. <strong>%placeholder</strong> The pathbuilders relying on this adapter will become much faster by doing this.",
            ['%placeholder' => $emphasized]
          ),
        ],
      ],
      'start_button' => [
        '#type' => 'button',
        '#value' => $button_label,
        '#ajax' => [
          'wrapper' => 'wisski-reasoner-block',
          'callback' => [$this, 'startReasoning'],
        ],
        '#prefix' => '<div id="wisski-reasoner-start-button">',
        '#suffix' => '</div>',
      ],
      'delete_reasoning_button' => [
        '#type' => 'button',
        '#value' => $this->t('Delete reasoning tables'),
        '#ajax' => [
          'wrapper' => 'wisski-reasoner-block',
          'callback' => [$this, 'deleteReasoning'],
        ],
        '#prefix' => '<div id="wisski-reasoner-delete-button">',
        '#suffix' => '</div>',
      ],
      'always_reason_this_store' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Always do reasoning on this adapter.'),
        '#default_value' => $always_reason,
      ],
    ];
    if ($in_cache) {
      $form['reasoner']['start_button']['#disabled'] = !$form_state->getValue('flush_button');
      $form['reasoner']['flush_button'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Re-Compute results'),
        '#default_value' => FALSE,
        '#description' => $this->t('You already have reasoning results in your cache'),
        '#ajax' => [
          'wrapper' => 'wisski-reasoner-start-button',
          'callback' => [$this, 'checkboxAjax'],
        ],
      ];
      $classes_n_properties = ((array) $this->getClasses()) + ((array) $this->getProperties());
      if ($this->getClasses() === FALSE || $this->getProperties() === FALSE) {
        $this->messenger()
          ->addStatus($this->t('Bad class and property cache.'));
      }
      $form['reasoner']['tester'] = [
        '#type' => 'details',
        '#title' => $this->t('Check reasoning results'),
        'selected_prop' => [
          '#type' => 'select',
          '#options' => $classes_n_properties,
          '#empty_value' => 'empty',
          '#empty_option' => $this->t('select a class or property'),
          '#ajax' => [
            'wrapper' => 'wisski-reasoner-check',
            'callback' => [$this, 'checkTheReasoner'],
          ],
        ],
        'check_results' => [
          '#type' => 'textarea',
          '#prefix' => '<div id="wisski-reasoner-check">',
          '#suffix' => '</div>',
        ],
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $val = $form_state->getValue('always_reason_this_store');

    $always_reason = \Drupal::state()->get('wisski_always_reason');
    $always_reason[$this->adapterId()] = $val;

    \Drupal::state()->set('wisski_always_reason', $always_reason);

    $this->allow_inverse_property_pattern = $form_state->getValue('allow_inverse_property_pattern');
  }

  /**
   *
   */
  public function checkboxAjax(array $form, FormStateInterface $form_state) {
    return $form['reasoner']['start_button'];
  }

  /**
   *
   */
  public function checkTheReasoner(array $form, FormStateInterface $form_state) {

    $candidate = $form_state->getValue($form_state->getTriggeringElement()['#name']);
    if ($this->isAProperty($candidate)) {
      $stored = $this->getClassesFromStore($candidate);
      $cached = $this->getClassesFromCache($candidate);
    }
    else {
      $stored = $this->getPropertiesFromStore($candidate);
      $cached = $this->getPropertiesFromCache($candidate);
    }
    $more_stored = array_diff($stored, $cached);
    $more_cached = array_diff($cached, $stored);
    if (empty($more_stored) && empty($more_cached)) {
      $result = $this->t('Same results for cache and direct query');
      $full_results = $stored;
    }
    else {
      $stored_text = empty($more_stored) ? '' : $this->t('more in store:') . "\n\t" . implode("\n\t", $more_stored);
      $cached_text = empty($more_cached) ? '' : $this->t('more in cache:') . "\n\t" . implode("\n\t", $more_cached);
      $result = $this->t('Different results:') . "\n" . $stored_text . "\n" . $cached_text;
      $full_results = array_unique(array_merge($stored, $cached));
    }
    $form['reasoner']['tester']['check_results']['#value'] = $candidate . "\n" . $result . "\n\n" . $this->t('Full list of results') . "\n\t" . implode("\n\t", $full_results);
    return $form['reasoner']['tester']['check_results'];
  }

  /**
   *
   */
  public function deleteReasoning(array $form, FormStateInterface $form_state) {
    $this->prepareTables(TRUE);
    $this->messenger()->addMessage("Reasoning-Tables are reset.");
    $form_state->setRedirect('<current>');
    return $form['reasoner'];
  }

  /**
   *
   */
  public function startReasoning(array $form, FormStateInterface $form_state) {

    $this->doTheReasoning();
    $form_state->setRedirect('<current>');
    return $form['reasoner'];
  }

  /**
   *
   */
  public function doTheReasoning() {

    $properties = [];
    $super_properties = [];
    $sub_properties = [];

    // Prepare database connection and reasoner tables
    // if there's something wrong stop working.
    if ($this->prepareTables() === FALSE) {
      return;
    }

    // Find properties.
    $result = $this->directQuery("SELECT ?property WHERE { GRAPH ?g {{?property a owl:ObjectProperty.} UNION {?property a rdf:Property}} }");

    $insert = $this->prepareInsert('properties');
    foreach ($result as $row) {
      $prop = $row->property->getUri();
      $properties[$prop] = $prop;
      $insert->values(['property' => $prop]);
    }
    $insert->execute();
    // $cid = 'wisski_reasoner_properties';
    // \Drupal::cache()->set($cid,$properties);
    // find one step property hierarchy, i.e. properties that are direct children or direct parents to each other
    // no sub-generations are gathered.
    $result = $this->directQuery(
    // We don't do the graph thing in reasoner-queries because graphdb 9.10 does not support this
    // anymore :(
    //      "SELECT ?property ?super WHERE { GRAPH ?g {".
      "SELECT ?property ?super WHERE { {"
      . "{{?property a owl:ObjectProperty.} UNION {?property a rdf:Property}}} . { "
      . "?property rdfs:subPropertyOf ?super.  "
      . "FILTER NOT EXISTS {?mid_property rdfs:subPropertyOf+ ?super. ?property rdfs:subPropertyOf ?mid_property.}"
      . "} } ");
    foreach ($result as $row) {
      $prop = $row->property->getUri();
      $super = $row->super->getUri();
      $super_properties[$prop][$super] = $super;
      $sub_properties[$super][$prop] = $prop;
      if (!isset($properties[$prop])) {
        $properties[$prop] = $prop;
      }
    }

    // $cid = 'wisski_reasoner_sub_properties';
    // \Drupal::cache()->set($cid,$sub_properties);
    // $cid = 'wisski_reasoner_super_properties';
    // \Drupal::cache()->set($cid,$super_properties);
    // now lets find inverses
    $insert = $this->prepareInsert('inverses');
    $inverses = [];

    // $results = $this->directQuery("SELECT ?prop ?inverse WHERE { GRAPH ?g {{?prop owl:inverseOf ?inverse.} UNION {?inverse owl:inverseOf ?prop.}} }");
    $results = $this->directQuery("SELECT ?prop ?inverse WHERE { {{?prop owl:inverseOf ?inverse.} UNION {?inverse owl:inverseOf ?prop.}} }");
    foreach ($results as $row) {
      $prop = $row->prop->getUri();
      $inv = $row->inverse->getUri();
      $inverses[$prop] = $inv;
      $insert->values(['property' => $prop, 'inverse' => $inv]);
    }
    $insert->execute();
    // $cid = 'wisski_reasoner_inverse_properties';
    // \Drupal::cache()->set($cid,$inverses);
    // now the same things for classes
    // find all classes
    $insert = $this->prepareInsert('classes');
    $classes = [];

    $results = $this->directQuery("SELECT ?class WHERE { {{?class a owl:Class.} UNION {?class a rdfs:Class}} FILTER (!isBlank(?class)) }");
    // $results = $this->directQuery("SELECT ?class WHERE { GRAPH ?g {{?class a owl:Class.} UNION {?class a rdfs:Class}} }");
    foreach ($results as $row) {
      $class = $row->class->getUri();
      $classes[$class] = $class;
      $insert->values(['class' => $class]);
    }
    $insert->execute();
    // uksort($classes,'strnatcasecmp');
    // \Drupal::cache()->set('wisski_reasoner_classes',$classes);
    // find full class hierarchy.
    $super_classes = [];
    $sub_classes = [];

    $results = $this->directQuery("SELECT ?class ?super WHERE { {"
      // $results = $this->directQuery("SELECT ?class ?super WHERE { GRAPH ?g {"
      . "?class rdfs:subClassOf+ ?super. "
      . "FILTER (!isBlank(?class)) "
      // ."FILTER (!isBlank(?super)) } . GRAPH ?g1 {"
      . "FILTER (!isBlank(?super)) } . {"
      . "{{?super a owl:Class.} UNION {?super a rdfs:Class.}} "
      . "} }");
    foreach ($results as $row) {
      $sub = $row->class->getUri();
      $super = $row->super->getUri();
      $super_classes[$sub][$super] = $super;
      $sub_classes[$super][$sub] = $sub;
    }

    // \Drupal::cache()->set('wisski_reasoner_sub_classes',$sub_classes);
    // \Drupal::cache()->set('wisski_reasoner_super_classes',$super_classes);
    // explicit top level domains
    $domains = [];

    $results = $this->directQuery(
      "SELECT ?property ?domain WHERE { {"
      // "SELECT ?property ?domain WHERE { GRAPH ?g {"
      . " { "
      . " ?property rdfs:domain ?domain . } "
      . " UNION {"
      . "   ?property rdfs:domain ?metaClass . "
      . "   ?metaClass owl:unionOf ?collection . "
      . "   ?collection rdf:rest*/rdf:first ?domain . "
      . " }"
      // We only need top level domains, so no proper subClass of the domain shall be taken into account.
      . " FILTER (NOT EXISTS { "
      . "   ?domain rdfs:subClassOf+ ?super_domain . "
      . "   ?property rdfs:domain ?super_domain . }  && "
      . "   isIRI(?domain)) "
      . " } }");
    foreach ($results as $row) {
      $domains[$row->property->getUri()][$row->domain->getUri()] = $row->domain->getUri();
    }

    // Clear up, avoid DatatypeProperties.
    $domains = array_intersect_key($domains, $properties);

    // Explicit top level ranges.
    $ranges = [];

    $results = $this->directQuery(
      "SELECT ?property ?range WHERE { { "
      // "SELECT ?property ?range WHERE { GRAPH ?g {"
      . " { "
      . " ?property rdfs:range ?range . } "
      . " UNION {"
      . "   ?property rdfs:range ?metaClass . "
      . "   ?metaClass owl:unionOf ?collection . "
      . "   ?collection rdf:rest*/rdf:first ?range . "
      . " } "
      // We only need top level ranges, so no proper subClass of the range shall be taken into account.
      . " FILTER (NOT EXISTS { "
      . "   ?range rdfs:subClassOf+ ?super_range . "
      . "   ?property rdfs:range ?super_range .}  && "
      . "   isIRI(?range)) "
      . " } } ");
    foreach ($results as $row) {
      $ranges[$row->property->getUri()][$row->range->getUri()] = $row->range->getUri();
    }

    // Clear up, avoid DatatypeProperties.
    $ranges = array_intersect_key($ranges, $properties);

    // Explicit top level datatypes.
    $primitives = [];

    $results = $this->directQuery(
      "SELECT ?property ?domain WHERE { { "
      // "SELECT ?property ?domain WHERE { GRAPH ?g {"
      . " { ?property rdfs:domain ?domain . } "
      . " UNION { "
      . "   ?property rdfs:domain ?metaClass . "
      . "   ?metaClass owl:unionOf ?collection . "
      . "   ?collection rdf:rest*/rdf:first ?domain . "
      . " } . "
      . " {{ ?property a owl:DatatypeProperty.} UNION {?property a rdf:Property }} . "
      // We only need top level domains, so no proper subClass of the domain shall be taken into account.
      . " FILTER (NOT EXISTS { ?domain rdfs:subClassOf+ ?super_domain. ?property rdfs:domain ?super_domain.} && "
      . " isIRI(?domain)) "
      . " } } ");
    foreach ($results as $row) {
      $primitives[$row->property->getUri()][$row->domain->getUri()] = $row->domain->getUri();
    }

    // Clear up, avoid DatatypeProperties
    // $domains = array_intersect_key($domains,$properties);.
    // Take all properties with no super property.
    $top_properties = array_diff_key($properties, $super_properties);

    $invalid_definitions = [];
    // Check if they all have domains and ranges set.
    $dom_check = array_diff_key($top_properties, $domains);
    if (!empty($dom_check)) {
      $this->messenger()
        ->addError('No domains for top-level properties: ' . implode(', ', $dom_check));
      ;
      $invalid_definitions = array_merge($invalid_definitions, $dom_check);
      // $valid_definitions = FALSE;
      // foreach($dom_check as $dom) {
      //  $domains[$dom] = ['TOPCLASS'=>'TOPCLASS'];
      // }
    }
    $rng_check = array_diff_key($top_properties, $ranges);
    if (!empty($rng_check)) {
      $this->messenger()
        ->addError('No ranges for top-level properties: ' . implode(', ', $rng_check));
      $invalid_definitions = array_merge($invalid_definitions, $rng_check);
      // $valid_definitions = FALSE;
      // foreach ($rng_check as $rng) {
      //  $ranges[$rng] = ['TOPCLASS'=>'TOPCLASS'];
      // }
    }

    // $prim_check = array_diff_key($top_properties,$primitives);
    // if (!empty($prim_check)) {
    // $this->messenger()->addError('No Domain for top-level primitive datatype property: '.implode(', ',$prim_check));;
    // $invalid_definitions = array_merge($invalid_definitions, $prim_check);
    // $valid_definitions = FALSE;
    // //foreach($dom_check as $dom) {
    // //  $domains[$dom] = ['TOPCLASS'=>'TOPCLASS'];
    // //}
    // }
    // Set of properties where the domains and ranges are not fully set
    $not_set = array_diff_key($properties, $top_properties);
    $not_set = array_diff_key($not_set, $invalid_definitions);
    // dpm($invalid_definitions, "invalid!");
    // dpm($not_set, "not set");.
    // While there are unchecked properties cycle throgh them, gather domain/range defs from all super properties and inverses
    // and include them into own definition.
    $runs = 0;
    while (!empty($not_set)) {

      $runs++;
      // Take one of the properties.
      $prop = array_shift($not_set);
      // dpm($prop, "was not set");
      // check if all super_properties have their domains/ranges set.
      $supers = $super_properties[$prop];
      $invalid_supers = array_intersect($supers, $invalid_definitions);
      // $invalid_supers = $invalid_definitions;
      // dpm($supers, "for prop $prop, still to check: " . serialize($not_set));
      if (empty($invalid_supers)) {

        $to_check = array_intersect($supers, $not_set);
        if (!empty($to_check)) {
          array_push($not_set, $prop);
          continue;
        }

        // Take all the definitions of super properties and add them here.
        $new_domains = $domains[$prop] ?? [];
        $new_ranges = $ranges[$prop] ?? [];
        // dpm($domains);
        foreach ($supers as $super_prop) {
          if (isset($domains[$super_prop])) {
            $new_domains += $domains[$super_prop];
          }
          if (isset($ranges[$super_prop])) {
            $new_ranges += $ranges[$super_prop];
          }
        }
        $new_domains = array_unique($new_domains);
        $new_ranges = array_unique($new_ranges);

        $remove_domains = [];
        foreach ($new_domains as $domain_1) {
          foreach ($new_domains as $domain_2) {
            if ($domain_1 !== $domain_2) {
              if (isset($super_classes[$domain_1]) && in_array($domain_2, $super_classes[$domain_1])) {
                $remove_domains[] = $domain_2;
              }
            }
          }
        }
        $new_domains = array_diff($new_domains, $remove_domains);

        $domains[$prop] = array_combine($new_domains, $new_domains);

        $remove_ranges = [];
        foreach ($new_ranges as $range_1) {
          foreach ($new_ranges as $range_2) {
            if ($range_1 !== $range_2) {
              if (isset($super_classes[$range_1]) && in_array($range_2, $super_classes[$range_1])) {
                $remove_ranges[] = $range_2;
              }
            }
          }
        }
        $new_ranges = array_diff($new_ranges, $remove_ranges);

        $ranges[$prop] = array_combine($new_ranges, $new_ranges);

      }
      else {
        // Append this property to the end of the list to be checked again later-on
        // array_push($not_set,$prop);.
        $this->messenger()
          ->addError("I could not check $prop, because it has the invalid superproperties: " . implode(', ', $invalid_supers));
        continue;
      }
    }
    $this->messenger()->addStatus('Definition checkup runs: ' . $runs);
    // Remember sub classes of domains are domains, too.
    // if a property has exactly one domain set, we can add all subClasses of that domain
    // if there are multiple domains we can only add those being subClasses of ALL of the domains.
    foreach ($properties as $property) {
      if (isset($domains[$property])) {
        $add_up = [];
        foreach ($domains[$property] as $domain) {
          if (isset($sub_classes[$domain]) && $sub_domains = $sub_classes[$domain]) {
            $add_up = empty($add_up) ? $sub_domains : array_intersect_key($add_up, $sub_domains);
          }
        }
        $domains[$property] = array_merge($domains[$property], $add_up);
      }
      if (isset($ranges[$property])) {
        $add_up = [];
        foreach ($ranges[$property] as $range) {
          if (isset($sub_classes[$range]) && $sub_ranges = $sub_classes[$range]) {
            $add_up = empty($add_up) ? $sub_ranges : array_intersect_key($add_up, $sub_ranges);
          }
        }
        $ranges[$property] = array_merge($ranges[$property], $add_up);
      }
    }

    foreach ($primitives as $property => $values) {
      if (isset($primitives[$property])) {
        $add_up = [];
        foreach ($primitives[$property] as $domain) {
          if (isset($sub_classes[$domain]) && $sub_domains = $sub_classes[$domain]) {
            $add_up = empty($add_up) ? $sub_domains : array_intersect_key($add_up, $sub_domains);
          }
        }
        $primitives[$property] = array_merge($primitives[$property], $add_up);
      }
    }

    $insert = $this->prepareInsert('domains');
    foreach ($domains as $prop => $classes) {
      foreach ($classes as $class) {
        $insert->values(['property' => $prop, 'class' => $class]);
      }
    }
    $insert->execute();

    $insert = $this->prepareInsert('ranges');
    foreach ($ranges as $prop => $classes) {
      foreach ($classes as $class) {
        $insert->values(['property' => $prop, 'class' => $class]);
      }
    }
    $insert->execute();

    $insert = $this->prepareInsert('primitives');
    foreach ($primitives as $prop => $classes) {
      foreach ($classes as $class) {
        $insert->values(['property' => $prop, 'class' => $class]);
      }
    }
    $insert->execute();

    // //for the pathbuilders to work correctly, we also need inverted search
    //    $reverse_domains = array();
    //    foreach ($domains as $prop => $classes) {
    //      foreach ($classes as $class) $reverse_domains[$class][$prop] = $prop;
    //    }
    //    $reverse_ranges = array();
    //    foreach ($ranges as $prop => $classes) {
    //      foreach ($classes as $class) $reverse_ranges[$class][$prop] = $prop;
    //    }
    //    $cid = 'wisski_reasoner_domains';
    //    \Drupal::cache()->set($cid,$domains);
    //    $cid = 'wisski_reasoner_ranges';
    //    \Drupal::cache()->set($cid,$ranges);
    //    $cid = 'wisski_reasoner_reverse_domains';
    //    \Drupal::cache()->set($cid,$reverse_domains);
    //    $cid = 'wisski_reasoner_reverse_ranges';
    //    \Drupal::cache()->set($cid,$reverse_ranges);
  }

  /**
   *
   */
  public function getInverseProperty($property_uri) {

    /* cache version
    $inverses = array();
    $cid = 'wisski_reasoner_inverse_properties';
    if ($cache = \Drupal::cache()->get($cid)) {
    $inverses = $cache->data;
    if (isset($properties[$property_uri])) return $inverses[$property_uri];
    }
     */

    // DB version.
    $inverse = $this->retrieve('inverses', 'inverse', 'property', $property_uri);
    // dpm($inverse, "got from cache?");.
    if (!empty($inverse)) {
      return current($inverse);
    }

    // Up to now this was the current code. However this is evil in case there are several answers.
    // it will then return the upper one which is bad.
    // so in case there is an easy answer, give the easy answer.
    $results = $this->directQuery(
      "SELECT ?sub_inverse ?inverse WHERE {"
      . "{"
      . "{GRAPH ?g1 {?sub_inverse owl:inverseOf ?inverse.}}"
      . " UNION "
      . "{GRAPH ?g2 {?inverse owl:inverseOf ?sub_inverse.}}"
      . "}"
      // ."{GRAPH ?g3 {<$property_uri> rdfs:subPropertyOf* ?sub_inverse}}"
      . "{ {<$property_uri> rdfs:subPropertyOf* ?sub_inverse}}"
      . "}"
    );

    $inverse = '';
    foreach ($results as $row) {
      $inverse = $row->inverse->getUri();
      // If we had the requested property, we do not need to search for a sub...
      if ($row->sub_inverse->getUri() == $property_uri) {
        break;
      }
    }
    $inverses[$property_uri] = $inverse;
    // \Drupal::cache()->set($cid,$inverses);
    return $inverse;
  }

  /**
   *
   */
  protected function isPrepared() {
    try {
      $result = !empty(\Drupal::service('database')
        ->select($this->adapterId() . '_classes', 'c')
        ->fields('c')
        ->range(0, 1)
        ->execute());
      return $result;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   *
   */
  public function prepareTables($drop = FALSE) {

    try {
      $database = \Drupal::service('database');
      $schema = $database->schema();
      $adapter_id = $this->adapterId();
      foreach (self::getReasonerTableSchema() as $type => $table_schema) {
        $table_name = $adapter_id . '_' . $type;
        if ($schema->tableExists($table_name)) {
          if ($drop) {
            $database->schema()->dropTable($table_name);
          }
          else {
            $database->truncate($table_name)->execute();
          }
        }
        else {
          $schema->createTable($table_name, $table_schema);
        }
      }
      return TRUE;
    }
    catch (\Exception $ex) {
    }
    return FALSE;
  }

  /**
   *
   */
  private function prepareInsert($type) {

    $fieldS = [];
    foreach (self::getReasonerTableSchema()[$type]['fields'] as $field_name => $field) {
      if ($field['type'] !== 'serial') {
        $fields[] = $field_name;
      }
    }
    $table_name = $this->adapterId() . '_' . $type;
    return \Drupal::service('database')->insert($table_name)->fields($fields);
  }

  /**
   *
   */
  public function retrieve($type, $return_field = NULL, $condition_field = NULL, $condition_value = NULL) {

    $table_name = $this->adapterId() . '_' . $type;
    $query = \Drupal::service('database')
      ->select($table_name, 't')
      ->fields('t');
    if (!is_null($condition_field) && !is_null($condition_value)) {
      $query = $query->condition($condition_field, $condition_value);
    }
    try {
      $result = $query->execute();
      if (!is_null($return_field)) {
        $result = array_keys($result->fetchAllAssoc($return_field));
        usort($result, 'strnatcasecmp');
        return array_combine($result, $result);
      }
      return $result->fetchAll();
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Implements hook_schema()
   */
  public static function getReasonerTableSchema() {

    $schema['classes'] = [
      'description' => 'hold information about triple store classes',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this class',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'class' => [
          'description' => 'the uri of the class',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    $schema['properties'] = [
      'description' => 'hold information about triple store properties',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this property',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'property' => [
          'description' => 'the uri of the property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    $schema['domains'] = [
      'description' => 'hold information about domains of triple store properties',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this pairing',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'property' => [
          'description' => 'the uri of the property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
        'class' => [
          'description' => 'the uri of the domain class',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    $schema['primitives'] = [
      'description' => 'hold information about domains of primitive triple store properties',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this pairing',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'property' => [
          'description' => 'the uri of the property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
        'class' => [
          'description' => 'the uri of the domain class',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    $schema['ranges'] = [
      'description' => 'hold information about ranges of triple store properties',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this pairing',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'property' => [
          'description' => 'the uri of the property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
        'class' => [
          'description' => 'the uri of the range class',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    $schema['inverses'] = [
      'description' => 'hold information about ranges of triple store properties',
      'fields' => [
        'num' => [
          'description' => 'the Serial Number for this pairing',
          'type' => 'serial',
          'size' => 'normal',
          'not null' => TRUE,
        ],
        'property' => [
          'description' => 'the uri of the property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
        'inverse' => [
          'description' => 'the uri of the inverse property',
          'type' => 'varchar',
          'length' => '2048',
          'not null' => TRUE,
        ],
      ],
      'primary key' => ['num'],
    ];

    return $schema;
  }

}
