/**
 * @file
 * JavaScript behaviors for Authority Document autocomplete.
 */

(function ($, Drupal, once) {
  'use strict';

  /**
   * Attaches behavior for Authority Document autocomplete.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.authorityDocumentAutocomplete = {
    attach: function (context, settings) {
      once('authority-document-autocomplete', '.wisski-authority-autocomplete', context).forEach(function (element) {
        let $input = $(element);
        let authorityTypeFieldId = $input.data('authority-type-field');
        let minChars = $input.data('min-chars') || 3;
        let linkFieldId = $input.data('link-field');
        let entityTypeFieldId = $input.data('entity-type-field');
        let fieldId = $input.data('field-id');

        // Attach autocomplete behavior
        $input.autocomplete({
          minLength: minChars,
          source: function(request, response) {
            // Find the closest parent form that contains everything
            let $form = $input.closest('form');

            // Get the authority type value
            let authorityType = '';
            if (authorityTypeFieldId) {
              // Look within the form for radio buttons that have the authority type field ID in their data-drupal-selector
              let $radioButtons = $form.find('input[type="radio"][data-drupal-selector*="' + authorityTypeFieldId + '"]');

              let $checkedRadio = $radioButtons.filter(':checked');
              if ($checkedRadio.length) {
                authorityType = $checkedRadio.val();
              }
            }

            // Get entity restriction values from checkboxes
            let entityTypes = [];
            if (entityTypeFieldId) {
              // Look for checkboxes within the form that contain the entity type field ID
              let $checkboxes = $form.find('input[type="checkbox"][data-drupal-selector*="' + entityTypeFieldId + '"]:checked');

              $checkboxes.each(function() {
                entityTypes.push($(this).val());
              });
            }

            // Prepare request data
            let requestData = {
              q: request.term,
              authority_type: authorityType,
              field_id: fieldId
            };

            // Add entity types as individual parameters (entity_type=Tp&entity_type=Tu)
            if (entityTypes.length > 0) {
              requestData.entity_type = entityTypes.join(',');
            }

            // Make AJAX request to the autocomplete endpoint
            $.ajax({
              url: Drupal.url('wisski/authority-document/autocomplete'),
              dataType: 'json',
              data: requestData,
              success: function(data) {
                // Check data structure in detail
                console.log('Autocomplete response:', data);
                if (data === null || data === undefined) {
                  console.error('Autocomplete response is null or undefined');
                  response([]);
                  return;
                }

                if (!Array.isArray(data)) {
                  console.error('Autocomplete response is not an array:', typeof data);

                  // Try to convert to array if it's an object with properties
                  if (typeof data === 'object') {
                    const dataArray = Object.values(data);
                    response(dataArray);
                    return;
                  }

                  response([]);
                  return;
                }

                // Transform data to format expected by jQuery UI autocomplete if needed
                if (data.length > 0) {
                  // Check if items have required properties
                  const hasRequiredProps = data[0].hasOwnProperty('value') &&
                                          (data[0].hasOwnProperty('label') || data[0].value);

                  if (!hasRequiredProps) {
                    console.error('Response items missing required properties (value/label)');
                    // Try to transform the data
                    const transformed = data.map(item => ({
                      value: item.value || item.label || item.name || item.title || String(item),
                      label: item.label || item.value || item.name || item.title || String(item)
                    }));
                    response(transformed);
                  } else {
                    // No transformation needed if data is already in the right format
                    response(data);
                  }
                } else {
                  response([]);
                }
              },
              error: function(xhr, status, error) {
                console.error('Autocomplete error:', status, error);

                // Try to parse the response if possible
                try {
                  const errorData = JSON.parse(xhr.responseText);
                  console.error('Parsed error response:', errorData);
                } catch (e) {
                  console.error('Could not parse error response');
                }

                response([]);
              }
            });
          },
          select: function(event, ui) {
            // Decode HTML entities from the value
            function decodeHtmlEntities(str) {
              var textArea = document.createElement('textarea');
              textArea.innerHTML = str;
              return textArea.value;
            }

            // Update the input field with the decoded selected value
            $input.val(decodeHtmlEntities(ui.item.value));

            // If a link field is specified, update it with the URI
            if (linkFieldId && ui.item.uri) {
              // Find the closest parent form
              let $form = $input.closest('form');

              // Find link input within the form that contains the link field ID
              let $linkField = $form.find('input[data-drupal-selector*="' + linkFieldId + '"][data-drupal-selector*="uri"]');

              if ($linkField.length) {
                $linkField.val(ui.item.uri);
              }
            }

            return false;
          }
                }).autocomplete('instance')._renderItem = function(ul, item) {
          // Custom rendering of the autocomplete item to show the label with tooltip
          var tooltipText = '';

          //@todo: move to a separate function.
          // Decode HTML entities function
          function decodeHtmlEntities(str) {
            var textArea = document.createElement('textarea');
            textArea.innerHTML = str;
            return textArea.value;
          }

          // Build tooltip content from available item properties
          if (item.tooltip) {
            tooltipText = decodeHtmlEntities(item.tooltip);
          }

          // Decode the display text as well to handle special characters
          var displayText = decodeHtmlEntities(item.label || item.value);
          var $div = $('<div>').text(displayText);

          // Add tooltip if we have content for it
          if (tooltipText) {
            $div.attr('title', tooltipText);
          }

          return $('<li>')
            .append($div)
            .appendTo(ul);
        };
      });
    }
  };
})(jQuery, Drupal, once);
