<?php

namespace Drupal\wisski_doi\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Controller for DOI Settings.
 */
class WisskiDoiRepositorySettings extends ConfigFormBase {

  /**
   * The DOI settings storage.
   */
  const DOI_SETTINGS = 'wisski_doi.wisski_doi_settings';


  /**
   * The shoulder suffix delimiters.
   */
  const SHOULDER_SUFFIX_DELIMITERS = [
    '/' => '/',
    '-' => '-' ,
    '_' => '_',
    '#' => '#',
    '\\' => '\\',
    '.' => '.',
  ];

  /**
   * The selection for the "nameType" of the "creator" repository setting.
   */
  const NAME_TYPE_SELECTION = [
    'IGNORE' => 'Ignore',
    'ORGANIZATIONAL' => 'Organizational',
    'PERSONAL' => 'Personal',
  ];

  /**
   * The form id.
   */
  public function getFormId(): string {
    return 'wisski_doi_settings';
  }

  /**
   * Gets the configuration names that will be editable.
   *
   * @return array
   *   An array of configuration object names that are editable if called in
   *   conjunction with the trait's config() method.
   */
  protected function getEditableConfigNames() {
    return [
      static::DOI_SETTINGS,
    ];
  }

  /**
   * The form field validation.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // @todo Change the autogenerated stub.
    parent::validateForm($form, $form_state);
    $continue = TRUE;

    $onlyText = 'äöü ÄÖÜ 0-9 a-z-_ß.';
    $onlyUri = '%^(?:(?:https?|ftp)://)(?:\S+(?::\S*)?@|\d{1,3}(?:\.\d{1,3}){3}|(?:(?:[a-z\d\x{00a1}-\x{ffff}]+-?)*[a-z\d\x{00a1}-\x{ffff}]+)(?:\.(?:[a-z\d\x{00a1}-\x{ffff}]+-?)*[a-z\d\x{00a1}-\x{ffff}]+)*(?:\.[a-z\x{00a1}-\x{ffff}]{2,6}))(?::\d+)?(?:[^\s]*)?$%iu';
    $onlyFloats = '/^\d+[.]?\d*$/';

    if (preg_match('/[' . $onlyText . ']+/', $form_state->getValue('doi_provider')) !== 1) {
      $form_state->setErrorByName('doi_provider', $this->t('You have not allowed charakters in your field "Provider"!'));
      $continue = FALSE;
    }
    if (preg_match('/[' . $onlyText . ']+/', $form_state->getValue('data_publisher')) !== 1) {
      $form_state->setErrorByName('data_publisher', $this->t('You have not allowed charakters in your field "Data publisher"!'));
      $continue = FALSE;
    }
    if (preg_match('/[' . $onlyText . ']+/', $form_state->getValue('data_creator')) !== 1) {
      $form_state->setErrorByName('data_creator', $this->t('You have not allowed charakters in your field "Data creator"!'));
      $continue = FALSE;
    }

    if (preg_match($onlyUri, $form_state->getValue('doi_base_uri')) !== 1) {
      $form_state->setErrorByName('doi_base_uri', $this->t('Provide a valid URL, like "https://api.test.datacite.org/dois"!'));
      $continue = FALSE;
    }

    if (preg_match('/[' . $onlyText . ']+/', $form_state->getValue('doi_repository_id')) !== 1) {
      $form_state->setErrorByName('doi_repository_id', $this->t('You have not allowed charakters in your field "Provider"!'));
      $continue = FALSE;
    }

    if (preg_match($onlyFloats, $form_state->getValue('doi_schema_version')) !== 1) {
      $form_state->setErrorByName('doi_schema_version', $this->t('You have not allowed charakters in your field "Provider"!'));
      $continue = FALSE;
    }

    if (preg_match($onlyFloats, $form_state->getValue('doi_prefix')) !== 1) {
      $form_state->setErrorByName('doi_prefix', $this->t('You have not allowed charakters in your field "Provider"!'));
      $continue = FALSE;
    }

    $containsSpecialCharakters = preg_match('/\W\./', $form_state->getValue('doi_shoulder'));
    if ($containsSpecialCharakters) {
      $form_state->setErrorByName('doi_shoulder', $this->t('You have special charakters in your Field "Shoulder"!'));
      $continue = FALSE;
    }

    return $continue;
  }

  /**
   * The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form = [];
    $config = $this->config(static::DOI_SETTINGS);

    $form['#wisski_doi_settings'] = $config;

    $form['doi_provider'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Provider'),
      '#default_value' => $config->get('doiSettings.doi_provider'),
      '#description' => $this->t('The provider of your DOI repository, like "DataCite" or "CrossRef".'),
    ];

    $form['data_publisher'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Data publisher'),
      '#default_value' => $config->get('doiSettings.data_publisher'),
      '#description' => $this->t('The publisher of the data or the head person respectively institute, like "Germanic National Museum".'),
    ];

    $form['data_creator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Data creator'),
      '#default_value' => $config->get('doiSettings.data_creator'),
      '#description' => $this->t('The creator of the data or the project name, like "German Sales".'),
    ];

    $form['data_creator_name_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type of the creator of the data'),
      '#options' => static::NAME_TYPE_SELECTION,
      '#default_value' => $config->get('doiSettings.data_creator_name_type'),
      '#description' => $this->t('Can be either "Organizational" or "Personal" (or ignored; optional field).'),
    ];

    $form['doi_base_uri'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base URI'),
      '#default_value' => $config->get('doiSettings.doi_base_uri'),
      '#description' => $this->t('The Endpoint of your REST API, like "https://api.test.datacite.org/dois"'),
    ];

    $form['doi_repository_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Repository id'),
      '#default_value' => $config->get('doiSettings.doi_repository_id'),
      '#description' => $this->t('The place where you administer your DOIs, like "My DOI Repository".'),
    ];

    $form['doi_schema_version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Schema version'),
      '#default_value' => $config->get('doiSettings.doi_schema_version'),
      '#description' => $this->t('The version of DOI kernel, like "4.4".'),
    ];

    $form['doi_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#default_value' => $config->get('doiSettings.doi_prefix'),
      '#description' => $this->t('The DOI prefix, respectively your repository ID, like "10.3435".'),
    ];

    $form['doi_shoulder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Shoulder'),
      '#default_value' => $config->get('doiSettings.doi_shoulder'),
      '#description' => $this->t('The DOI shoulder, the part between prefix and suffix, i.e. "my_project".
      Allowed are only alphanumeric characters and underscores (_).'),
    ];

    $form['doi_shoulder_suffix_delimiter'] = [
      '#type' => 'select',
      '#title' => $this->t('Delimiter between shoulder and suffix'),
      '#options' => static::SHOULDER_SUFFIX_DELIMITERS,
      '#default_value' => $config->get('doiSettings.doi_shoulder_suffix_delimiter'),
      '#description' => $this->t('With a shoulder like "my_project", a delimiter like "-", and a suffix like "620661ed7d752"
      the DOI would be "https://doi.org/10.82102/my_project-620661ed7d752".'),
    ];

    $form['doi_repository_password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#default_value' => $config->get('doiSettings.doi_repository_password'),
      '#description' => $this->t('The password for your repository account.'),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * The submit action.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::DOI_SETTINGS)
      ->set('doiSettings', $form_state->getValues())
      ->save();

    $this->messenger()->addStatus($this->t('Changed DOI settings'));

    $form_state->setRedirect('system.admin_config');
  }

}
