<?php

declare(strict_types=1);

namespace Drupal\wisski_ddm_translations\Drush\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for WissKI DDM translations.
 */
final class WisskiDdmTranslationsCommands extends DrushCommands {

  /**
   * The active configuration storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  private StorageInterface $configStorage;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  private ModuleExtensionList $moduleExtensionList;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * Constructs the command object.
   *
   * @param \Drupal\Core\Config\StorageInterface $configStorage
   *   The active configuration storage.
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleExtensionList
   *   The module extension list.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    StorageInterface $configStorage,
    ModuleExtensionList $moduleExtensionList,
    ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct();
    $this->configStorage = $configStorage;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->configFactory = $configFactory;
  }

  /**
   * Import/update German configuration translations from this module.
   *
   * Copies all YAML files from this module's config/sync/language/de directory
   * into the active language.de configuration collection. Existing entries are
   * overwritten, new ones are created.
   *
   * @command wisski-ddm:import-translations
   * @aliases wddm:import-translations
   *
   * @option prune
   *   Delete language.de configs that are not present in the module source.
   *
   * @usage drush wisski-ddm:import-translations
   *   Import/update German configuration translations from this module.
   * @usage drush wisski-ddm:import-translations --prune
   *   Import/update German configuration translations from this module
   *   and prune obsolete configs.
   */
  public function importTranslations(array $options = ['prune' => FALSE]): void {
    $modulePath = $this->moduleExtensionList->getPath('wisski_ddm_translations');
    $sourceDir = $modulePath . '/config/sync/language/de';
    $this->io()->text('Source directory: ' . $sourceDir . '.');

    $sourceStorage = new FileStorage($sourceDir);

    $sourceNames = $sourceStorage->listAll();
    if (empty($sourceNames)) {
      $this->io()->warning('No translation config files found in source directory.');
      return;
    }

    $activeDefaultStorage = $this->configStorage;
    $activeLanguageDeStorage = $activeDefaultStorage->createCollection('language.de');

    foreach ($sourceNames as $configName) {
      $data = $sourceStorage->read($configName);
      if ($data === FALSE) {
        $this->io()->warning("Skipping '$configName' (could not read).");
        continue;
      }

      $activeLanguageDeStorage->write($configName, $data);
      $this->io()->success("Imported/updated: language.de::$configName.");
    }

    if (!empty($options['prune'])) {
      $allActiveNames = $activeLanguageDeStorage->listAll();
      $toDelete = array_diff($allActiveNames, $sourceNames);

      foreach ($toDelete as $configName) {
        $activeLanguageDeStorage->delete($configName);
        $this->io()->warning("Deleted obsolete: language.de::$configName.");
      }
    }

    $this->configFactory->reset();
    $this->io()->success('WissKI DDM German translations imported.');
  }

}
