<?php

namespace Drupal\wordsonline_connector\Controller;

use Drupal\Core\Archiver\ArchiverManager;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\tmgmt_file\Format\FormatManager;
use Drupal\wordsonline_connector\Common\WordsOnlineConfig;
use Drupal\wordsonline_connector\Common\ZipHandle;
use Drupal\wordsonline_connector\Entity\JobItemListDataResponse;
use Drupal\wordsonline_connector\Entity\JobListDataResponse;
use Drupal\wordsonline_connector\Entity\WOFile;
use Drupal\wordsonline_connector\WordsOnlineConnectorManager;
use Drupal\wordsonline_connector\WordsOnlineConst;
use Drupal\wordsonline_connector\WordsOnlineState;
use Drupal\wordsonline_connector\WordsOnlineStatus;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\wordsonline_connector\Entity\WolRequestJsonResponse;
use Drupal\wordsonline_connector\Common\WordsOnlineHelper;

/**
 * Controller for handle request.
 */
class RequestController extends ControllerBase
{
  use StringTranslationTrait;

  /**
   * Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * Database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * ModuleExtensionList.
   *
   * @var Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * Archiver.
   *
   * @var Drupal\Core\Archiver\ArchiverManager
   */
  protected $archiver;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file format manager.
   *
   * @var \Drupal\tmgmt_file\Format\FormatManager
   */
  protected $formatManager;

  /**
   * WordsOnline Connector Manager.
   *
   * @var \Drupal\wordsonline_connector\WordsOnlineConnectorManager
   */
  protected $wo_manager;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   Http client.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   Messenger.
   * @param \Drupal\Core\Archiver\ArchiverManager $archiver
   *   Archiver manager.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   File System.
   * @param \Drupal\tmgmt_file\Format\FormatManager $formatManager
   *   The file format manager.
   * @param \Drupal\Core\Extension\ModuleExtensionList
   *   The list module extension list.
   */
  public function __construct(ClientInterface $client, Connection $database, Messenger $messenger, ArchiverManager $archiver, FileSystemInterface $fileSystem, FormatManager $formatManager, ModuleExtensionList $moduleExtensionList, WordsOnlineConnectorManager $wo_manager)
  {
    $this->client = $client;
    $this->database = $database;
    $this->messenger = $messenger;
    $this->archiver = $archiver;
    $this->fileSystem = $fileSystem;
    $this->formatManager = $formatManager;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->wo_manager = $wo_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    $client = $container->get('http_client');
    $database = $container->get('database');
    $messenger = $container->get('messenger');
    $archiver = $container->get('plugin.manager.archiver');
    $fileSystem = $container->get('file_system');
    $manager = $container->get('plugin.manager.tmgmt_file.format');
    $moduleExtensionList = $container->get('extension.list.module');
    $wo_manager = $container->get('plugin.manager.tmgmt.wordsonline');
    return new static(
      $client,
      $database,
      $messenger,
      $archiver,
      $fileSystem,
      $manager,
      $moduleExtensionList,
      $wo_manager
    );
  }

  /**
   * Request screen.
   *
   * @param Symfony\Component\HttpFoundation\Request $request
   *   The request from client.
   */
  public function requests(Request $request)
  {

    $html = ' <div id="app" ></div>';
    $form["#prefix"] = $html;
    $form["#attached"]["library"][] = "wordsonline_connector/wordsonline_request_ui";
    return $form;
  }

  /**
   * Get list requests.
   *
   * @param string $filter
   *   The filter string.
   * @param int $skip
   *   Skip record.
   * @param int $top
   *   Number of record per page.
   * @param string $token
   * The token.
   */
  public function getListRequestFromWordsOnline($filter, $skip, $top = 10, $token = NULL)
  {
    $page = 0;
    if ($skip != NULL && $skip != "") {
      $page = (int)$skip;
    }

    $page = $skip * $top;
    if ($token == NULL) {
      $token = $this->wo_manager->getToken();
    }
    if ($token != NULL) {
      $url = WordsOnlineConfig::getApiUrl() . 'Requests?$skip=' . $page . '&$top=' . $top;
      if ($filter != NULL && $filter != "") {
        $url = $url . '&' . $filter;
      }
      $url .= '&$orderby=CreatedAt desc ';
      $result = $this->client->get(
        $url,
        [
          "headers" => [
            "Authorization" => "Bearer " . $token,
            "Accept" => "application/json",
            "Referer" => "ClientAPI",
          ],
          "timeout" => 3600,
        ]
      );
      if ($result->getStatusCode() == 200) {
        $data = json_decode($result->getBody()->getContents());
        if ($data->status == 1) {

          return [
            "count" => $data->result->count,
            "list" => $data->result->list
          ];
        }
      }
    }
    return NULL;
  }

  /**
   * Build the filter query for list request screen.
   *
   * @param int $type
   * The job type.
   * @param string $keyword
   * The keyword for search.
   */
  public function buildFilterQuery($type = -1, $keyword = "")
  {
    if ($type == -1 && $keyword == "") {
      return "";
    }
    $result = '$filter=';
    $query_type = "";
    $query_keyword = "";
    if ($type != -1) {
      $query = "select request_name from wordsonline_connector_jobs j where j.type = $type order by request_name ";
      $request_names = $this->database->query($query)->fetchAll();
      if ($request_names != NULL) {
        $query_type = "(RequestName in (";
        $i = 1;
        $count = count($request_names);
        foreach ($request_names as $rn) {
          $query_type .= '%27' . $rn->request_name . '%27';
          if ($i < $count) {
            $query_type .= ",";
          }
          $i++;
        }
        $query_type .= "))";
      } else {
        $result = "no data";
        return $result;
      }
    }

    if ($keyword != '') {
      $query_keyword = '(contains(tolower(RequestName),tolower(%27' . $keyword . '%27))';
      $is_numeric = is_numeric($keyword);
      if ($is_numeric == TRUE || $is_numeric == 1) {
        $query_keyword .= ' or (RequestId eq ' . $keyword . ') ';
      }
      $query_keyword .= ')';
    }
    if ($query_type != '' && $query_keyword != '') {
      $result .= "($query_type and $query_keyword)";
    } else {
      $result .= $query_type . $query_keyword;
    }
    return $result;
  }

  /**
   * Get job list.
   *
   * @param Symfony\Component\HttpFoundation\Request $request
   *   The request from client.
   */
  public function getJobList(Request $request)
  {
    $per_page = $request->query->get("per_page");
    if (!isset($per_page)) {
      $per_page = 10;
    }
    $skip = $request->query->get("skip");
    if (!isset($skip)) {
      $skip = 0;
    }
    $type = $request->query->get("type");
    if (!isset($type)) {
      $type = -1;
    }
    $keyword = $request->query->get("q");
    if (!isset($keyword) || $keyword == NULL) {
      $keyword = "";
    }
    $show_retry = FALSE;
    $is_retry = $request->query->get("show_retry");
    if (!isset($is_retry)) {
      $show_retry = FALSE;
    } else if ($is_retry == "true") {
      $show_retry = TRUE;
    }
    $res = [];
    $filter = "";
    $count = 0;
    if ($show_retry == TRUE) {
      $retry_list = $this->getListForRetry($type, $keyword);
      $count = count($retry_list);
      foreach ($retry_list as $itm) {
        array_push($res, $itm);
      }

    } else {
      $filter = $this->buildFilterQuery($type, $keyword);
      if ($filter == "no data") {
        return new JsonResponse([
          "count" => $count,
          "data" => $res
        ]);
      }
      $list = $this->getListRequestFromWordsOnline($filter, $skip, $per_page);
      $count = $list["count"];
      $data = $list["list"];
      if ($count == 0) {
        return new JsonResponse([
          "count" => $count,
          "data" => $res
        ]);
      }

      if ($count > 0) {
        $request_guids = [];


        foreach ($data as $d) {
          array_push($request_guids, $d->requestGuid);
        }
        $records = $this->database->select(WordsOnlineConst::JOB_TABLE, "wol");
        $records->fields("wol");
        $records->condition("wol.request_guid", $request_guids, "in");
        $records->orderBy("wol.job_id", "DESC");
        $job_records = $records->execute()->fetchAll();
        foreach ($data as $d) {
          $returnObject = new JobListDataResponse(0, $d->requestName, $d->requestGuid);
          $returnObject->request_id = $d->requestId;
          $returnObject->state = $d->state;
          $returnObject->source_language_name = $d->sourceLanguageName;
          $returnObject->source_language_code = $d->sourceLanguageCode;
          $returnObject->target_language_name = $d->targetLanguageName;
          $returnObject->target_language_code = $d->targetLanguageCode;
          $returnObject->order_id = $d->orderId;
          $returnObject->project_id = $d->projectId;
          $returnObject->service_level = $d->serviceLevel;
          $returnObject->due_date = $d->dueDate;
          foreach ($job_records as $jr) {
            if ($jr->request_guid == $returnObject->request_guid) {
              $job = Job::load($jr->job_id);
              $returnObject->id = $jr->id;
              $returnObject->job_type = $jr->type;
              if ($job) {
                $auto_approve_quote = $job->getSetting('auto_approve_quote');
                $returnObject->job_id = $jr->job_id;
                $returnObject->status = $d->status;
                $returnObject->content_type = $job->getSetting('content_type');
                $returnObject->is_show_view_job_item = 1;
                if (!$job->isContinuous() && !$auto_approve_quote && !in_array($d->state, [WordsOnlineState::QUOTE_APPROVED, WordsOnlineState::PARTIALLY_DELIVERED, WordsOnlineState::DELIVERED])) {
                  $returnObject->due_date = $job->getSetting('due_date');
                }

                switch ($returnObject->status) {
                  case WordsOnlineStatus::IN_PROGRESS_STATUS:
                    $returnObject->is_show_view_quote = 1;
                    break;
                  case WordsOnlineStatus::Unpaid:
                    $returnObject->is_show_view_quote = 1;
                  case WordsOnlineStatus::QUOTE_SUBMITTED:
                    $returnObject->is_show_view_quote = 1;
                    break;
                  case WordsOnlineStatus::ORDERED:
                    $returnObject->is_show_view_quote = 1;
                    break;
                  case WordsOnlineStatus::DELIVERED:
                    $returnObject->is_show_import = 1;
                    $returnObject->is_show_download = 1;
                    break;
                  case WordsOnlineStatus::IMPORTED_STATUS:
                  case WordsOnlineStatus::IMPORT_FAIL_STATUS:
                  case WordsOnlineStatus::ACCEPT_FAILED_STATUS:
                    $returnObject->is_show_reimport = 1;
                    $returnObject->is_show_download = 1;
                    break;
                  case WordsOnlineStatus::FINISHED_STATUS:
                    $returnObject->is_show_view_job = 1;
                    break;
                }
                // if($jr->status == WordsOnlineConst::JOB_CREATION_FAILED) {
                //   $returnObject->is_show_retry = 1;
                // }
                $returnObject->label = html_entity_decode($job->label());
                $job_items = $this->wo_manager->getWolJobItemIdByRequestGuid($returnObject->request_guid);
                $returnObject->job_items_count = count($job_items);
              }
              break;
            }
          }
          array_push($res, $returnObject);
        }
      }
    }

    return new JsonResponse([
      "count" => $count,
      "data" => $res
    ]);
  }

  /**
   * Get retry list.
   *
   * @param int $type
   * The job type.
   * @param string $keyword
   * The keyword for search.
   */
  public function getListForRetry($type = -1, $keyword = "")
  {
    $records = $this->database->select(WordsOnlineConst::JOB_TABLE, "wol");
    $records->fields("wol");
    $records->condition("wol.status", WordsOnlineConst::JOB_CREATION_FAILED);
    if ($type != -1) {
      $records->condition("wol.type", $type);
    }
    if ($keyword != "") {
      $records->condition('wol.request_name', "%" . $this->database->escapeLike($keyword) . "%", 'LIKE');
    }
    $records->orderBy("wol.job_id", "DESC");
    $job_records = $records->execute()->fetchAll();
    $results = [];
    foreach ($job_records as $row) {
      $returnObject = new JobListDataResponse($row->id, $row->request_name, $row->request_guid);
      $returnObject->project_id = $row->project_guid;
      $returnObject->job_id = $row->job_id;
      $returnObject->job_type = $row->type;
      $returnObject->status = WordsOnlineConst::CREATION_FAILED;
      $returnObject->is_show_retry = 1;
      $job = Job::load($row->job_id);
      if ($job) {
        if ($returnObject->job_type == 0) {
          $returnObject->due_date = $job->getSetting('due_date');
        }
        $returnObject->service_level = WordsOnlineHelper::getServiceLevel($job);
        $returnObject->content_type = $job->getSetting('content_type');
        $returnObject->source_language_code = $job->getSourceLangcode();
        $returnObject->target_language_code = $job->getTargetLangcode();
        $returnObject->target_language_name = $job->getTargetLanguage()->getName();
        $returnObject->source_language_name = $job->getSourceLanguage()->getName();
        $returnObject->is_show_view_job_item = 1;
        $returnObject->label = html_entity_decode($job->label());
        $job_items = $this->wo_manager->getWolJobItemIdByRequestGuid($row->request_guid);
        $returnObject->job_items_count = count($job_items);
      }
      array_push($results, $returnObject);
    }
    return $results;
  }

  /**
   * Retry create new request
   */
  public function retry(Request $request)
  {
    $id = 0;
    if ($request->query->get("id")) {
      $id = $request->query->get("id");
    }
    if (!isset($id) || $id == 0) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Id is null or empty."));
    }
    $job_id = 0;
    if ($request->query->get("job_id")) {
      $job_id = $request->query->get("job_id");
    }
    if (!isset($job_id) || $job_id == 0) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Job id is null or empty."));
    }
    $this->wo_manager->retryCreateRequest($job_id, $id);
    $results = $this->database->query("SELECT status FROM wordsonline_connector_jobs WHERE id=:id", [
      ":id" => $id
    ])->fetchAssoc();
    if ($results != NULL && $results["status"] != WordsOnlineConst::JOB_CREATION_FAILED) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("success", ""));
    } else {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", ""));
    }

  }

  /**
   * Get the quote data.
   */
  public function getQuoteData(Request $request)
  {
    $res["status"] = "error";
    $request_guid = $request->query->get("request_guid");
    if (isset($request_guid)) {
      $request_guid = $request->query->get("request_guid");
      $token = $this->wo_manager->getToken();
      if ($token != NULL) {
        if ($request_guid != NULL) {
          $quotes = wordsonline_connector_get_quote($request_guid, $token);
          if ($quotes != NULL && $quotes->status == 1) {
            $res["status"] = "success";
            $res["data"] = $quotes->result;
          }
          $request = $this->getRequestFormWol($token, $request_guid);
          $res["items_to_translate"] = 0;
          $res["addition_preferences"] = [];
          if ($request != NULL) {
            $res["items_to_translate"] = $request->itemsToTranslate;
            $res["addition_preferences"] = $request->additionPreferences;
          }
        }
        return new JsonResponse($res);
      } else {
        return new JsonResponse($res);
      }
    } else {
      return new JsonResponse($res);
    }
  }

  /**
   * Get request from wordsonline by request guid.
   *
   * @param string $token
   * The token.
   * @param string $request_guid
   * The request guid
   */
  public function getRequestFormWol($token, $request_guid)
  {
    $url = WordsOnlineConfig::getApiUrl() . 'Requests/' . $request_guid;
    $result = $this->client->get(
      $url,
      [
        "headers" => [
          "Authorization" => "Bearer " . $token,
          "Accept" => "application/json",
          "Referer" => "ClientAPI",
        ],
        "timeout" => 3600,
      ]
    );
    if ($result->getStatusCode() == 200) {
      $data = json_decode($result->getBody()->getContents());
      if ($data->status == 1) {
        return $data->result;
      }
    }
    return NULL;
  }

  /**
   * Get request detail
   */
  public function getRequestDetail(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    if (!isset($request_guid) || $request_guid == NULL || $request_guid == '') {
      return new JsonResponse(WolRequestJsonResponse::createFull("error", "", []));
    }
    $records = $this->database
      ->query(
        "SELECT ji.*,j.status FROM wordsonline_connector_jobs_items ji JOIN wordsonline_connector_jobs j ON ji.wol_job_id = j.id WHERE j.request_guid =:request_guid",
        [":request_guid" => (string)$request_guid]
      )
      ->fetchAll();
    $datas = [];
    foreach ($records as $row) {
      $job_item_id = $row->job_Item_id;
      $job_item = JobItem::load($job_item_id);
      $returnObject = new JobItemListDataResponse($job_item_id);
      $returnObject->source_id = $job_item->getItemId();
      $returnObject->source_label = $job_item->getSourceLabel();
      $returnObject->show_view_job_item = 1;
      $state = $job_item->getState();
      switch ($state) {
        case 0:
          $returnObject->status = "Inactive";
          break;
        case 3:
          $returnObject->status = "Accepted";
          $returnObject->show_view_job_item = 1;
          break;
        case 4:
          $returnObject->status = "Aborted";
          break;
        default:
          if ($row->status == WordsOnlineConst::JOB_PARTIALLY_IMPORTED || $row->status == WordsOnlineConst::JOB_IMPORTED) {
            if ($this->wo_manager->isItemImported($job_item) == TRUE) {
              $returnObject->status = WordsOnlineConst::NEED_REVIEW;
              $returnObject->show_reimport_job_item = 1;
            } else {
              $returnObject->status = WordsOnlineStatus::IMPORT_FAILED_STATUS;
              $returnObject->show_reimport_job_item = 1;
            }
          } else {
            if ($job_item->isActive()) {
              $returnObject->status = WordsOnlineStatus::IN_PROGRESS_STATUS;
            }
          }
          break;
      }
      //$storage = \Drupal::entityTypeManager()->getStorage('node');
      $returnObject->revision_id = $row->revision_id;
      // $revision = $storage->loadRevision($returnObject->revision_id);
      // if(isset($revision)) {
      //   $returnObject->revision_label = $revision->label();
      // }
      if (isset($row->revision_id) && $row->revision_id != NULL && is_numeric($row->revision_id)) {
        $revision_id = intval($row->revision_id);
        $returnObject->revision_label = $this->wo_manager->getRevisionTimestamp($revision_id);
      }
      array_push($datas, $returnObject);
    }
    // In old logic, the wordsonline_connector_jobs_items table is not exist .we will fix for normal job by get list item by job id
    if (count($datas) == 0) {
      $wol_job = $this->database->query(
        "SELECT job_id FROM wordsonline_connector_jobs j WHERE j.request_guid =:request_guid LIMIT 1",
        [
          ":request_guid" => (string)$request_guid
        ]
      )
        ->fetchAssoc();
      if ($wol_job != NULL) {
        $job_id = $wol_job["job_id"];
        $job = Job::load($job_id);
        if (!$job->isContinuous()) {
          $job_items = array_values($job->getItems());
          foreach ($job_items as $job_item) {
            $returnObject = new JobItemListDataResponse($job_item->id());
            $returnObject->source_id = $job_item->getItemId();
            $returnObject->source_label = $job_item->getSourceLabel();
            $state = $job_item->getState();
            $returnObject->show_view_job_item = 1;
            switch ($state) {
              case 0:
                $returnObject->status = "Inactive";
                break;
              case 3:
                $returnObject->status = "Accepted";
                break;
              case 4:
                $returnObject->status = "Aborted";
                break;
              default:
                if ($row->status == WordsOnlineConst::JOB_PARTIALLY_IMPORTED || $row->status == WordsOnlineConst::JOB_IMPORTED) {
                  if ($this->wo_manager->isItemImported($job_item) == TRUE) {
                    $returnObject->status = WordsOnlineConst::NEED_REVIEW;
                    $returnObject->show_reimport_job_item = 1;
                  } else {
                    $returnObject->status = WordsOnlineStatus::IMPORT_FAILED_STATUS;
                    $returnObject->show_reimport_job_item = 1;
                  }
                } else {
                  if ($job_item->isActive()) {
                    $returnObject->status = WordsOnlineStatus::IN_PROGRESS_STATUS;
                  }
                }
                break;
            }
            array_push($datas, $returnObject);
          }
        }

      }
    }
    return new JsonResponse(WolRequestJsonResponse::createFull("success", "", $datas));
  }

  /**
   * Confirm the quote.
   */
  public function confirmQuote(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    if (!isset($request_guid) || $request_guid == "" || $request_guid == NULL) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Request guid is null or empty."));
    }
    $due_date = "";
    if ($request->query->get("due_date")) {
      $due_date = $request->query->get("due_date");
    }
    $token = $this->wo_manager->getToken();
    if ($token != NULL) {
      $status = wordsonline_connector_approve_quote($request_guid, "Approve", $due_date, $token);
      if ($status == 1) {
        $this->database->update(WordsOnlineConst::JOB_TABLE)
          ->condition('request_guid', $request_guid)
          ->fields(['status' => WordsOnlineConst::JOB_ORDERED])
          ->execute();
        return new JsonResponse(WolRequestJsonResponse::createWithoutData("success", "Order is confirmed!"));
      } else {
        return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Can't confirm order."));
      }
    } else {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Can't get token."));
    }
  }

  /**
   * Cancel the quote.
   */
  public function cancelQuote(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    if (!isset($request_guid) || $request_guid == "" || $request_guid == NULL) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Request guid is null or empty."));
    }
    $job_id = "";
    if ($request->query->get("job_id")) {
      $job_id = $request->query->get("job_id");
    }
    $due_date = "";
    if ($request->query->get("due_date")) {
      $due_date = $request->query->get("due_date");
    }
    $token = $this->wo_manager->getToken();
    if ($token != NULL) {
      $status = wordsonline_connector_approve_quote($request_guid, "Cancel", $due_date, $token);
      if ($status == 1) {
        $this->database->update(WordsOnlineConst::JOB_TABLE)
          ->condition('request_guid', $request_guid)
          ->fields(['status' => WordsOnlineConst::JOB_CANCELLED])
          ->execute();
        $job = Job::load($job_id);
        if ($job) {
          $job->rejected("Cancel order for this job.");
        }
        return new JsonResponse(WolRequestJsonResponse::createWithoutData("success", "Order is cancelled!"));
      } else {
        return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Can't cancel order."));
      }
    } else {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Can't get token."));
    }
  }

  /**
   * Import data for job.
   */
  public function import(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    if (!isset($request_guid) || $request_guid == "" || $request_guid == NULL) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Request guid is null or empty."));
    }
    $job_id = 0;
    if ($request->query->get("job_id")) {
      $job_id = $request->query->get("job_id");
    }
    if (!isset($job_id) || $job_id == 0) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Job id is null or empty."));
    }
    $status = $this->wo_manager->import($job_id, $request_guid);
    $msg = $status == "error" ? "Import failed" : "Import successfully";
    return new JsonResponse(WolRequestJsonResponse::createWithoutData($status, $msg));
  }

  /**
   * Show the delivery file.
   */
  public function showDeliveryFile(Request $request)
  {
    $xmlData = [];
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    $token = $this->wo_manager->getToken();
    $fName = "";
    if ($token != NULL) {
      if ($request_guid != NULL) {
        $files = wordsonline_connector_get_request_files_list(
          $request_guid,
          $token
        );
        if ($files != NULL) {
          foreach ($files->result as $fi) {
            $file_guid = $fi->guid;
            $data = wordsonline_connector_download_file(
              $request_guid,
              $file_guid,
              $token
            );
            $fName = "public://" . $fi->name;
            file_put_contents($fName, $data);
            $zip = new ZipHandle($fName, $this->archiver, $this->fileSystem);
            $fileContent = $zip->fileContent;
            $fileNames = $zip->fileNames;
            //$this->fileSystem->delete($fName);
            if ($fileContent != NULL
              && count($fileContent) > 0
            ) {
              $i = 0;
              foreach ($fileContent as $fcontent) {
                array_push($xmlData, new WOFile(
                  $fileNames[$i],
                  $fcontent
                ));
                $i++;
              }
            }
          }
        }
      }
    }
    return new JsonResponse([
      "data" => $xmlData,
      "file_name" => $fName
    ]);
  }

  /**
   * Get zip file content.
   *
   * @param Symfony\Component\HttpFoundation\Request $request
   *   The request from client.
   */
  public function downloadJobFile(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    $token = $this->wo_manager->getToken();
    $fileName = "";
    if ($token != NULL) {
      if ($request_guid != NULL) {
        $files = wordsonline_connector_get_request_files_list(
          $request_guid,
          $token
        );
        if ($files != NULL) {
          foreach ($files->result as $fi) {
            $fileName = $fi->name;
            break;
          }
        }
      }
    }
    $uri = 'public://' . $fileName;
    $headers = [
      'Content-Type' => 'application/zip',
      'Content-Disposition' => 'attachment;filename="' . $fileName . '"',
    ];
    return new BinaryFileResponse($uri, 200, $headers, TRUE);
  }

  /**
   * Auto action.
   */
  public function autoAction()
  {
    $records = $this->database->select(WordsOnlineConst::JOB_TABLE, "wol");
    $records->fields("wol");
    $records->orderBy("wol.job_id", "DESC");
    $old_datas = $records->execute()->fetchAll();
    $this->wo_manager->translate();
    $new_datas = $records->execute()->fetchAll();
    $is_change = FALSE;
    foreach ($old_datas as $o) {
      foreach ($new_datas as $n) {
        if ($n->request_guid == $o->request_guid && $n->job_id == $o->job_id) {
          if ($n->status != $o->status) {
            $is_change = TRUE;
          }
          break;
        }
      }
      if ($is_change) {
        break;
      }
    }
    return new JsonResponse([
      "is_change" => $is_change
    ]);
  }

  /**
   * Import job item
   */
  public function importItem(Request $request)
  {
    $request_guid = "";
    if ($request->query->get("request_guid")) {
      $request_guid = $request->query->get("request_guid");
    }
    if (!isset($request_guid) || $request_guid == "" || $request_guid == NULL) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Request guid is null or empty."));
    }
    $job_id = 0;
    if ($request->query->get("job_id")) {
      $job_id = $request->query->get("job_id");
    }
    if (!isset($job_id) || $job_id == 0) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Job id is null or empty."));
    }
    $job_item_id = 0;
    if ($request->query->get("job_item_id")) {
      $job_item_id = $request->query->get("job_item_id");
    }
    if (!isset($job_item_id) || $job_item_id == 0) {
      return new JsonResponse(WolRequestJsonResponse::createWithoutData("error", "Job item id is null or empty."));
    }
    $status = $this->wo_manager->importForJobItem($job_id, $job_item_id, $request_guid);
    $msg = $status == "error" ? "Import failed" : "Import successfully";
    return new JsonResponse(WolRequestJsonResponse::createWithoutData($status, $msg));
  }
}
