<?php

namespace Drupal\wordsonline_connector;

use DateTime;
use DateTimeZone;
use Drupal;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\TranslatorPluginUiBase;
use Drupal\wordsonline_connector\Common\WordsOnlineConfig;
use Drupal\wordsonline_connector\Entity\WOLServiceLevel;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;
use Drupal\wordsonline_connector\WordsOnlineMessage;
use Drupal\wordsonline_connector\Common\WordsOnlineHelper;

/**
 * WordsOnline translator UI.
 */
class WordsOnlineTranslatorUi extends TranslatorPluginUiBase
{
  use StringTranslationTrait;

  /**
   * Service list.
   *
   * @var array
   */
  public $services;

  /**
   * Default service level.
   *
   * @var string
   */
  public $defaultServiceLevel;

  /**
   * Guzzle HTTP client.
   *
   * @var ClientInterface
   */
  protected $client;

  /**
   * Database.
   *
   * @var Connection
   */
  protected $database;

  /**
   * Messenger.
   *
   * @var Messenger
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var LoggerInterface
   */
  protected $logger;

  /**
   * Constructor.
   */
  public function __construct()
  {
    $this->client = Drupal::httpClient();
    $this->database = Database::getConnection();
    $this->messenger = Drupal::messenger();
    $this->logger = Drupal::logger('wordsonline_connector');
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   Form.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return array
   *   Form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state)
  {
    $form = parent::buildConfigurationForm($form, $form_state);
    $results = $this->database->select('wordsonline_connector_configs', 'wol')->extend('\Drupal\Core\Database\Query\PagerSelectExtender')->fields('wol')->limit(1)->execute()->fetchAssoc();
    $username = NULL;
    $password = NULL;
    $project_key = NULL;
    $retry_interval = WordsOnlineConfig::getRetryInterval();
    if ($results != NULL) {
      $username = $results["username"];
      $password = $results["password"];
      $project_key = $results["project_key"];
    }
    $form['username'] = ['#type' => 'textfield', '#title' => $this->t('Username'), '#value' => $username, '#description' => $this->t('Please enter the username.'), '#description_display' => 'after', '#required' => TRUE, '#placeholder' => 'Please enter the username.', '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    $form['password'] = ['#type' => 'password', '#title' => $this->t('Password'), '#value' => $password, '#description' => $this->t('Please enter the password.'), '#description_display' => 'after', '#required' => TRUE, '#placeholder' => 'Please enter the password.', '#attributes' => ['value' => $password], '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    $form['project_key'] = ['#type' => 'textfield', '#title' => $this->t('Project ID'), '#value' => $project_key, '#description' => $this->t('Please enter the Project ID.'), '#description_display' => 'after', '#required' => TRUE, '#placeholder' => 'Please enter the Project ID.', '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    $form['retry_interval'] = ['#type' => 'number', '#title' => $this->t('Cronjob Retry Interval (seconds)'), '#min' => 0, '#max' => 120, '#value' => $retry_interval, '#description' => $this->t('Please enter the retry interval in seconds. This will apply to cron job actions.'), '#description_display' => 'after', '#required' => FALSE, '#placeholder' => 'Please enter the retry interval (seconds).', '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    if (isset($_GET['change_api_url']) && $_GET['change_api_url'] == '1') {
      $api_url = WordsOnlineConfig::getApiUrl();
      $form['api_url'] = ['#type' => 'textfield', '#title' => $this->t('WordsOnline API URL'), '#value' => $api_url, '#description' => $this->t('Please enter WordsOnline API URL and include https:// or http://'), '#description_display' => 'after', '#required' => TRUE, '#placeholder' => 'Please enter the WordsOnline API URL.', '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    }
    if (isset($_GET['change_decode']) && $_GET['change_decode'] == '1'){
      $form['decode_html_entity'] = ['#title' => $this->t('Decode HTML Entities'), '#description' => $this->t('Decode HTML entities in the translated content before a job item is imported.'), '#description_display' => 'after', '#type' => 'checkbox', '#value' => WordsOnlineConfig::getDecodeHtmlEntity(), '#prefix' => '<div class="wol-checkbox-field">', '#suffix' => '</div>',];
    }
    if (isset($_GET['change_format']) && $_GET['change_format'] == '1'){
      $form['xliff_format'] = [
        '#title' => $this->t('XLIFF Format'),
        '#description' => $this->t('Change the XLIFF format.'),
        '#description_display' => 'after',
        '#type' => 'radios',
        '#options' => [
          'html_encoded' => $this->t('HTML Encoded'),
          'xliff_processing' => $this->t('Extended XLIFF Processing'),
          'xliff_cdata' => $this->t('XLIFF with CDATA'),
        ],
        '#default_value' => WordsOnlineConfig::getXliffFormat(),
      ];
    }
    $form['grant_type'] = ['#type' => 'hidden', '#default_value' => 'password', '#value' => 'password',];
    $form['scope'] = ['#type' => 'hidden', '#default_value' => 'webapi enterprise enterpriseordering', '#value' => 'webapi enterprise enterpriseordering',];
    $form += parent::addConnectButton();
    $form['#attached']['library'][] = 'wordsonline_connector/wordsonline_form';
    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   Form.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state)
  {
    parent::validateConfigurationForm($form, $form_state);
    if ($form_state->hasAnyErrors()) {
      return;
    }
    try {
      $field = $form_state->getUserInput();
      WordsOnlineConfig::setRetryInterval((int)$field['settings']['retry_interval'] ?? 60);
      $fields["username"] = $field['settings']['username'];
      $fields["password"] = $field['settings']['password'];
      $fields["scope"] = $field['settings']['scope'];
      $fields["grant_type"] = $field['settings']['grant_type'];
      if (isset($_GET['change_api_url']) && $_GET['change_api_url'] == '1') {
        $api_url = $field['settings']['api_url'] ?? '';
        if (Drupal\Component\Utility\UrlHelper::isValid($api_url)) {
          WordsOnlineConfig::setApiUrl($api_url);
        }
      }

      if (isset($_GET['change_decode']) && $_GET['change_decode'] == '1') {
        $decode_html_entity = $field['settings']['decode_html_entity'] ?? false;
        WordsOnlineConfig::setDecodeHtmlEntity((bool)$decode_html_entity);
      }

      if (isset($_GET['change_format']) && $_GET['change_format'] == '1') {
        $xliff_format = $field['settings']['xliff_format'] ?? 'xliff_cdata';
        WordsOnlineConfig::setXliffFormat($xliff_format);
      }

      $result = wordsonline_connector_get_status_code('POST', $fields);
      if ($result == 401) {
        $authen_err = WordsOnlineMessage::AUTHEN_ERROR;
        $form_state->setErrorByName('settings][username', $authen_err);
      }
      if ($result == 200) {
        try {
          WordsOnlineHelper::clearTokenCache();
          $this->database->delete('wordsonline_connector_configs')->execute();
          $f = ['username', 'password', 'scope', 'grant_type', 'api_url', 'config_id', 'project_key',];
          $values = ['username' => $fields["username"], 'password' => $fields["password"], 'scope' => $fields["scope"], 'grant_type' => $fields["grant_type"], 'api_url' => WordsOnlineConfig::getApiUrl(), 'config_id' => 1, 'project_key' => $field['settings']['project_key'],];
          $this->database->insert('wordsonline_connector_configs')->fields($f)->values($values)->execute();
          $authen_saved = WordsOnlineMessage::AUTHEN_SAVED;
          $this->messenger->addMessage($authen_saved);
        } catch (\Exception $ex) {
          $this->logger->error($ex->getMessage());
        }
      }
    } catch (\Exception $e) {
      $this->logger->error($e->getMessage());
    }
  }

  /**
   * Checkout settings form.
   *
   * @param array $form
   *   Form.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   * @param Drupal\tmgmt\JobInterface $job
   *   Job.
   */
  public function checkoutSettingsForm(array $form, FormStateInterface $form_state, JobInterface $job)
  {
    $translator = $job->getTranslator();
    $projects = [];
    $results = $this->database->query(WordsOnlineConst::SELECT_CONFIG)->fetchAssoc();
    $contentsarray = [];
    $project_key = $job->getSetting('project_key');
    $timezone = date_default_timezone_get();
    if ($results != NULL) {
      $auth["username"] = $results["username"];
      $auth["password"] = $results["password"];
      $auth["scope"] = $results["scope"];
      $auth["grant_type"] = $results["grant_type"];
      $response = wordsonline_connector_get_token('POST', $auth);
      $response = json_decode($response, TRUE);
      $token = $response['access_token'];
      if ($project_key == NULL) {
        $project_key = $results['project_key'];
      }
      if ($token == NULL || $token == '') {
        $this->messenger->addError($this->pleaseContactHelpDeskMessage());
        return;
      }
      $timezone = timezone_name_from_abbr($response['timezone'] ?? 'UTC');
      $projects = $this->getProject('GET', $project_key, $token);
      if ($projects) {
        foreach ($projects['contentTypes'] as $row) {
          $contentsarray[$row] = $row;
        }
        $this->services = $projects["serviceLevels"];
      }
    } else {
      $this->messenger->addError($this->pleaseContactHelpDeskMessage());
    }

    $form['project_key'] = ['#type' => 'hidden', '#required' => TRUE, '#default_value' => $project_key, '#value' => $project_key,];
    $form['content_type'] = ['#title' => $this->t('Content Type'), '#type' => 'select', '#required' => TRUE, "#empty_option" => $this->t('- Select -'), '#options' => $contentsarray, '#default_value' => $job->getSetting('content_type'), '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    foreach ($contentsarray as $ct) {
      $selectOpts1 = [];
      foreach ($this->services as $s) {
        if ($s->contentType == $ct) {
          $selectOpts1[$s->serviceName] = $s->serviceName;
        }
      }
      $select_sl_name = WordsOnlineHelper::getServiceLevelName($ct);
      $defaultServiceLevelVal = $job->getSetting($select_sl_name);
      $form[$select_sl_name] = ['#title' => $this->t('Service Level'), '#type' => 'select', "#empty_option" => $this->t('- Select -'), '#default_value' => $defaultServiceLevelVal, '#options' => $selectOpts1, '#states' => ['visible' => ['select[name="settings[content_type]"]' => ['value' => $ct]], 'required' => ['select[name="settings[content_type]"]' => ['value' => $ct]],], '#prefix' => '<div class="wol-text-field" id="service_level_wrapper">', '#suffix' => '</div>'];
    }
    $this->defaultServiceLevel = $job->getSetting('service_level');
    //keep for old version
    $form['service_level'] = ['#type' => 'hidden', '#default_value' => $this->defaultServiceLevel, '#value' => $this->defaultServiceLevel,];
    if (!$this->isContinuousJob($job)) {
      if ($timezone) {
        $due_date_title = 'Due Date (' . $timezone . ')';
        $form['due_date'] = ['#title' => $due_date_title, '#type' => 'date', '#default_value' => $job->getSetting('due_date'), '#format' => 'YYY-MM-DD h:mm', '#required' => TRUE, '#attributes' => ['min' => (new DateTime("tomorrow", new DateTimeZone($timezone)))->format('Y-m-d'), 'onchange' => "onDateChange(this)",], '#prefix' => '<div class="wol-text-field wol-des">', '#suffix' => '</div>',];
      } else {
        $due_date_title = 'Due Date';
        $form['due_date'] = ['#title' => $due_date_title, '#type' => 'date', '#default_value' => $job->getSetting('due_date'), '#format' => 'YYY-MM-DD h:mm', '#required' => TRUE, '#attributes' => ['onchange' => "onDateChange(this)",], '#prefix' => '<div class="wol-text-field wol-des">', '#suffix' => '</div>',];
      }

      $request_name = $job->getSetting('request_name');
      if (!$request_name) {
        $request_name = wordsonline_connector_get_request_name();
      }
      $form['request_name'] = ['#type' => 'textfield', '#title' => $this->t('Request Name'), '#default_value' => $request_name, '#required' => TRUE, '#placeholder' => 'Please enter the request name.', '#prefix' => '<div class="wol-text-field">', '#suffix' => '</div>',];
    }

    $form['auto_approve_quote'] = ['#title' => $this->t('Auto Approve Quote'), '#description' => $this->t('Quote of this order will be approved automatically.'), '#description_display' => 'after', '#type' => 'checkbox', '#default_value' => $job->getSetting('auto_approve_quote'), '#prefix' => '<div class="wol-checkbox-field">', '#suffix' => '</div>',];
    $is_auto = $translator->isAutoAccept();
    if ($is_auto == TRUE || $is_auto == 1) {
      $form['auto_import'] = ['#title' => $this->t('Auto import upon receiving delivery files'), '#description' => $this->t('Auto import the translation back to Drupal when the files delivered.'), '#description_display' => 'after', '#type' => 'checkbox', '#default_value' => TRUE, '#prefix' => '<div class="wol-checkbox-field d-hidden" >', '#suffix' => '</div>',];
    } else {
      $form['auto_import'] = ['#title' => $this->t('Auto import upon receiving delivery files'), '#description' => $this->t('Auto import the translation back to Drupal when the files delivered.'), '#description_display' => 'after', '#type' => 'checkbox', '#default_value' => $job->getSetting('auto_import'), '#prefix' => '<div class="wol-checkbox-field">', '#suffix' => '</div>',];
    }
    $form['#attached']['library'][] = 'wordsonline_connector/wordsonline_form';
    return parent::checkoutSettingsForm($form, $form_state, $job);
  }

  /**
   * Get error message when can't call wordsonline api.
   *
   */
  public function pleaseContactHelpDeskMessage()
  {
    return $this->t('Cannot find WordsOnline provider settings.
      Please contact <a href="mailto:@url">@url<a/>
      to obtain WordsOnline provider settings.', ["@url" => WordsOnlineConst::HELPDESK_MAIL,]);
  }

  /**
   * Get list wordsonline project.
   *
   * @param string $method
   *   Method of request.
   * @param string $project_key
   *   Project key.
   * @param string $token
   *   Token.
   *
   * @return array
   *   List project.
   */
  public function getProject($method = 'GET', $project_key = "", $token = NULL)
  {
    $options = [];
    $projects = [];
    $url = WordsOnlineConfig::getApiUrl() . WordsOnlineConst::GET_PROJECT_URL . $project_key;
    try {
      $options['headers'] = ['Authorization' => 'Bearer ' . $token, 'Referer' => 'ClientAPI', 'Content-Type' => 'application/json',];
      $options['timeout'] = 3600;
      $response = $this->client->request($method, $url, $options);
    } catch (RequestException $e) {
      if (!$e->hasResponse()) {
        throw new TMGMTException(WordsOnlineMessage::UNABLE_TO_CONNECT_SERVICES, ['@error' => $e->getMessage()], $e->getCode());
      }
      $response = $e->getResponse();
      throw new TMGMTException(WordsOnlineMessage::UNABLE_TO_CONNECT_SERVICES, ['@error' => $response->getReasonPhrase()], $response->getStatusCode());
    }
    $received_data = $response->getBody()->getContents();
    if ($response->getStatusCode() != 200) {
      throw new TMGMTException(WordsOnlineMessage::UNABLE_TO_CONNECT_SERVICES_WITH_URL, ['@error' => $response->getStatusCode(), '@url' => $url,]);
    }

    $orderResponse = json_decode($received_data, TRUE);
    if ($orderResponse["status"] == 1) {
      $lstProject = $orderResponse["result"];
      foreach ($orderResponse["result"] as $record) {
        foreach ($record["contentTypes"] as $row) {
          $projects["contentTypes"][] = $row;
        }
        foreach ($record["serviceLevels"] as $row) {
          $projects["serviceLevels"][] = new WOLServiceLevel($row['name'], $row['contentType']);
        }
        break;
      }
    }
    return $projects;
  }

  /**
   * Check job is continuous.
   *
   * @param Drupal\tmgmt\JobInterface $job
   * Job.
   *
   */
  public function isContinuousJob($job)
  {
    $job_type = $job->getJobType();
    return $job_type == JobInterface::TYPE_CONTINUOUS;
  }

  /**
   * Submit callback to pull translations from WordsOnlineTranslator.
   *
   * @param array $form
   *   Form.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   */
  public function submitPullTranslations(array $form, FormStateInterface $form_state)
  {
    $job = $form_state->getFormObject()->getEntity();
    $translator_plugin = $job->getTranslator()->getPlugin();
    $translator_plugin->fetchTranslations($job);
    tmgmt_write_request_messages($job);
  }

}
