<?php

namespace Drupal\wordsonline_connector;

use Drupal;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Archiver\ArchiverManager;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\file\FileRepository;
use Drupal\tmgmt\Data;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt_file\Format\FormatManager;
use Drupal\wordsonline_connector\Common\WordsOnlineConfig;
use Drupal\wordsonline_connector\Common\WordsOnlineHelper;
use Drupal\wordsonline_connector\Common\ZipHandle;
use Drupal\wordsonline_connector\Entity\AggregatedJobItem;
use GuzzleHttp\ClientInterface;
use XMLReader;
use ZipArchive;

/**
 * A service manager for wordsonline jobs.
 */
class WordsOnlineConnectorManager
{
  /**
   * Guzzle HTTP client.
   *
   * @var ClientInterface
   */
  protected $client;

  /**
   * Database.
   *
   * @var Connection
   */
  protected $database;

  /**
   * ModuleExtensionList.
   *
   * @var ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * Messenger.
   *
   * @var Messenger
   */
  protected $messenger;

  /**
   * Archiver.
   *
   * @var ArchiverManager
   */
  protected $archiver;

  /**
   * The file system service.
   *
   * @var FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file format manager.
   *
   * @var FormatManager
   */
  protected $formatManager;

  /**
   * The tmgmt data.
   *
   * @var Data
   */
  protected $tmgmt_data;


  /**
   * The file repository service.
   *
   * @var FileRepository
   */
  protected $fileRepository;

  /**
   * The lock backend service.
   *
   * @var LockBackendInterface
   */
  protected $lock;

  /**
   * Constructor.
   *
   * @param ClientInterface $client
   *   Http client.
   * @param Connection $database
   *   Database connection.
   * @param Messenger $messenger
   *   Messenger.
   * @param ArchiverManager $archiver
   *   Archiver manager.
   * @param FileSystemInterface $fileSystem
   *   File System.
   * @param FormatManager $formatManager
   *   The file format manager.
   * @param ModuleExtensionList
   *   The list module extension list.
   * @param Data $tmgmt_data
   * The tmgmt data service.
   * @param FileRepository $fileRepository
   * The file repository service.
   * @param LockBackendInterface $lock
   * The lock backend service.
   */
  public function __construct(ClientInterface $client, Connection $database, Messenger $messenger, ArchiverManager $archiver, FileSystemInterface $fileSystem, FormatManager $formatManager, ModuleExtensionList $moduleExtensionList, Data $tmgmt_data, FileRepository $fileRepository, LockBackendInterface $lock)
  {
    $this->client = $client;
    $this->database = $database;
    $this->messenger = $messenger;
    $this->archiver = $archiver;
    $this->fileSystem = $fileSystem;
    $this->formatManager = $formatManager;
    $this->moduleExtensionList = $moduleExtensionList;
    $this->tmgmt_data = $tmgmt_data;
    $this->fileRepository = $fileRepository;
    $this->lock = $lock;
  }

  /**
   * Clear the token cache.
   */
  public function clearTokenCache()
  {
    $cache = Drupal::cache(WordsOnlineConst::MODULE_ID);
    $cache->delete('token');
  }

  /**
   * Acquire a lock for synchronizing WordsOnline jobs.
   */
  public function acquireSyncLock($wol_job_id, $timeout = 0.5): bool
  {
    return $this->lock->acquire(WordsOnlineConst::SYNC_LOCK_NAME . $wol_job_id, $timeout);
  }

  /**
   * Set Xliff Options
   * @param JobInterface|Job $job
   * @return void
   */
  public function setXliffOptions(JobInterface|Job $job)
  {
    $format = WordsOnlineConfig::getXliffFormat();
    switch ($format) {
      case 'html_encoded':
        $job->settings->xliff_processing = FALSE;
        $job->settings->xliff_cdata = FALSE;
        break;
      case 'xliff_processing':
        $job->settings->xliff_processing = TRUE;
        $job->settings->xliff_cdata = FALSE;
        break;
      case 'xliff_cdata':
      default:
        $job->settings->xliff_cdata = TRUE;
        $job->settings->xliff_processing = FALSE;
        break;
    }
    # ensure the export format is xlf
    $job->settings->export_format = 'xlf';

    # ensure empty target
    $job->settings->format_configuration = [
      'target' => '',
    ];
    $job->save();
  }

  /**
   * Release the lock for synchronizing WordsOnline jobs.
   *
   * @param int $wol_job_id
   * The job id.
   */
  public function releaseSyncLock($wol_job_id): void
  {
    $this->lock->release(WordsOnlineConst::SYNC_LOCK_NAME . $wol_job_id);
  }

  /**
   * Translate action.
   */
  public function translate()
  {
    $token = $this->getToken();
    if ($token != NULL) {
      $records = $this->getRecordsForTranslate(WordsOnlineConst::JOB_TABLE);
      $data = $this->GetDataFromWordsOnline($token);
      foreach ($records as $row) {
        $job = Job::load($row->job_id);
        $job_id = $row->job_id;
        $id = $row->id;
        if ($job) {
          try {
            if (!$this->acquireSyncLock($row->id)) {
              Drupal::logger(WordsOnlineConst::MODULE_ID)->error("Failed to acquire lock for job {$row->job_id}. Another process may be updating this job.");
              continue;
            }
            $translator = $job->getTranslator();
            $is_auto = $translator->isAutoAccept();
            foreach ($data as $d) {
              if ($d->requestGuid == $row->request_guid) {
                if ($d->state == WordsOnlineState::AUTOMATION_FAILED && $row->status != WordsOnlineConst::JOB_AUTOMATION_FAILED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_AUTOMATION_FAILED);
                  continue;
                }
                if (($d->status == WordsOnlineStatus::UNPAID || $d->status == WordsOnlineStatus::QUOTE_SUBMITTED) && $row->status != WordsOnlineConst::JOB_QUOTED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_QUOTED);
                  continue;
                }
                if ($d->status == WordsOnlineStatus::IN_PROGRESS_STATUS && $row->status != WordsOnlineConst::JOB_QUOTED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_QUOTED);
                  continue;
                }
                if ($d->status == WordsOnlineStatus::IMPORTED_STATUS && $row->status == WordsOnlineConst::JOB_DELIVERED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_IMPORTED);
                }
                if (($d->state == WordsOnlineState::QUOTE_SUBMITTED && ($d->status == WordsOnlineStatus::UNPAID || $d->status == WordsOnlineStatus::QUOTE_SUBMITTED)) && $row->status != WordsOnlineConst::JOB_QUOTED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_QUOTED);
                  continue;
                }
                // if ($row->status == WordsOnlineConst::JOB_PARTIALLY_IMPORTED && $d->status != WordsOnlineStatus::PARTIALLY_IMPORTED) {
                //   $this->requestAction($token, $row->request_guid, WordsOnlineStatus::PARTIALLY_IMPORTED);
                //   continue;
                // }
                if ($row->status == WordsOnlineConst::JOB_IMPORTED && $d->status != WordsOnlineStatus::IMPORTED_STATUS) {
                  $this->requestAction($token, $row->request_guid, WordsOnlineStatus::IMPORTED_STATUS);
                  continue;
                }

                if ($row->status != WordsOnlineConst::JOB_PREPARING_QUOTE && $d->status == WordsOnlineStatus::PREPARING_QUOTE) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_PREPARING_QUOTE);
                  continue;
                }

                if (($d->status == WordsOnlineStatus::DELIVERED || $d->status == WordsOnlineStatus::IMPORT_FAIL_STATUS || $d->status == WordsOnlineStatus::IMPORT_FAILED_STATUS) && $row->status != WordsOnlineConst::JOB_DELIVERED) {
                  $this->updateWordsOnlineJobStatusById($id, WordsOnlineConst::JOB_DELIVERED);
                  continue;
                }
              }
            }
            $job_item_ids = $this->getListJobItemsIdByRequestGuid($row->request_guid);
            if ($row->status == WordsOnlineConst::JOB_ORDERED || $row->status == WordsOnlineConst::JOB_QUOTED || $row->status == WordsOnlineConst::JOB_DELIVERED) {

              $auto_import = $job->getSetting("auto_import");
              $files = wordsonline_connector_get_request_files_list($row->request_guid, $token);
              if ($files != NULL) {
                if (isset($files->result) && $files->result && count($files->result) > 0) {
                  if ($auto_import == TRUE || $auto_import == 1 || $is_auto == TRUE || $is_auto == 1) {
                    $this->updateWordsOnlineJobStatusById($row->id, WordsOnlineConst::JOB_DELIVERED);
                    foreach ($files->result as $fi) {
                      $file_guid = $fi->guid;
                      $data = wordsonline_connector_download_file($row->request_guid, $file_guid, $token);
                      $fName = "public://" . $fi->name;
                      file_put_contents($fName, $data);
                      $zip = new ZipHandle($fName, $this->archiver, $this->fileSystem);
                      $fileContent = $zip->fileContent;
                      $this->fileSystem->delete($fName);
                      $is_has_error = FALSE;
                      if ($fileContent != NULL && count($fileContent) > 0) {
                        foreach ($fileContent as $fcontent) {
                          $msg = $this->importTranslation($job, $fcontent, $job_item_ids);

                          if ($msg != "") {
                            $this->messenger->addError("an error occured when import job {$row->job_id} with zip file {$fName}");
                            $this->requestAction($token, $row->request_guid, WordsOnlineStatus::IMPORT_FAIL_STATUS, $msg);
                            $is_has_error = TRUE;
                          }
                        }
                      }
                      if ($is_has_error == TRUE) {
                        continue;
                      }

                      $request_status = $this->getRequestImportStatus($job_item_ids);
                      if ($request_status != "") {
                        $local_wo_status = $request_status == WordsOnlineStatus::IMPORTED_STATUS ? WordsOnlineConst::JOB_IMPORTED : WordsOnlineConst::JOB_PARTIALLY_IMPORTED;
                        if ($request_status == WordsOnlineStatus::IMPORTED_STATUS) {
                          $this->requestAction($token, $row->request_guid, $request_status);
                        }
                        $this->updateWordsOnlineJobStatusById($row->id, $local_wo_status);
                        $row->status = $local_wo_status;
                      }
                    }
                  } else {
                    $this->updateWordsOnlineJobStatusById($row->id, WordsOnlineConst::JOB_DELIVERED);
                  }
                }
              }
            }

            $job_items = array_filter(JobItem::loadMultiple($job_item_ids));
            if ($row->status == WordsOnlineConst::JOB_IMPORTED && ($is_auto == TRUE || $is_auto == 1)) {
              foreach ($job_items as $ji) {
                if ($ji->getState() == JobItemInterface::STATE_REVIEW) {
                  $this->retryAcceptTranslation($ji, $row->request_guid);
                }
              }
              if (!$this->isAllJobItemAccepted($job_items)) {
                $this->requestAction($token, $row->request_guid, WordsOnlineStatus::ACCEPT_FAILED_STATUS);
              } else {
                $this->requestAction($token, $row->request_guid, WordsOnlineStatus::FINISHED_STATUS);
                $this->updateWordsOnlineJobStatusById($row->id, WordsOnlineConst::JOB_FINISHED);
              }
            }
            if ($row->status == WordsOnlineConst::JOB_IMPORTED && $job->getState() == WordsOnlineConst::STATE_FINISHED) {
              $this->requestAction($token, $row->request_guid, WordsOnlineStatus::FINISHED_STATUS);
              $this->updateWordsOnlineJobStatusById($row->id, WordsOnlineConst::JOB_FINISHED);
              continue;
            }
            $is_job_accepted = $this->isAllJobItemAccepted($job_items);
            if ($row->status == WordsOnlineConst::JOB_IMPORTED && $is_job_accepted) {
              $this->requestAction($token, $row->request_guid, WordsOnlineStatus::FINISHED_STATUS);
              $this->updateWordsOnlineJobStatusById($row->id, WordsOnlineConst::JOB_FINISHED);
              continue;
            }
          } catch (\Exception $e) {
            Drupal::logger(WordsOnlineConst::MODULE_ID)->error('Error syncing job: @job_id, message: @message', ['@job_id' => $job_id, '@message' => $e->getMessage()]);
          } finally {
            $this->releaseSyncLock($row->id);
          }
        }
      }
    }
  }

  /**
   * Get the token.
   */
  public function getToken()
  {
    $cache = Drupal::cache(WordsOnlineConst::MODULE_ID);
    $cached_token = $cache->get('token');
    if ($cached_token && $cached_token->data && $cached_token->expire > time()) {
      return $cached_token->data;
    }
    $results = $this->database->query(WordsOnlineConst::SELECT_CONFIG)->fetchAssoc();
    if ($results != NULL) {
      $auth["username"] = $results["username"];
      $auth["password"] = $results["password"];
      $auth["scope"] = $results["scope"];
      $auth["grant_type"] = $results["grant_type"];
      $response = wordsonline_connector_get_token('POST', $auth);
      $response = json_decode($response, TRUE);
      $token = $response['access_token'];
      $cache->set('token', $token, strtotime('+12 hour'));
    } else {
      $this->messenger->addError($this->pleaseContactHelpDeskMessage());
    }
    return $token;
  }

  /**
   * Get records ordered or delivered
   *
   * @param string $table_name
   * The name of table.
   */
  public function getRecordsForTranslate($table_name)
  {
    $records = $this->database->select($table_name, "wol");
    $records->fields("wol");
    $records->condition("wol.status", [WordsOnlineStatus::FINISHED_STATUS], "not in");
    $records->orderBy("wol.job_id", "DESC");
    $res = $records->execute()->fetchAll();
    return $res;
  }

  /**
   * Get Data From WordsOnline.
   *
   * @param string $token
   * The token.
   */
  public function GetDataFromWordsOnline($token)
  {
    $records = [];
    $skip = 0;
    $top = 100;
    $count = 0;
    $total = 0;
    $url = WordsOnlineConfig::getApiUrl() . 'Requests?$skip=' . $skip . '&$top=' . $top . '&$orderby=CreatedAt desc ';
    do {
      $url = WordsOnlineConfig::getApiUrl() . 'Requests?$skip=' . $skip . '&$top=' . $top . '&$orderby=CreatedAt desc ';
      $result = $this->client->get($url, ["headers" => ["Authorization" => "Bearer " . $token, "Accept" => "application/json", "Referer" => "ClientAPI",], "timeout" => 3600,]);
      if ($result->getStatusCode() == 200) {
        $data = json_decode($result->getBody()->getContents());
        if ($data->status == 1) {
          $count = $data->result->count;
          $res = $data->result->list;
          foreach ($res as $r) {
            array_push($records, $r);
          }

          $skip = $skip + 100;
          $total = count($records);
        } else {
          break;
        }
      } else {
        break;
      }
    } while ($total < $count);
    return $records;
  }

  /**
   * Update status of WordsOnline Job table
   *
   * @param int $job_id
   * Job id.
   * @param string $status
   * Status.
   */
  public function updateWordsOnlineJobStatus($job_id, $status)
  {
    $this->database->update(WordsOnlineConst::JOB_TABLE)->condition("job_id", $job_id)->fields(["status" => $status,])->execute();
  }

  /**
   * Update status of WordsOnline Job table by ID
   *
   * @param int $id
   * Job id.
   * @param string $status
   * Status.
   */
  public function updateWordsOnlineJobStatusById($id, $status)
  {
    $this->database->update(WordsOnlineConst::JOB_TABLE)->condition("id", $id)->fields(["status" => $status,])->execute();
  }

  /**
   * Update status of WordsOnline Job table by Request Guid
   *
   * @param string $request_guid
   * Job id.
   * @param string $status
   * Status.
   */
  public function updateWordsOnlineJobStatusByRequestGuid($request_guid, $status)
  {
    $this->database->update(WordsOnlineConst::JOB_TABLE)->condition("request_guid", $request_guid)->fields(["status" => $status,])->execute();
  }

  /**
   * Call request to WordsOnline api for action.
   *
   * @param string $token
   * Token.
   * @param string $request_guid
   * Request guid.
   * @param string $status
   * Status.
   * @param string $msg
   * Message
   */
  public function requestAction($token, $request_guid, $status, $msg = "")
  {
    wordsonline_connector_request_action($request_guid, $status, $msg, $token);
  }

  /**
   * Imports the given translation data.
   *
   * @param JobInterface $job
   *   The job to import translations for.
   * @param string $translation
   *   The translation data.
   *
   * @throws PluginException
   *   Throws an exception if XLF plugin does not exist.
   * @throws TMGMTException
   *   Throws an exception in case of a neeror.
   */
  public function importTranslation(JobInterface $job, $translation, $job_item_ids)
  {
    $this->setXliffOptions($job);
    $xliff = $this->formatManager->createInstance("xlf");
    $messages = $this->getXliffError($job, $translation);
    if ($messages != NULL && !empty($messages) && $messages != "") {
      return $messages;
    }
    if ($data = $xliff->import($translation, FALSE)) {
      $data = array_filter($data, function ($job_item_id) use ($job_item_ids) {
        return in_array($job_item_id, $job_item_ids);
      }, ARRAY_FILTER_USE_KEY);
      $decode_html_entity = WordsOnlineConfig::getDecodeHtmlEntity();
      if ($decode_html_entity) {
        $data = $this->decodeHtmlEntity($data, $job_item_ids);
      }
      if ($data) {
        $job->addTranslatedData($data, NULL, TMGMT_DATA_ITEM_STATE_TRANSLATED);
        $job->addMessage("The translation has been received.");
      }
    }
    return "";
  }

  /**
   * Determine if a translated data field need decode operation.
   * @param $value
   * @return bool
   */
  public function needsDecodeEntity($value)
  {
    return !isset($value['#format']) || !in_array($value['#format'], array('basic_html', 'restricted_html', 'full_html'));
  }

  /**
   * Decode HTML entity for translated data
   * @param $data
   * @param $job_item_ids
   * @return array|mixed
   */
  public function decodeHtmlEntity($data, $job_item_ids)
  {
    if (empty($data)) {
      return $data;
    }
    $job_items = JobItem::loadMultiple(($job_item_ids));
    foreach ($job_items as $job_item) {
      $job_item_id = $job_item->id();
      if (!isset($data[$job_item_id])) {
        continue;
      }
      $translation = $data[$job_item_id];
      $job_data = $job_item->getData();

      // Process the translation data recursively
      $data[$job_item_id] = $this->decodeHtmlEntityRecursive($translation, $job_data);
    }
    return $data;
  }

  /**
   * Recursively decode HTML entities in translated data
   * @param array $translation Translation data
   * @param array $source_data Source data structure for reference
   * @return array Processed translation data
   */
  private function decodeHtmlEntityRecursive($translation, $source_data)
  {
    if (!is_array($translation) || !is_array($source_data)) {
      return $translation;
    }

    foreach ($source_data as $key => $value) {
      if (!isset($translation[$key])) {
        continue;
      }

      if (is_array($value) && is_array($translation[$key]) && !isset($value['#status'])) {
        // Recursively process nested arrays
        $translation[$key] = $this->decodeHtmlEntityRecursive($translation[$key], $value);
      } elseif (is_array($value) && isset($value['#translate']) && $value['#translate'] &&
        isset($value['#text']) && $this->needsDecodeEntity($value) &&
        isset($translation[$key]['#text'])) {
        // Decode HTML entities in text values
        $translation[$key]['#text'] = html_entity_decode($translation[$key]['#text']);
      }
    }

    return $translation;
  }

  /**
   * Decode HTML entity for translated data at job item level
   * @param $translation Translation data for a job item
   * @param $job_item_id Job item ID
   * @return array|mixed Processed translation data
   */
  public function decodeHtmlEntityByItem($translation, $job_item_id)
  {
    if (empty($translation)) {
      return $translation;
    }
    $job_item = JobItem::load($job_item_id);
    $job_data = $job_item->getData();

    // Process the translation data recursively
    return $this->decodeHtmlEntityRecursive($translation, $job_data);
  }

  /**
   * Get Error of xliff xml data.
   *
   * @param JobInterface $job
   * Job.
   * @param string $importedXML
   * Xliff content.
   */
  public function getXliffError($job, $importedXML)
  {
    $xml = simplexml_load_string($importedXML);
    $xml->registerXPathNamespace('xliff', 'urn:oasis:names:tc:xliff:document:1.2');
    $phase = $xml->xpath("//xliff:phase[@phase-name='extraction']");
    if ($phase) {
      $phase = reset($phase);
    } else {
      return 'The imported file is missing required XLIFF phase information.';
    }
    // Check if the job has a valid job reference.
    if (!isset($phase['job-id'])) {
      return 'The imported file does not contain a job reference.';
    }
    // Attempt to load the job if none passed.
    $job = (Job::load((int)$phase['job-id']));
    if (empty($job)) {
      return "The imported file job id " . $phase['job-id'] . " is not available.";
    }
    // Compare source language.
    if (!isset($xml->file['source-language']) || $job->getRemoteSourceLanguage() != $xml->file['source-language']) {
      $file_language = empty($xml->file['source-language']) ? t('none') : $xml->file['source-language'];
      $messages = 'The imported file source language ' . $file_language . ' does not match the job source language ' . $job->getRemoteSourceLanguage() . '.';
      return $messages;
    }
    // Compare target language.
    if (!isset($xml->file['target-language']) || $job->getRemoteTargetLanguage() != $xml->file['target-language']) {
      $file_language = empty($xml->file['target-language']) ? t('none') : $xml->file['target-language'];
      $messages = 'The imported file target language ' . $file_language . ' does not match the job target language ' . $job->getRemoteTargetLanguage() . '.';
      return $messages;
    }
    $targets = $this->getImportedTargets($xml, $job);

    if (empty($targets) || $targets == NULL) {
      return 'The imported file seems to be missing translation.';
    }
    return "";
  }

  /**
   * Get imported targets.
   *
   * @param SimpleXMLElement $importedXML
   * Xml element.
   * @param JobInterface $job
   * Job.
   *
   */
  public function getImportedTargets($importedXML, $job)
  {
    $importedTransUnits = NULL;
    $reader = new XMLReader();
    foreach ($importedXML->xpath('//xliff:trans-unit') as $unit) {
      if (!$job->getSetting('xliff_processing')) {
        $importedTransUnits[(string)$unit['id']]['#text'] = (string)$unit->target;
        continue;
      }

      $reader->XML($unit->target->asXML());
      $reader->read();
      $importedTransUnits[(string)$unit['id']]['#text'] = $this->processForImport($reader->readInnerXML(), $job);
    }
    return $importedTransUnits;
  }

  /**
   * Processes trans-unit/target to rebuild back the HTML.
   *
   * @param string $translation
   *   Job data array.
   * @param JobInterface $job
   *   Translation job.
   *
   * @return string
   */
  public function processForImport($translation, JobInterface $job)
  {
    // In case we do not want to do xliff processing return the translation as
    // is.
    if (!$job->getSetting('xliff_processing')) {
      return $translation;
    }

    $reader = new XMLReader();
    $reader->XML('<translation>' . $translation . '</translation>');
    $text = '';

    while ($reader->read()) {
      // If the current element is text append it to the result text.
      if ($reader->name == '#text' || $reader->name == '#cdata-section') {
        $text .= $reader->value;
      } elseif ($reader->name == 'x') {
        if ($reader->getAttribute('ctype') == 'lb') {
          $text .= '<br />';
        }
      } elseif ($reader->name == 'ph') {
        if ($reader->getAttribute('ctype') == 'image') {
          $text .= '<img';
          while ($reader->moveToNextAttribute()) {
            // @todo - we have to use x-html: prefixes for attributes.
            if ($reader->name != 'ctype' && $reader->name != 'id') {
              $text .= " {$reader->name}=\"{$reader->value}\"";
            }
          }
          $text .= ' />';
        }
      }
    }
    return $text;
  }

  /**
   * Import
   *
   * @param int $job_id
   * Job id.
   * @param string $request_guid
   * Request guid.
   */
  public function import($job_id, $request_guid)
  {
    $token = $this->getToken();
    $status = "";
    if ($token != NULL) {
      $files = wordsonline_connector_get_request_files_list($request_guid, $token);
      if ($files != NULL) {
        if ($files->result != NULL && count($files->result) > 0) {
          $status = "downloaded";
          $job = Job::load($job_id);
          if ($job) {
            $is_import = FALSE;
            $translator = $job->getTranslator();
            $is_auto = $translator->isAutoAccept();
            $job_item_ids = $this->getListJobItemsIdByRequestGuid($request_guid);
            foreach ($files->result as $fi) {
              $file_guid = $fi->guid;
              $data = wordsonline_connector_download_file($request_guid, $file_guid, $token);
              $fName = "public://" . $fi->name;
              file_put_contents($fName, $data);
              $zip = new ZipHandle($fName, $this->archiver, $this->fileSystem);
              $fileContent = $zip->fileContent;
              $this->fileSystem->delete($fName);
              $is_has_error = FALSE;
              if ($fileContent != NULL && count($fileContent) > 0) {
                foreach ($fileContent as $fcontent) {
                  $msg = $this->importTranslation($job, $fcontent, $job_item_ids);
                  if ($msg != "") {
                    //$this->messenger->addError("an error occured when import job {$job_id} with zip file {$fName}");
                    Drupal::logger(WordsOnlineConst::MODULE_ID)->error($msg);
                    $this->requestAction($token, $request_guid, WordsOnlineStatus::IMPORT_FAIL_STATUS, $msg);
                    $is_has_error = TRUE;
                  }
                }
              }
              if ($is_has_error == TRUE) {
                continue;
              }

              $request_status = $this->getRequestImportStatus($job_item_ids);
              if ($request_status != "") {
                if ($request_status == WordsOnlineStatus::IMPORTED_STATUS) {
                  $this->requestAction($token, $request_guid, $request_status);
                }
                $local_wo_status = $request_status == WordsOnlineStatus::IMPORTED_STATUS ? WordsOnlineConst::JOB_IMPORTED : WordsOnlineConst::JOB_PARTIALLY_IMPORTED;
                $this->updateWordsOnlineJobStatusByRequestGuid($request_guid, $local_wo_status);
                $is_import = TRUE;
              }
            }
            if ($is_import) {
              $status = "success";
              if ($this->getRequestImportStatus($job_item_ids) == WordsOnlineStatus::IMPORTED_STATUS && ($is_auto == TRUE || $is_auto == 1) && $job->getState() != WordsOnlineConst::STATE_FINISHED) {
                foreach ($job_item_ids as $jid) {
                  $job_itm = JobItem::load($jid);
                  if ($job_itm) {
                    $this->retryAcceptTranslation($job_itm, $request_guid);
                  }
                }
                if (!$this->isAllJobItemAccepted($job_item_ids)) {
                  $this->requestAction($token, $request_guid, WordsOnlineStatus::ACCEPT_FAILED_STATUS);
                } else {
                  $this->updateWordsOnlineJobStatusByRequestGuid($request_guid, WordsOnlineConst::JOB_FINISHED);
                  $this->requestAction($token, $request_guid, WordsOnlineStatus::FINISHED_STATUS);
                }
              }
            }
          }
        }
      }
    } else {
      $status = "error";
    }
    return $status;
  }

  /**
   * Get Request import status.
   *
   * @param array $job_item_ids
   * List job item.
   */
  public function getRequestImportStatus($job_item_ids)
  {
    $status = "";
    $num_of_imported = 0;
    $item_count = count($job_item_ids);
    foreach ($job_item_ids as $jid) {
      $ji = JobItem::load($jid);
      if ($ji) {
        if ($this->isItemImported($ji) == TRUE) {
          $num_of_imported++;
        }
      }
    }
    if ($num_of_imported == $item_count) {
      $status = WordsOnlineStatus::IMPORTED_STATUS;
    } else if ($num_of_imported > 0) {
      $status = WordsOnlineStatus::PARTIALLY_IMPORTED;
    }
    return $status;
  }

  /**
   * Is All Job Item Accepted.
   *
   * @param array $job_items
   * List AggregatedJobItem job item.
   */
  public function isAllJobItemAccepted($job_items)
  {
    return (bool)!array_filter($job_items, function ($item) {
      return $item->getState() != JobItemInterface::STATE_ACCEPTED && $item->getState() != JobItemInterface::STATE_ABORTED;
    });
  }

  /**
   * Get ids of records for checking status
   *
   * @param string $table_name
   * The name of table.
   */
  public function getRecordIdsForTranslate($table_name): array
  {
    $records = $this->database->select($table_name, "wol");
    $records->fields("wol", ['id']);
    $records->condition("wol.status", [WordsOnlineStatus::FINISHED_STATUS], "not in");
    $records->orderBy("wol.id", "DESC");
    $res = $records->execute()->fetchCol();
    return $res;
  }


  /**
   * Get records by job ids
   *
   * @param string $table_name
   * @id array $ids array of job ids
   * The name of table.
   */
  public function getRecordsByIds(string $table_name, array $ids): array
  {
    $records = $this->database->select($table_name, "wol");
    $records->fields("wol");
    $records->condition("wol.status", [WordsOnlineStatus::FINISHED_STATUS], "not in")->condition("wol.id", $ids, "IN")->orderBy("wol.id", "DESC");
    $res = $records->execute()->fetchAll();
    return $res;
  }

  /**
   * Get translators which are provided by WordsOnline Connector.
   */
  public function getWordsOnlineTranslators()
  {
    static $translators;
    if (isset($translators)) {
      return $translators;
    }
    $translator_manager = Drupal::service('plugin.manager.tmgmt.translator');
    $translators = array_keys(array_filter($translator_manager->getDefinitions(), function ($t) {
      return $t['provider'] == WordsOnlineConst::MODULE_ID;
    }));
    return $translators;
  }

  /**
   * Retry accept job item
   */
  public function retryAcceptTranslation(JobItem|JobItemInterface $job_item, string $request_guid, $times = 3, $in_cronjob = FALSE)
  {
    $i = 0;
    do {
      try {
        $is_accepted = $job_item->acceptTranslation();
        if ($is_accepted == TRUE) {
          break;
        }
      } catch (\Exception $e) {
        $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::ACCEPT_FAILED_STATUS, $e->getMessage() . "\nStackTrace:" . $e->getTraceAsString());
      }
      if ($in_cronjob == TRUE) {
        $retry_interval = WordsOnlineConfig::getRetryInterval();
        sleep($retry_interval);
      }
    } while ($i < $times);
  }

  /**
   ** Import translation by job item
   **/
  public function importTranslationByItem(JobItem|JobItemInterface $job_item, $translation)
  {
    $this->setXliffOptions($job);
    $xliff = $this->formatManager->createInstance("xlf");
    $job = $job_item->getJob();
    $messages = $this->getXliffError($job, $translation);
    if ($messages != NULL && !empty($messages) && $messages != "") {
      return $messages;
    }
    if ($data = $xliff->import($translation, FALSE)) {
      $job_item_id = $job_item->id();
      if (isset($data[$job_item_id])) {
        $decode_html_entity = WordsOnlineConfig::getDecodeHtmlEntity();
        $translation = $decode_html_entity ? $this->decodeHtmlEntityByItem($data[$job_item_id], $job_item_id) : $data[$job_item_id];
        $job_item->addTranslatedData($translation, [], TMGMT_DATA_ITEM_STATE_TRANSLATED);
        $job_item->addMessage("The translation has been received.");
      }
    }
    return "";
  }

  /**
   ** Check if a job item has translation imported
   **/
  public function isItemImported(JobItem|JobItemInterface $job_item): bool
  {
    return $job_item->getCountPending() == 0 && ($job_item->getCountTranslated() > 0 || $job_item->getCountAccepted() > 0 || $job_item->getCountReviewed() > 0);
  }


  /**
   * Get WordsOnline jobs.
   */
  public function getWolJobs(array $conditions = array()): array
  {
    $query = $this->database->select(WordsOnlineConst::JOB_TABLE, 'j')->fields('ji');
    foreach ($conditions as $key => $condition) {
      if (is_array($condition)) {
        $operator = $condition['operator'] ?? '=';
        $query->condition($key, $condition['value'], $operator);
      } else {
        $query->condition($key, $condition);
      }
    }
    return $query->execute()->fetchAll();
  }

  /**
   * Get WordsOnline job item.
   */
  public function getWolJobItem(int $item_id): object|bool
  {
    $query = $this->database->select(WordsOnlineConst::JOB_ITEMS_TABLE, 'ji')->fields('ji')->condition('ji.job_Item_id', $item_id);
    return $query->execute()->fetch();
  }

  /**
   * Get WordsOnline job items.
   */
  public function getWolJobItems(array $item_ids): array
  {
    $query = $this->database->select(WordsOnlineConst::JOB_ITEMS_TABLE, 'ji')->fields('ji')->condition('ji.job_Item_id', $item_ids, 'IN');
    return $query->execute()->fetchAll();
  }

  /**
   * Get aggreated job items by wol job id.
   */
  public function getAggregatedJobItems(int $job_id): array
  {
    $item_ids = $this->getWolJobItemIdsByWolJob($job_id);
    return $item_ids ? AggregatedJobItem::loadMultiple($item_ids) : [];
  }


  /**
   * Get list job items Id  by wol request guid.
   */
  public function getListJobItemsIdByRequestGuid(string $request_guid): array
  {
    $records = $this->getWolJobItemIdByRequestGuid($request_guid);
    $item_ids = [];
    foreach ($records as $r) {
      if (is_numeric($r->job_Item_id)) {
        array_push($item_ids, intval($r->job_Item_id));
      }

    }
    return $item_ids;
  }

  /**
   * Get Wol Job item Id by Request Guid
   */
  public function getWolJobItemIdByRequestGuid(string $request_guid)
  {
    $records = $this->database
      ->query(
        "SELECT ji.job_Item_id FROM wordsonline_connector_jobs_items ji JOIN wordsonline_connector_jobs j ON ji.wol_job_id = j.id WHERE j.request_guid =:request_guid",
        [":request_guid" => (string)$request_guid]
      )
      ->fetchAll();
    return $records;
  }

  /**
   * Get WordsOnline job items by wol job id.
   */
  public function getWolJobItemIdsByWolJob(int $job_id): array
  {
    $query = $this->database->select(WordsOnlineConst::JOB_ITEMS_TABLE, 'ji')->fields('ji', ['job_Item_id'])->condition('ji.wol_job_id', $job_id);
    return $query->execute()->fetchCol();
  }


  public function syncStatus(object $wol_job, string $request_guid)
  {
    $token = $this->getToken();
    $response = wordsonline_connector_request("Requests/$request_guid", "GET", [], $token);
    if ($response) {
      $response = json_decode($response);
    } else {
      return FALSE;
    }
    $new_wol_status = $wol_status = $wol_job->status;
    if (isset($response->result)) {
      $status = $response->result->status;
      $state = $response->result->state;
      switch ($state) {
        case WordsOnlineState::AUTOMATION_FAILED:
          if ($wol_status != WordsOnlineConst::JOB_AUTOMATION_FAILED) {
            $new_wol_status = WordsOnlineConst::JOB_AUTOMATION_FAILED;
          }
          break;
        case WordsOnlineState::PREPARINGQUOTE:
          break;
        case WordsOnlineState::PAYMENT_FAILED:
        case WordsOnlineState::QUOTE_SUBMITTED:
        case WordsOnlineState::QUOTE_CREATED:
          if ($wol_status != WordsOnlineConst::JOB_QUOTED) {
            $new_wol_status = WordsOnlineConst::JOB_QUOTED;
          }
          break;
        default:
          switch ($status) {
            case WordsOnlineStatus::UNPAID:
            case WordsOnlineStatus::QUOTE_SUBMITTED:
            case WordsOnlineStatus::IN_PROGRESS_STATUS:
              if ($wol_status != WordsOnlineConst::JOB_QUOTED) {
                $new_wol_status = WordsOnlineConst::JOB_QUOTED;
              }
              break;
            case WordsOnlineStatus::IMPORTED_STATUS:
              if ($wol_status == WordsOnlineConst::JOB_DELIVERED) {
                $new_wol_status = WordsOnlineConst::JOB_IMPORTED;
              }
              break;
            case WordsOnlineStatus::FINISHED_STATUS:
            case WordsOnlineStatus::DELIVERED:
            case WordsOnlineStatus::IMPORT_FAIL_STATUS:
            case WordsOnlineStatus::IMPORT_FAILED_STATUS:
              if (!in_array($wol_status, [WordsOnlineConst::JOB_DELIVERED, WordsOnlineConst::JOB_IMPORTED, WordsOnlineConst::JOB_FINISHED, WordsOnlineConst::JOB_PARTIALLY_IMPORTED])) {
                $new_wol_status = WordsOnlineConst::JOB_DELIVERED;
              }
              break;
          }
          return $new_wol_status;
      }
      if (!empty($new_wol_status) && $new_wol_status != $wol_status) {
        $this->updateWordsOnlineJobStatusById($wol_job->id, $new_wol_status);
      }
      return $new_wol_status;
    }
    return FALSE;
  }


  public function syncAfterDeliveryStatus(mixed $wol_job)
  {
    $item_ids = $this->getWolJobItemIdsByWolJob($wol_job->id);
    $new_wol_status = $wol_status = $wol_job->status;
    $items = JobItem::loadMultiple($item_ids);
    $job = Job::load($wol_job->job_id);
    if (!$job) {
      return $wol_status;
    }
    $not_imported_count = count(array_filter($items, function ($item) {
      return !$item->isAborted() && !$this->isItemImported($item);
    }));
    $imported_count = count(array_filter($items, function ($item) {
      return !$item->isAborted() && $this->isItemImported($item);
    }));
    $not_finished_count = count(array_filter($items, function ($item) {
      return !$item->isAborted() && !$item->isAccepted();
    }));
    $request_guid = $wol_job->request_guid;
    $job_finished = $job->getState() == WordsOnlineConst::STATE_FINISHED || $job->isContinuous();

    switch ($wol_status) {
      case WordsOnlineConst::JOB_DELIVERED:
        if ($not_imported_count == 0) {
          $new_wol_status = WordsOnlineConst::JOB_IMPORTED;
          $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::IMPORTED_STATUS);
          if ($not_finished_count == 0 && $job_finished) {
            $new_wol_status = WordsOnlineConst::JOB_FINISHED;
            $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::FINISHED_STATUS);
          }
        } else if ($imported_count > 0) {
          $new_wol_status = WordsOnlineConst::JOB_PARTIALLY_IMPORTED;
          // $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::PARTIALLY_IMPORTED);
        }
        break;
      case WordsOnlineConst::JOB_IMPORTED:
        if ($not_finished_count == 0 && $job_finished) {
          $new_wol_status = WordsOnlineConst::JOB_FINISHED;
          $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::FINISHED_STATUS);
        }
        break;
      case WordsOnlineConst::JOB_PARTIALLY_IMPORTED:
        if ($not_imported_count == 0) {
          $new_wol_status = WordsOnlineConst::JOB_IMPORTED;
          $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::IMPORTED_STATUS);
          if ($not_finished_count == 0 && $job_finished) {
            $new_wol_status = WordsOnlineConst::JOB_FINISHED;
            $this->requestAction($this->getToken(), $request_guid, WordsOnlineStatus::FINISHED_STATUS);
          }
        }
        break;
    }
    if (!empty($new_wol_status) && $new_wol_status != $wol_status) {
      $this->updateWordsOnlineJobStatusById($wol_job->id, $new_wol_status);
    }
    return $new_wol_status;
  }

  /**
   * Make new request
   *
   * @param JobInterface $job
   * Translation job.
   */
  public function makeNewRequest($job)
  {
    $job_items = array_values($job->getItems());
    $this->makeRequest($job_items, $job->id());
  }

  /**
   * Retry create request.
   *
   * @param int $job_id
   * The job id.
   * @param int $id
   * The wordsonline job id.
   */
  public function retryCreateRequest($job_id, $id)
  {
    $job = Job::load($job_id);
    $job_items = array_values($job->getItems());
    $this->makeRequest($job_items, $job_id, $id);
  }

  /**
   * Make create request to wordsonline.
   *
   * @param array $job_items
   * List job item.
   * @param int $job_id
   * The job id
   * @param int $wol_job_id
   * The wol job id.
   */
  public function makeRequest(array $job_items, int $job_id, int $wol_job_id = 0)
  {
    $job = Job::load($job_id);
    $this->setXliffOptions($job);
    $params = $this->getParams($job);
    if (isset($params['request_name']) && preg_match('/[\\\\\\/:*?"<>|]/', $params['request_name'])) {
      throw new TMGMTException("The request name contains invalid characters: \\ / : * ? \" < > |");
    }
    $is_continuous = $job->isContinuous();
    $resultInserts = $this->insertOrUpdateNewWordsOnlineJob($params, $job_id, $is_continuous, $wol_job_id);
    $wol_job_items = $resultInserts["wol_job_items"];
    $wol_job_id = $resultInserts["wol_job_id"];
    $params['wol_job_id'] = $wol_job_id;
    $xliff = $this->formatManager->createInstance('xlf', $job->getSetting('format_configuration'));
    $xliff_content = $xliff->export($job);
    $file_zip = "Job{$job_id}.zip";
    $zip_files = [];
    $path = "public://";
    $source_lang = $job->getSourceLangcode();
    $target_lang = $job->getTargetLangcode();
    foreach ($job_items as $job_item) {
      if ($job_item->getState() == JobItemInterface::STATE_ABORTED) {
        continue;
      }
      $job_item_id = $job_item->id();
      if ($wol_job_id != 0 && count($wol_job_items) > 0) {
        $is_exits = FALSE;
        foreach ($wol_job_items as $wji) {
          if ($wji->job_Item_id == $job_item_id) {
            $is_exits = TRUE;
            break;
          }
        }
        if ($is_exits == FALSE) {
          continue;
        }
      }
      $xliff = $this->formatManager->createInstance('xlf', $job->getSetting('format_configuration'));
      $conditions = ['tjiid' => ['value' => $job_item_id]];
      $xliff_content = $xliff->export($job, $conditions);
      // Build a file name.
      $file_name = "Job_{$job_id}_{$job_item_id}.xlf";
      array_push($zip_files, "public://Job_{$job_id}_{$job_item_id}.xlf");
      if ($this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY)) {
        $version = (float)(str_replace("-dev", "", Drupal::VERSION));
        if ($version >= 9.3) {
          $this->fileRepository->writeData($xliff_content, $path . $file_name);
        } else {
          file_save_data($xliff_content, $path . $file_name);
        }
      }

      $fields["job_id"] = $job_id;
      $fields["wol_job_id"] = $wol_job_id;
      $fields["job_Item_id"] = $job_item_id;
      //$nid = $job_item->getItemId();
      $fields["file_name"] = $file_name;
      $fields["file_path"] = $path . $file_name;
      $fields["source_language"] = $source_lang;
      $fields["target_language"] = $target_lang;

      $this->insertOrUpdateNewWordsOnlineJobItems($wol_job_items, $job_item_id, $wol_job_id, $fields);
      if ($is_continuous) {
        $job_item->active();
      }
    }
    $this->createZip($zip_files, $path . $file_zip);
    $token = $this->getToken();
    $is_created = FALSE;
    if ($token != NULL) {
      $files = $this->uploadFiles($path . $file_zip, $token);
      if ($files["status"] >= 0) {
        $params['fileList'] = $files["result"];
        $is_created = $this->createOrder(WordsOnlineConst::CREATE_REQUEST_URL, 'POST', $params, $token, $is_continuous);
      }
    }
    $this->fileSystem->delete($path . $file_zip);
  }

  public function getRevisionId($job_item_id)
  {
    $job_item = JobItem::load($job_item_id);
    if ($job_item) {
      $job = $job_item->getJob();
      if ($job) {
        $langcode = $job->getSourceLangcode();
        $data = $this->database
          ->query(
            "SELECT r.vid FROM tmgmt_job_item ji JOIN node_revision r ON ji.item_id = r.nid  WHERE ji.tjiid =:tjiid AND r.revision_timestamp <= ji.changed AND r.langcode =:langcode   ORDER BY r.revision_timestamp DESC LIMIT 1",
            [
              ":langcode" => (string)$langcode,
              ":tjiid" => $job_item_id
            ]
          )
          ->fetchAssoc();
        if ($data != NULL) {
          if ($data["vid"] == NULL) {
            return $job_item->getItemId();
          }
          return $data["vid"];
        }
      }
    }

    return 0;
  }

  /**
   * Get Revision Timestamp
   *
   * @param int $revision_id
   * Revision Id.
   */
  public function getRevisionTimestamp($revision_id)
  {
    if ($revision_id == 0) return "";
    $data = $this->database
      ->query(
        "SELECT r.revision_timestamp FROM  node_revision r WHERE  r.vid=:vid  LIMIT 1",
        [
          ":vid" => $revision_id
        ]
      )
      ->fetchAssoc();
    if ($data != NULL) {
      if ($data["revision_timestamp"] != NULL) {
        $revision_timestamp = $data["revision_timestamp"];
        if ($revision_timestamp != '' && $revision_timestamp != 0 && is_numeric($revision_timestamp)) {
          $revision_tmp = date("Y-m-d H:m", intval($revision_timestamp));
          return $revision_tmp;
        }

      }
    }
    return "";
  }

  public function getParams($job)
  {
    $source_lang = $job->getSourceLangcode();
    $target_lang = $job->getTargetLangcode();
    $request_name = $job->getSetting('request_name');
    $projectId = $job->getSetting('project_key');
    if ($projectId == NULL) {
      $projectId = '';
    }

    if (!$request_name) {
      $request_name = wordsonline_connector_get_request_name();
    }
    $params['request_name'] = $request_name;
    $params['projectId'] = $projectId;
    $params['sourceLanguage'] = $source_lang;
    $params['targetLanguages'] = $target_lang;
    $params['contentTypeId'] = $job->getSetting('content_type');
    $params['serviceLevel'] = WordsOnlineHelper::getServiceLevel($job);
    $params['dueDate'] = $job->getSetting('due_date');
    $auto_approve_quote = $job->getSetting('auto_approve_quote');
    $params['auto_approve_quote'] = $auto_approve_quote == TRUE || $auto_approve_quote == 1 ? 'true' : 'false';
    return $params;
  }

  public function insertOrUpdateNewWordsOnlineJob($params, $job_id, $is_continuous, $wol_job_id = 0)
  {
    $orders["job_id"] = $job_id;
    $orders["request_guid"] = '';
    $orders["request_name"] = $params['request_name'];
    $orders["project_guid"] = $params['projectId'];
    $orders["status"] = WordsOnlineConst::JOB_NEW;
    $orders["type"] = $is_continuous ? 1 : 0;
    $wol_job_items = [];
    if ($wol_job_id == 0) {
      $orders["created"] = \Drupal::time()->getRequestTime();
      $wol_job_id = $this->database->insert(WordsOnlineConst::JOB_TABLE)->fields($orders)->execute();
    } else {
      $this->database->update(WordsOnlineConst::JOB_TABLE)->condition('id', $wol_job_id)->fields($orders)->execute();
      $records = $this->database->select(WordsOnlineConst::JOB_ITEMS_TABLE, "wol");
      $records->fields("wol");
      $records->condition("wol.wol_job_id", $wol_job_id)->orderBy("wol.job_Item_id", "DESC");
      $wol_job_items = $records->execute()->fetchAll();
    }
    $res["wol_job_id"] = $wol_job_id;
    $res["wol_job_items"] = $wol_job_items;
    return $res;
  }

  public function insertOrUpdateNewWordsOnlineJobItems($wol_job_items, $job_item_id, $wol_job_id, $fields)
  {
    $fields["updated"] = \Drupal::time()->getRequestTime();
    $is_exits = FALSE;
    foreach ($wol_job_items as $wji) {
      if ($wji->job_Item_id == $job_item_id) {
        $is_exits = TRUE;
        break;
      }
    }
    if ($is_exits) {
      $this->database->update(WordsOnlineConst::JOB_ITEMS_TABLE)->condition('wol_job_id', $wol_job_id)->condition('job_Item_id', $job_item_id)->fields($fields)->execute();
    } else {
      // $revision_id = Drupal::entityTypeManager()->getStorage('node')->getLatestRevisionId($nid);
      $revision_id = $this->getRevisionId($job_item_id);
      $fields["revision_id"] = $revision_id;
      $this->database->insert(WordsOnlineConst::JOB_ITEMS_TABLE)->fields($fields)->execute();
    }
  }

  /**
   * Creates a compressed zip file.
   *
   * @param array $files
   *   The path file array.
   * @param string $destination
   *   The target zip path.
   * @param bool $overwrite
   *   Is Overite.
   *
   * @return bool
   *   Check is succeess.
   */
  private function createZip(array $files = [], $destination = '', $overwrite = FALSE)
  {
    if (file_exists($destination) && !$overwrite) {
      $overwrite = TRUE;
    }
    $valid_files = [];
    if (is_array($files)) {
      foreach ($files as $file) {
        if (file_exists($file)) {
          $valid_files[] = $file;
        }
      }
    }
    if (count($valid_files)) {
      $zip = new ZipArchive();
      $destination = $this->fileSystem->realpath($destination);
      if ($zip->open($destination, $overwrite ? ZipArchive::OVERWRITE : ZipArchive::CREATE) !== TRUE) {
        return FALSE;
      }
      foreach ($valid_files as $file) {
        $realFile = $this->fileSystem->realpath($file);
        $zip->addFile($realFile, str_replace("public://", "", $file));
      }
      $zip->close();
      return file_exists($destination);
    } else {
      return FALSE;
    }
  }

  /**
   * Get quote.
   *
   * @param string $file_path
   *   Path of file.
   * @param string $token
   *   Token.
   *
   * @return string
   *   Receive data.
   */
  public function uploadFiles($file_path, $token)
  {
    $url = WordsOnlineConfig::getApiUrl() . 'Files';
    $options = [];
    $options['headers'] = ['Authorization' => 'Bearer ' . $token, 'Referer' => 'ClientAPI',];
    $options['multipart'] = [['name' => 'Source', 'filename' => $file_path, 'contents' => fopen($file_path, 'r'), 'headers' => ['Content-Type' => '<Content-type header>',],]];
    $options['timeout'] = 3600;
    $result = $this->client->post($url, $options);
    if ($result->getStatusCode() == 200) {
      $data = json_decode($result->getBody()->getContents(), TRUE);
      return $data;
    } else {
      return new WordsOnlineResponse(NULL, -1, "B0001", "error");
    }
  }

  /**
   * Create WordsOnline order.
   *
   * @param string $path
   *   The path of api.
   * @param string $method
   *   The method is used.
   * @param array $params
   *   List params.
   * @param string $token
   *   The token for authorization.
   *
   * @return bool
   *   Check is succeess.
   */
  public function createOrder($path, $method = 'POST', array $params = [], $token = NULL, $is_continuous = FALSE)
  {
    $options = [];
    $wol_job_id = $params['wol_job_id'];
    $url = WordsOnlineConfig::getApiUrl() . $path;
    try {
      $json = [];
      if ($is_continuous) {
        $json = ['fileList' => $params['fileList'], 'requestName' => $params['request_name'], 'projectId' => $params['projectId'], 'sourceLanguage' => $params['sourceLanguage'], 'targetLanguages' => [$params['targetLanguages'],], 'contentTypeId' => $params['contentTypeId'], 'serviceLevel' => $params['serviceLevel'], 'description' => 'Request from drupal', "clientRequestId" => "Default ID", "isAutoApprove" => $params['auto_approve_quote'],];
      } else {
        $json = ['fileList' => $params['fileList'], 'requestName' => $params['request_name'], 'projectId' => $params['projectId'], 'sourceLanguage' => $params['sourceLanguage'], 'targetLanguages' => [$params['targetLanguages'],], 'contentTypeId' => $params['contentTypeId'], 'serviceLevel' => $params['serviceLevel'], 'description' => 'Request from drupal', "clientRequestId" => "Default ID", "isAutoApprove" => $params['auto_approve_quote'], "dueDate" => $params['dueDate'],];
      }
      $response = $this->client->post($url, ['headers' => ['Authorization' => 'Bearer ' . $token, 'Accept' => 'application/json', 'Referer' => 'ClientAPI',], 'json' => $json, 'timeout' => 3600,]);
    } catch (RequestException $e) {
      $this->database->update(WordsOnlineConst::JOB_TABLE)->condition('id', $wol_job_id)->fields(['status' => WordsOnlineConst::JOB_CREATION_FAILED])->execute();
      $this->logger->error($e->getMessage());
      $this->messenger->addError(WordsOnlineMessage::FAILED_TO_CREATE_NEW_ORDER);
      return FALSE;
    }

    $received_data = $response->getBody()->getContents();
    $err_msg = WordsOnlineMessage::FAILED_TO_CREATE_NEW_ORDER;
    if ($response->getStatusCode() != 200) {
      $this->messenger->addError(WordsOnlineMessage::FAILED_TO_CREATE_NEW_ORDER);
      return FALSE;
    }
    $orderResponse = json_decode($received_data, TRUE);
    if ($orderResponse["status"] == 1) {
      $this->database->update(WordsOnlineConst::JOB_TABLE)->condition('id', $wol_job_id)->fields(['status' => WordsOnlineConst::JOB_CREATED, 'request_guid' => $orderResponse["result"],])->execute();

      $this->messenger->addMessage(WordsOnlineMessage::NEW_ORDER_CREATED);
      return TRUE;
    } else {
      $this->database->update(WordsOnlineConst::JOB_TABLE)->condition('id', $wol_job_id)->fields(['status' => WordsOnlineConst::JOB_CREATION_FAILED])->execute();
      if ($orderResponse != NULL && $orderResponse['message'] != NULL) {
        $err_msg = $err_msg . "\r\n" . $orderResponse['message'];
      }
      $this->messenger->addError($err_msg);
    }
    return FALSE;
  }

  /**
   * Import
   *
   * @param int $job_id
   * Job id.
   * @param int $job_item_id
   * Job  item id.
   * @param string $request_guid
   * Request guid.
   */
  public function importForJobItem($job_id, $job_item_id, $request_guid)
  {
    $token = $this->getToken();
    $status = "";
    if ($token != NULL) {
      $files = wordsonline_connector_get_request_files_list($request_guid, $token);
      if ($files != NULL) {
        if ($files->result != NULL && count($files->result) > 0) {
          $status = "downloaded";
          $job = Job::load($job_id);
          if ($job) {
            $job_item = JobItem::load($job_item_id);
            $is_import = FALSE;
            $translator = $job->getTranslator();
            $is_auto = $translator->isAutoAccept();
            $job_item_ids = $this->getListJobItemsIdByRequestGuid($request_guid);
            foreach ($files->result as $fi) {
              $file_guid = $fi->guid;
              $data = wordsonline_connector_download_file($request_guid, $file_guid, $token);
              $fName = "public://" . $fi->name;
              file_put_contents($fName, $data);
              $zip = new ZipHandle($fName, $this->archiver, $this->fileSystem);
              $fileContent = $zip->fileContent;
              $this->fileSystem->delete($fName);
              $is_has_error = FALSE;
              if ($fileContent != NULL && count($fileContent) > 0) {
                foreach ($fileContent as $key => $fcontent) {
                  if (str_contains($zip->fileNames[$key], "Job_{$job_id}_{$job_item_id}.xlf")) {
                    $msg = $this->importTranslationByItem($job_item, $fcontent);
                    if ($msg != "") {
                      $this->messenger->addError("an error occured when import job item {$job_item_id} with zip file {$fName}");
                      $is_has_error = TRUE;
                    }
                  }
                }
              }
              if ($is_has_error == TRUE) {
                continue;
              }
              $request_status = $this->getRequestImportStatus($job_item_ids);
              if ($request_status != "") {
                if ($request_status == WordsOnlineStatus::IMPORTED_STATUS) {
                  $this->requestAction($token, $request_guid, $request_status);
                }
                $local_wo_status = $request_status == WordsOnlineStatus::IMPORTED_STATUS ? WordsOnlineConst::JOB_IMPORTED : WordsOnlineConst::JOB_PARTIALLY_IMPORTED;
                $this->updateWordsOnlineJobStatusByRequestGuid($request_guid, $local_wo_status);
                $is_import = TRUE;
              }
            }
            if ($is_import) {
              $status = "success";
            }
          }
        }
      }
    } else {
      $status = "error";
    }
    return $status;
  }
}
