<?php

namespace Drupal\wordsonline_connector\Common;

use Drupal;

/**
 * WordsOnline Config Helper
 */
class WordsOnlineConfig
{
  /**
   * Default api url
   */
  const DEFAULT_API_URL = "https://api.wordsonline.com/v5/";
  /**
   * Default settings key
   */
  const SETTINGS_KEY = "wordsonline_connector.settings";
  /**
   * Setting key: API URL
   */
  const API_URL = "api_url";
  /**
   * Setting key: retry interval
   */
  const RETRY_INTERVAL = "retry_interval";
  /**
   * Setting key: decode html entity
   */
  const DECODE_HTML_ENTITY = "decode_html_entity";

  /**
   * Setting key: XLIFF format
   */
  const XLIFF_FORMAT = "xliff_format";

  /**
   * @var string Cached api URL
   */
  static string $API_URL;

  /**
   * Get API base URL
   * @return string base URL
   */
  public static function getApiUrl(): string
  {
    if (isset(self::$API_URL) && !empty(self::$API_URL)) {
      return self::$API_URL;
    }
    $url = Drupal::config(self::SETTINGS_KEY)->get(self::API_URL);
    if (!empty($url)) {
      self::$API_URL = $url;
      return self::$API_URL;
    }
    return self::DEFAULT_API_URL;
  }

  /**
   * Set API URL
   * @param string $url
   * @return void
   */
  public static function setApiUrl(string $url): void
  {
    if (!empty($url)) {
      $config = Drupal::configFactory()->getEditable(self::SETTINGS_KEY);
      $config->set(self::API_URL, $url)->save();
      self::$API_URL = $url;
    }
  }

  /**
   * Get retry interval
   * @return int Retry interval in seconds
   */
  public static function getRetryInterval(): int
  {
    $retry_interval = Drupal::config(self::SETTINGS_KEY)->get(self::RETRY_INTERVAL);
    if (is_numeric($retry_interval)) {
      return (int)$retry_interval;
    }
    return 60;
  }

  /**
   * Set retry interval
   * @param int $interval seconds
   * @return void
   */
  public static function setRetryInterval(int $interval): void
  {
    $interval = min(max(0, $interval), 120);
    $config = Drupal::configFactory()->getEditable(self::SETTINGS_KEY);
    $config->set(self::RETRY_INTERVAL, $interval)->save();
  }

  /**
   * Get retry interval
   * @return int Retry interval in seconds
   */
  public static function getDecodeHtmlEntity(): bool
  {
    $decode_html_entity = Drupal::config(self::SETTINGS_KEY)->get(self::DECODE_HTML_ENTITY);
    if (isset($decode_html_entity)) {
      return (bool)$decode_html_entity;
    }
    return true;
  }

  /**
   * Set retry interval
   * @param int $interval seconds
   * @return void
   */
  public static function setDecodeHtmlEntity(bool $value): void
  {
    $config = Drupal::configFactory()->getEditable(self::SETTINGS_KEY);
    $config->set(self::DECODE_HTML_ENTITY, $value)->save();
  }

  /**
   * Get XLIFF format
   * @return string XLIFF format
   */
  public static function getXliffFormat(): string
  {
    $xliff_format = Drupal::config(self::SETTINGS_KEY)->get(self::XLIFF_FORMAT);
    if (!empty($xliff_format)) {
      return $xliff_format;
    }
    return 'xliff_cdata';
  }

  /**
   * Set XLIFF format
   * @param string $format XLIFF format
   * @return void
   */
  public static function setXliffFormat(string $format): void
  {
    if(in_array($format, ['xliff_cdata', 'xliff_processing', 'html_encoded'])) {
      $config = Drupal::configFactory()->getEditable(self::SETTINGS_KEY);
      $config->set(self::XLIFF_FORMAT, $format)->save();
    }
  }
}
