<?php

namespace Drupal\wordsonline_connector\Plugin\QueueWorker;

use Drupal;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\SourceManager;
use Drupal\wordsonline_connector\Common\WordsOnlineConfig;
use Drupal\wordsonline_connector\WordsOnlineConnectorManager;
use Drupal\wordsonline_connector\WordsOnlineConst;
use Drupal\wordsonline_connector\WordsOnlineCronJobManager;
use Drupal\wordsonline_connector\WordsOnlinePendingJobStatus;
use Drupal\wordsonline_connector\WordsOnlineTMUpdateStatus;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes requests to update the Wordsonline Connector.
 *
 * @QueueWorker(
 *   id = "wol_request_create_queue",
 *   title = @Translation("WordsOnline Connector Create Queue"),
 *   cron = {
 *     "time" = 1800,
 *     "description" = @Translation("The WordsOnline Jobs Create Queue Worker periodically retrieves the pending job items from the WordsOnline Connector and creates new jobs in the Drupal Tmgmt system.")
 *   }
 * )
 */
class CreateQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface
{
  /**
   * The WordsOnline Connector manager.
   *
   * @var WordsOnlineConnectorManager
   */
  protected WordsOnlineConnectorManager $wolManager;

  /**
   * The WordsOnline CronJob manager.
   *
   * @var WordsOnlineCronJobManager
   */
  protected WordsOnlineCronJobManager $wolCronjobManager;

  /**
   * The database connection.
   *
   * @var Connection
   */
  protected Connection $database;

  /**
   * The source plugin manager.
   *
   * @var SourceManager
   */
  protected $sourcePluginManager;

  /**
   * The cache
   * @var Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The constructor.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, WordsOnlineConnectorManager $wol_manager, WordsOnlineCronJobManager $wol_cronjob_manager, Connection $database, SourceManager $source_manager, CacheBackendInterface $cache)
  {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->wolManager = $wol_manager;
    $this->wolCronjobManager = $wol_cronjob_manager;
    $this->database = $database;
    $this->sourcePluginManager = $source_manager;
    $this->cache = $cache;
  }

  /**
   * Check if we should create continuous job item of given job for given entity
   */
  public function shouldCreateContinuousItem(Job $job, $entity, $plugin, $item_type, $item_id)
  {
    $continuous_settings = $job->getContinuousSettings();
    $translation_manager = \Drupal::service('content_translation.manager');
    $translation = $entity->hasTranslation($job->getTargetLangcode()) ? $entity->getTranslation($job->getTargetLangcode()) : NULL;

    if ($entity && $entity->getEntityType()->hasKey('bundle')) {
      // The entity type has bundles, check both the entity type setting and
      // the bundle.
      if (!empty($continuous_settings[$plugin][$item_type]['bundles'][$entity->bundle()]) && !empty($continuous_settings[$plugin][$item_type]['enabled'])) {
        return TRUE;
      }
    } // No bundles, only check entity type setting.
    elseif (!empty($continuous_settings[$plugin][$item_type]['enabled'])) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * create queue worker.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('plugin.manager.tmgmt.wordsonline'), $container->get('wordsonline_connector.cronjob_manager'), $container->get('database'), $container->get('plugin.manager.tmgmt.source'), $container->get('cache.wordsonline_connector'));
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data)
  {

    if (!is_int($data) && !is_string($data)) {
      return;
    }

    $logger = Drupal::logger(WordsOnlineConst::MODULE_ID);
    $logger->notice('Start processing create queue item: @data', ['@data' => $data]);
    // Get the job item from the database.
    $item = $this->getPendingItem($data);
    if (!$item) {
      $logger->notice('Processing create queue item: cannot find a pending job item of ID @data', ['@data' => $data]);
      return;
    }

    // Check status
    if ($item->is_processed != WordsOnlinePendingJobStatus::AWAIT_CREATING) {
      $logger->notice('Processing create queue item: the status of pending item @data is not AWAIT_CREATING, skip creating', ['@data' => $data]);
      return;
    }

    $job_id = $item->job_id;
    $plugin = $item->plugin;
    $item_type = $item->item_type;
    $item_id = $item->item_id;
    $job = Job::load($job_id);
    if (!$job) {
      $logger->warning('Processing create queue item: the original continuous job (job_id = @job_id) of pending job item @data is not existed anymore, set status to JOB_DELETED', ['job_id' => $job_id, '@data' => $data]);
      $this->jobDeleted($data);
      return;
    }
    $entity = $this->sourcePluginManager->createInstance($plugin)->load($item_type, $item_id);
    if ($entity instanceof ContentEntityInterface && $entity->isTranslatable() && !$entity->getEntityType()->get('entity_revision_parent_type_field')) {
      // Check if there is ongoing item
      $most_recent_item = $this->wolCronjobManager->getMostRecentItemByLanguage($job->getSourceLanguage()->getId(), $job->getTargetLanguage()->getId(), $plugin, $item_type, $item_id);
      if ($most_recent_item) {
        $logger->notice('Processing create queue item @data: there is still a ongoing item of the content, will check in next round', ['@data' => $data]);
        $can_create_request = false;
      } else {
        // Check translation outdated && enabled
        $should_create_item = $this->shouldCreateContinuousItem($job, $entity, $plugin, $item_type, $item_id);
        if (!$should_create_item) {
          $logger->warning('Processing create queue item @data: shouldCreateContinuousItem = false, the translation exists but is up to date', ['@data' => $data]);
          $can_create_request = false;
        } else {
          // Get previous job of this item
          $item = $this->wolCronjobManager->getPreviousAggregatedItem($plugin, $item_type, $item_id, $job->getSourceLanguage()->getId(), $job->getTargetLanguage()->getId());
          $previous_request_guid = $item ? $item->getRequestGuid() : null;

          // Get TM update status and retry
          $retry_times = 0;
          $retry_interval = WordsOnlineConfig::getRetryInterval();
          while ($retry_times < 3) {
            $tm_update_status = $previous_request_guid ? wordsonline_connector_get_tm_update_state($previous_request_guid) : null;
            if ($tm_update_status == WordsOnlineTMUpdateStatus::FAILED) {
              $err_msg = "The TM update status is failed for request {$previous_request_guid}, the continuous job item cannot be created currently, please contact WordsOnline support for assistance";
              $logger->warning('Processing create queue item @data: ' . $err_msg, ['@data' => $data]);
              Drupal::messenger()->addError($err_msg);
              $this->tmUpdateFailed($data);
              return;
            } else if ($tm_update_status == WordsOnlineTMUpdateStatus::TIMEOUT) {
              $err_msg = "The TM update status is timeout for request {$previous_request_guid}, the continuous job item cannot be created currently, please contact WordsOnline support for assistance";
              $logger->warning('Processing create queue item @data: ' . $err_msg, ['@data' => $data]);
              Drupal::messenger()->addError($err_msg);
              $this->tmUpdateTimeout($data);
              return;
            }
            $can_create_request = !$previous_request_guid || $tm_update_status == WordsOnlineTMUpdateStatus::COMPLETED;
            if ($can_create_request) break;
            $retry_times++;
            $logger->notice('Processing create queue item @data: retry_times = @times , will wait @retry_interval seconds to check TM update status again', ['@data' => $data, '@times' => $retry_times, '@retry_interval' => $retry_interval]);
            sleep($retry_interval);
          }
        }
      }
      if ($can_create_request) {
        $item = $this->createJobItem($data, $job, $plugin, $item_type, $item_id);
        $logger->info('Processing create queue item @data: a new continuous job item $item_id has been created from the pending job item.', ['@data' => $data, '@item_id' => $item->id()]);
      } else {
        $this->wolCronjobManager->clearCreateQueuedCache($data);
      }
    } else {
      $this->contentDeleted($data);
      $logger->warning('Processing create queue item @data: cannot find the entity or the entity is not translatable anymore, set status to CONTENT_DELTED', ['@data' => $data]);
    }


  }

  /**
   * Get Pending Item
   * @param int|string $id
   * @return object|array|false
   */
  protected function getPendingItem(int|string $id): object
  {
    $query = $this->database->select(WordsOnlineConst::PENDING_JOB_ITEMS_TABLE, 'p')->fields('p')->condition('id', $id)->execute();
    return $query->fetch();
  }

  /**
   * Mark pending item as job deleted
   * @param $id
   * @return void
   */
  protected function jobDeleted($id)
  {
    $this->wolCronjobManager->clearCreateQueuedCache($id);
    $this->setStatus($id, WordsOnlinePendingJobStatus::JOB_DELETED);
  }

  /**
   * Update pending item status
   * @param $id
   * @param $status
   * @param $job_item_id
   * @return void
   */
  protected function setStatus($id, $status, $job_item_id = null)
  {
    $this->database->update(WordsOnlineConst::PENDING_JOB_ITEMS_TABLE)->condition('id', $id)->fields(['is_processed' => $status, 'job_item_id' => $job_item_id])->execute();
  }

  /**
   * Create a continuous job item
   * @param string|int $id
   * @param Job $job
   * @param $plugin
   * @param $item_type
   * @param $item_id
   * @return Drupal\tmgmt\JobItemInterface
   * @throws Drupal\tmgmt\TMGMTException
   */
  protected function createJobItem(string|int $id, Job $job, $plugin, $item_type, $item_id)
  {
    $item = $job->addItem($plugin, $item_type, $item_id);
    $this->created($id, $item->id());
    return $item;
  }

  /**
   * Mark pending item as created
   * @param $id
   * @param $job_item_id
   * @return void
   */
  protected function created($id, $job_item_id)
  {
    $this->wolCronjobManager->clearCreateQueuedCache($id);
    $this->setStatus($id, WordsOnlinePendingJobStatus::ITEM_CREATED, $job_item_id);
  }

  /**
   * Mark pending item as content deleted
   * @param $id
   * @return void
   */
  protected function contentDeleted($id)
  {
    $this->wolCronjobManager->clearCreateQueuedCache($id);
    $this->setStatus($id, WordsOnlinePendingJobStatus::CONTENT_DELETED);
  }

  /**
   * Mark pending item as job deleted
   * @param $id
   * @return void
   */
  protected function tmUpdateTimeout($id)
  {
    $this->wolCronjobManager->clearCreateQueuedCache($id);
    $this->setStatus($id, WordsOnlinePendingJobStatus::TM_UPDATE_TIMEOUT);
  }

  /**
   * Mark pending item as job deleted
   * @param $id
   * @return void
   */
  protected function tmUpdateFailed($id)
  {
    $this->wolCronjobManager->clearCreateQueuedCache($id);
    $this->setStatus($id, WordsOnlinePendingJobStatus::TM_UPDATE_FAILED);
  }

  /**
   * Requeue Item
   * @param $id
   * @return void
   */
  protected function requeueItem($id)
  {
    $cid = WordsOnlineCronJobManager::CREATE_QUEUE_ITEM_PREFIX . $id;
    Drupal::cache(WordsOnlineConst::MODULE_ID)->set($cid, true, strtotime('+1 day'));
    Drupal::queue(WordsOnlineConst::WOL_CREATE_QUEUE)->createItem($id);
  }
}
