<?php

namespace Drupal\wordsonline_connector\Plugin\QueueWorker;

use Drupal;
use Drupal\Core\Archiver\ArchiverManager;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\wordsonline_connector\Common\ZipHandle;
use Drupal\wordsonline_connector\Entity\AggregatedJobItem;
use Drupal\wordsonline_connector\WordsOnlineConnectorManager;
use Drupal\wordsonline_connector\WordsOnlineConst;
use Drupal\wordsonline_connector\WordsOnlineCronJobManager;
use Exception;
use Symfony\Component\DependencyInjection\ContainerInterface;


/**
 * Processes requests to update the Wordsonline Connector.
 *
 * @QueueWorker(
 *   id = "wol_request_update_queue",
 *   title = @Translation("WordsOnline Connector Update Queue"),
 *   cron = {
 *     "time" = 1800,
 *     "description" = @Translation("The WordsOnline Jobs Update Queue Worker periodically retrieves the latest status updates for WordsOnline Requests and updates them in the WordsOnline Connector's Jobs. It also periodically executes Auto Import and Auto Accept actions.")
 *   }
 * )
 */
class UpdateQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface
{
  /**
   * The WordsOnline Connector manager.
   *
   * @var WordsOnlineConnectorManager
   */
  protected WordsOnlineConnectorManager $wolManager;

  /**
   * The WordsOnline CronJob manager.
   *
   * @var WordsOnlineCronJobManager
   */
  protected WordsOnlineCronJobManager $wolCronjobManager;

  /**
   * The database connection.
   *
   * @var Connection
   */
  protected Connection $database;

  /**
   * Archiver.
   *
   * @var ArchiverManager
   */
  protected ArchiverManager $archiver;

  /**
   * The file system service.
   *
   * @var FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs a RequestUpdateQueueWorker object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, WordsOnlineConnectorManager $wol_manager, WordsOnlineCronJobManager $wol_cronjob_manager, Connection $database, ArchiverManager $archiver, FileSystemInterface $file_system)
  {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->wolManager = $wol_manager;
    $this->wolCronjobManager = $wol_cronjob_manager;
    $this->database = $database;
    $this->archiver = $archiver;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('plugin.manager.tmgmt.wordsonline'), $container->get('wordsonline_connector.cronjob_manager'), $container->get('database'), $container->get('plugin.manager.archiver'), $container->get('file_system'));
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data)
  {
    if (!is_array($data)) {
      return;
    }
    $logger = Drupal::logger(WordsOnlineConst::MODULE_ID);
    $logger->notice('Start processing update queue item: @data', ['@data' => implode(', ', $data)]);
    $records = $this->wolManager->getRecordsByIds(WordsOnlineConst::JOB_TABLE, $data);
    $records = array_reduce($records, function ($carry, $item) {
      $carry[$item->id] = $item;
      return $carry;
    }, []);


    foreach ($data as $job_id) {
      try {
        //$wol_manager->translate($records);
        if (!$this->wolManager->acquireSyncLock($job_id, 10)) {
          $logger->warning('Failed to acquire sync lock for job in update queue: @job_id', ['@job_id' => $job_id]);
          continue;
        }
        $aggregated_items = $this->wolManager->getAggregatedJobItems($job_id);
        $wol_job = $records[$job_id] ?? NULL;
        $job = $wol_job ? Job::load($wol_job->job_id) : NULL;
        if (!$wol_job || !$job) {
          $logger->warning('Job not found in database: @job_id', ['@job_id' => $job_id]);
          continue;
        }
        if (!$wol_job->request_guid) {
          $logger->notice('Job request guid not found: @job_id', ['@job_id' => $job_id]);
          continue;
        }
        // read settings
        $auto_accept = $job->getTranslator()->isAutoAccept();
        $auto_import = $auto_accept || $job->getSetting('auto_import');

        //sync status with WordsOnline
        $current_status = $this->wolManager->syncStatus($wol_job, $wol_job->request_guid);

        $importable_items = array_filter($aggregated_items, function ($item) {
          return $item->getState() == JobItemInterface::STATE_ACTIVE && !$this->wolManager->isItemImported($item->getJobItem());
        });
        $importable = $auto_import && in_array($current_status, [WordsOnlineConst::JOB_PARTIALLY_IMPORTED, WordsOnlineConst::JOB_DELIVERED]) && count($importable_items);

        $files = $importable ? $this->getFiles($wol_job->request_guid) : [];
        $file_contents = $importable ? $this->downloadFiles($wol_job->request_guid, $files) : [];
        /** @var AggregatedJobItem $aggregated_item */
        foreach ($aggregated_items as $aggregated_item) {
          if (!$aggregated_item) {
            continue;
          }
          $file_of_items = array_filter(array_keys($file_contents), function ($file_name) use ($aggregated_item) {
            return $aggregated_item->getFileName() == basename($file_name);
          });

          // auto import
          if ($importable && $file_of_items) {
            $file_name = reset($file_of_items);
            if ($file_name && $file_contents[$file_name]) {
              $job_item = $aggregated_item->getJobItem();
              try {
                $message = $this->wolManager->importTranslationByItem($job_item, $file_contents[$file_name]);
                if ($message) {
                  $logger->warning('Process xliff failed while importing @file, message: @message', ['$file' => $file_name, '@message' => $message]);
                  $this->wolManager->requestAction($this->wolManager->getToken(), $wol_job->request_guid, Drupal\wordsonline_connector\WordsOnlineStatus::IMPORT_FAIL_STATUS, "Error processing file $file_name, message: $message");
                }
              } catch (Exception $e) {
                // watchdog_exception(WordsOnlineConst::MODULE_ID, $e);
                $logger->error('Error importing file: @file, message: @message', ['@file' => $file_name, '@message' => $e->getMessage()]);
                $this->wolManager->requestAction($this->wolManager->getToken(), $wol_job->request_guid, Drupal\wordsonline_connector\WordsOnlineStatus::IMPORT_FAIL_STATUS, "Error import translation, file: $file_name, message: {$e->getMessage()}");
              }

            }
          }

          // retry auto accept
          if ($auto_accept && $aggregated_item->getState() == JobItemInterface::STATE_REVIEW) {
            $this->wolManager->retryAcceptTranslation($aggregated_item->getJobItem(), $wol_job->request_guid, 3, TRUE);
          }
        }

        // check after delivery status
        if (in_array($current_status, [WordsOnlineConst::JOB_IMPORTED, WordsOnlineConst::JOB_DELIVERED, WordsOnlineConst::JOB_PARTIALLY_IMPORTED])) {
          $current_status = $this->wolManager->syncAfterDeliveryStatus($wol_job);
        }

        // check items and job status
        if (!$job->isContinuous() && $this->wolCronjobManager->allJobItemsAcceptedOrAborted($job) && $job->getState() != JobInterface::STATE_FINISHED) {
          $job->set('state', JobInterface::STATE_FINISHED);
          $job->save();
        }
        $logger->notice('Finished processing request update: @job_id, job status: @job_status', ['@job_id' => $job_id, '@job_status' => $current_status]);
        $this->wolCronjobManager->clearUpdateQueuedCache($job_id);
      } catch (Exception $e) {
        // watchdog_exception(WordsOnlineConst::MODULE_ID, $e);
        $logger->error('Error processing update queue item, job_ids: @job_id, message: @message', ['@job_id' => $job_id, '@message' => $e->getMessage()]);
      } finally {
        $this->wolManager->releaseSyncLock($job_id);
      }
    }
  }

  function getFiles(string $request_guid)
  {
    $token = $this->wolManager->getToken();
    if (!$token) {
      return NULL;
    }
    // Get the files from the request
    $files = wordsonline_connector_get_request_files_list($request_guid, $token);
    if (!$files || !$files->result) {
      return [];
    }
    return $files->result;
  }

  function downloadFiles(string $request_guid, array $files): array
  {
    if (!$files) {
      return [];
    }
    $file_contents = [];
    foreach ($files as $file) {
      try {
        $result = $this->uncompressFile($request_guid, $file);
        if ($result) {
          $file_contents = array_merge($file_contents, $result);
        }
      } catch (Exception $e) {
        Drupal::logger('wordsonline_connector')->error('Error importing file: @file, message: @message', ['@file' => $file, '@message' => $e->getMessage()]);
      }
    }
    return $file_contents;
  }

  function uncompressFile(string $request_guid, $file): array
  {
    if (!$file || !$file->guid) {
      return [];
    }
    $file_guid = $file->guid;
    $data = wordsonline_connector_download_file($request_guid, $file_guid, $this->wolManager->getToken());
    $file_name = "public://" . $file->name;
    file_put_contents($file_name, $data);

    //unzip file
    $zip = new ZipHandle($file_name, $this->archiver, $this->fileSystem);
    $file_content = $zip->fileContent;
    $file_names = $zip->fileNames;

    $this->fileSystem->delete($file_name);
    if ($file_content) {
      return array_combine($file_names, $file_content);
    }
    return $file_content;
  }
}
