<?php

namespace Drupal\wordsonline_connector;

use Drupal\Core\Lock\LockBackendInterface;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\tmgmt\JobItemInterface;
use Drupal\wordsonline_connector\Entity\AggregatedJobItem;


class WordsOnlineCronJobManager
{
  /**
   * @var int
   * The number of records to process per item in the queue.
   */
  const RECORD_COUNT_PER_ITEM = 5;

  /**
   * @var string
   * The prefix for the cache IDs of the update queue items.
   */
  const UPDATE_QUEUE_ITEM_PREFIX = 'update_queue_item_';

  /**
   * @var string
   * The prefix for the cache IDs of the create queue items.
   */
  const CREATE_QUEUE_ITEM_PREFIX = 'create_queue_item_';

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * @var \Drupal\wordsonline_connector\WordsOnlineConnectorManager
   */
  protected $wolManager;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;


  /**
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $updateQueue;

  /**
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $createQueue;

  /**
   * The lock backend service.
   *
   * @var LockBackendInterface
   */
  protected $lock;

  /**
   * Constructs a WordsOnlineCronJobManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\wordsonline_connector\WordsOnlineConnectorManager $wolManager
   *   The WordsOnline connector manager.
   * @param \Drupal\Core\Lock\LockBackendInterface
   *   The lock backend.
   */
  public function __construct($database, $cache, $logger, $wolManager, $lock)
  {
    $this->database = $database;
    $this->cache = $cache;
    $this->wolManager = $wolManager;
    $this->logger = $logger;
    $this->updateQueue = \Drupal::queue(WordsOnlineConst::WOL_UPDATE_QUEUE);
    $this->createQueue = \Drupal::queue(WordsOnlineConst::WOL_CREATE_QUEUE);
    $this->lock = $lock;
  }


  /**
   * Enqueue items for update.
   *
   * This method retrieves record IDs for translation from the WordsOnline
   * connector manager, filters out already queued items, and creates queue
   * items for the remaining records.
   */

  public function enqueueUpdateItems()
  {
    $queue = $this->updateQueue;
    $record_ids = $this->wolManager->getRecordIdsForTranslate(WordsOnlineConst::JOB_TABLE);
    $record_cids = array_map(function ($id) {
      return self::UPDATE_QUEUE_ITEM_PREFIX . $id;
    }, $record_ids);
    $cache = $this->cache;
    $queued_cids = $cache->getMultiple($record_cids);
    // remove ids that are already in the queue, and set cache for the rest
    // to prevent them from being added again
    $record_ids = array_filter($record_ids, function ($id) use ($cache, $queued_cids) {
      $cid = self::UPDATE_QUEUE_ITEM_PREFIX . $id;
      if (isset($queued_cids[$cid])) {
        return FALSE;
      } else {
        $cache->set($cid, TRUE, strtotime('+1 day'));
        return TRUE;
      }
    });
    $chunks = array_chunk($record_ids, self::RECORD_COUNT_PER_ITEM);
    foreach ($chunks as $chunk) {
      $queue->createItem($chunk);
    }
  }

  public function getPreviousAggregatedItem($plugin, $item_type, $item_id, $source_language, $target_language)
  {
    $query = $this->database->select('tmgmt_job_item', 'ji');
    $query->join('tmgmt_job', 'j', 'j.tjid = ji.tjid');
    $query->condition('ji.plugin', $plugin)
      ->condition('ji.item_type', $item_type)
      ->condition('ji.item_id', $item_id)
      ->condition('j.translator', $this->wolManager->getWordsOnlineTranslators(), 'IN')
      ->condition('ji.state', [JobItemInterface::STATE_ABORTED, JobItemInterface::STATE_INACTIVE], 'NOT IN')
      ->condition('j.source_language', $source_language)
      ->condition('j.target_language', $target_language)
      ->orderBy('ji.tjiid', 'DESC')
      ->fields('ji', ['tjiid'])
      ->range(0, 1);
    $result = $query->execute()->fetchCol();
    if (!empty($result)) {
      return AggregatedJobItem::load(reset($result));
    }
    return NULL;
  }

  /**
   * Get the IDs of pending items.
   *
   * This method retrieves the IDs of items that are pending processing in the
   * WordsOnline connector's pending job items table.
   *
   * @return array
   *   An array of pending item IDs.
   */
  public function getPendingItemIds(): array
  {
    return $this->database->select(WordsOnlineConst::PENDING_JOB_ITEMS_TABLE, 'p')
      ->fields('p', ['id'])
      ->condition('is_processed', WordsOnlinePendingJobStatus::AWAIT_CREATING)
      ->execute()
      ->fetchCol();
  }

  /**
   * Get the most recent job item.
   *
   * @param int|string $job_id
   *  The job ID.
   * @param string $plugin
   *   The plugin name.
   * @param string $item_type
   *   The item type.
   * @param int $item_id
   *   The item ID.
   *
   * @return \Drupal\tmgmt\JobItemInterface|null
   *   The most recent job item or NULL if not found.
   */
  public function getMostRecentItem($job_id, $plugin, $item_type, $item_id)
  {
    $query = \Drupal::entityQuery('tmgmt_job_item')
      ->accessCheck(TRUE)
      ->condition('plugin', $plugin)
      ->condition('item_type', $item_type)
      ->condition('item_id', $item_id)
      ->condition('state', [JobItemInterface::STATE_ACTIVE, JobItemInterface::STATE_REVIEW, JobItemInterface::STATE_INACTIVE], 'IN')
      ->condition('tjid', $job_id)
      ->sort('tjiid', 'DESC')
      ->range(0, 1);
    $result = $query->execute();
    if (!empty($result)) {
      return JobItem::load(reset($result));
    }
    return NULL;
  }

  /**
   * Get the most recent job item by language.
   *
   * @param string $source_language
   *   The source language.
   * @param string $target_language
   *   The target language.
   * @param string $plugin
   *   The plugin name.
   * @param string $item_type
   *   The item type.
   * @param int $item_id
   *   The item ID.
   *
   * @return \Drupal\tmgmt\JobItemInterface|null
   *   The most recent job item or NULL if not found.
   */
  public function getMostRecentItemByLanguage($source_language, $target_language, $plugin, $item_type, $item_id)
  {
    $query = $this->database->select('tmgmt_job_item', 'ji');
    $query->join('tmgmt_job', 'j', 'j.tjid = ji.tjid');
    $query->condition('ji.plugin', $plugin)
      ->condition('ji.item_type', $item_type)
      ->condition('ji.item_id', $item_id)
      ->condition('ji.state', [JobItemInterface::STATE_ACTIVE, JobItemInterface::STATE_REVIEW, JobItemInterface::STATE_INACTIVE], 'IN')
      ->condition('j.source_language', $source_language)
      ->condition('j.target_language', $target_language)
      ->orderBy('ji.tjiid', 'DESC')
      ->fields('ji', ['tjiid'])
      ->range(0, 1);
    $result = $query->execute()->fetchCol();
    if (!empty($result)) {
      return JobItem::load(reset($result));
    }
    return NULL;
  }

  /**
   * Enqueue items for update.
   *
   * This method retrieves record IDs for translation from the WordsOnline
   * connector manager, filters out already queued items, and creates queue
   * items for the remaining records.
   */

  public function enqueueCreateItems()
  {
    $queue = $this->createQueue;
    $pending_ids = $this->getPendingItemIds();
    $pending_cids = array_map(function ($id) {
      return self::CREATE_QUEUE_ITEM_PREFIX . $id;
    }, $pending_ids);
    $cache = $this->cache;
    $queued_cids = $cache->getMultiple($pending_cids);
    // remove ids that are already in the queue, and set cache for the rest
    // to prevent them from being added again
    $pending_ids = array_filter($pending_ids, function ($id) use ($cache, $queued_cids) {
      $cid = self::CREATE_QUEUE_ITEM_PREFIX . $id;
      if (isset($queued_cids[$cid])) {
        return FALSE;
      } else {
        $cache->set($cid, TRUE, strtotime('+1 day'));
        return TRUE;
      }
    });
    foreach ($pending_ids as $id) {
      $queue->createItem($id);
    }
  }


  /**
   * Clear the queued cache for update queue.
   */
  public function clearUpdateQueuedCache(array|int|string $chunk_ids_or_id)
  {
    $this->clearsQueuedCache($chunk_ids_or_id, FALSE);
  }

  /**
   * Clear the queued cache for update queue.
   */
  public function clearCreateQueuedCache(array|int|string $chunk_ids_or_id)
  {
    $this->clearsQueuedCache($chunk_ids_or_id, TRUE);
  }


  /**
   * Clear the queued cache for a queue.
   */
  protected function clearsQueuedCache(array|int|string $chunk_ids_or_id, bool $create_or_update = FALSE)
  {
    $chunk_ids = is_array($chunk_ids_or_id) ? $chunk_ids_or_id : [$chunk_ids_or_id];
    $cache = $this->cache;
    $record_cids = array_map(function ($id) use ($create_or_update) {
      return ($create_or_update ? self::CREATE_QUEUE_ITEM_PREFIX : self::UPDATE_QUEUE_ITEM_PREFIX) . $id;
    }, $chunk_ids);
    $cache->deleteMultiple($record_cids);
  }

  /**
   * Check if a pending job item exists.
   *
   * @param string $plugin
   *   The plugin ID.
   * @param string $item_type
   *   The item type.
   * @param string $item_id
   *   The item ID.
   *
   * @return bool
   *   TRUE if the pending job item exists, FALSE otherwise.
   */
  public function pendingJobItemExists(int|string $job_id, string $plugin, string $item_type, string $item_id): bool
  {
    $query = $this->database->select(WordsOnlineConst::PENDING_JOB_ITEMS_TABLE, 'p')
      ->fields('p', ['id'])
      ->condition('job_id', $job_id)
      ->condition('plugin', $plugin)
      ->condition('item_type', $item_type)
      ->condition('item_id', $item_id)
      ->condition('is_processed', WordsOnlinePendingJobStatus::AWAIT_CREATING)
      ->range(0, 1);
    return (bool)$query->execute()->fetchField();
  }

  /**
   * Create a pending job item.
   *
   * @param \Drupal\tmgmt\JobItemInterface|\Drupal\tmgmt\JobItem $job_item
   *   The job item to create.
   */
  public function createPendingJobItem(JobItem|JobItemInterface $job_item)
  {
    $plugin = $job_item->getPlugin();
    $item_type = $job_item->getItemType();
    $item_id = $job_item->getItemId();
    $job_id = $job_item->getJobId();
    $this->createPendingJobItemById($job_id, $plugin, $item_type, $item_id);
  }

  /**
   * Create a pending job item by ids.
   *
   * @param \Drupal\tmgmt\JobItemInterface|\Drupal\tmgmt\JobItem $job_item
   *   The job item to create.
   */
  public function createPendingJobItemById(int|string $job_id, string $plugin, string $item_type, string $item_id)
  {
    $lock_key = "lock_create_pending_job_item_{$job_id}_{$plugin}_{$item_type}_{$item_id}";
    try {
      if (!$this->lock->acquire($lock_key)) {
        throw new \Exception('Could not acquire lock for creating pending job item');
      }
      if ($this->pendingJobItemExists($job_id, $plugin, $item_type, $item_id)) {
        return;
      }
      $this->database->insert(WordsOnlineConst::PENDING_JOB_ITEMS_TABLE)
        ->fields([
          'job_id' => $job_id,
          'item_id' => $item_id,
          'is_processed' => WordsOnlinePendingJobStatus::AWAIT_CREATING,
          'plugin' => $plugin,
          'item_type' => $item_type,
          'job_item_id' => NULL,
          'created' => \Drupal::time()->getRequestTime(),
        ])
        ->execute();
      \Drupal::logger(WordsOnlineConst::MODULE_ID)->info("Pending job item created for job {$job_id}, plugin: {$plugin}, item type: {$item_type}, item id: {$item_id}");
    } catch (\Exception $e) {
      $err_msg = 'Failed to create a pending job item';
      $this->logger->error($err_msg . ", plugin = {$plugin}, item_type = {$item_type}, item_id = {$item_id}, job_id = {$job_id}, error_message: " . $e->getMessage());
      \Drupal::messenger()->addError($err_msg);
    } finally {
      $this->lock->release($lock_key);
    }
  }


  /**
   * Check if all job items are accepted or aborted.
   *
   * @param \Drupal\tmgmt\Entity\Job $job
   *   The job entity.
   *
   * @return bool
   *   TRUE if all job items are accepted or aborted, FALSE otherwise.
   */

  public function allJobItemsAcceptedOrAborted(Job $job): bool
  {
    $items = $job->getItems();
    return (bool)!array_filter($items, function ($item) {
      return $item->getState() != JobItemInterface::STATE_ACCEPTED && $item->getState() != JobItemInterface::STATE_ABORTED;
    });
  }


}
