<?php


namespace Drupal\wordsonline_connector\Entity;

use Drupal\tmgmt\Entity\JobItem;
use Drupal\tmgmt\Entity\Job;
use Drupal\wordsonline_connector\WordsOnlineConst;

/**
 * Class AggregatedJobItem.
 *
 * @package Drupal\wordsonline_connector\Entity
 *
 * @property JobItemInterface $jobItem
 * @property object $wolJobItem
 */
class AggregatedJobItem
{

  /**
   * The job item.
   *
   * @var \Drupal\tmgmt\Entity\JobItem
   */
  protected JobItem $jobItem;

  /**
   * The WordsOnline job item.
   *
   * @var object
   */
  protected object $wolJobItem;

  /**
   * AggregatedJobItem constructor.
   *
   * @param \Drupal\tmgmt\Entity\JobItem $jobItem
   *   The job item.
   * @param object $wolJobItem
   *   The WordsOnline job item.
   */
  public function __construct($jobItem, $wolJobItem)
  {
    $this->jobItem = $jobItem;
    $this->wolJobItem = $wolJobItem;
  }

  /**
   * Get the job item.
   *
   * @return \Drupal\tmgmt\Entity\JobItem
   *   The job item.
   */
  public function getJobItem(): JobItem
  {
    return $this->jobItem;
  }

  /**
   * Get the WordsOnline job item.
   *
   * @return object
   *   The WordsOnline job item.
   */
  public function getWolJobItem()
  {
    return $this->wolJobItem;
  }

  /**
   * Get the WordsOnline job item ID.
   *
   * @return int
   *   The WordsOnline job item ID.
   */
  public function getWolJobItemId()
  {
    return $this->wolJobItem->id;
  }

  /**
   * Get the job item ID.
   *
   * @return int
   *   The job item ID.
   */
  public function getJobItemId()
  {
    return $this->jobItem->getItemId();
  }

  /**
   * Get the job.
   *
   * @return \Drupal\tmgmt\Entity\Job
   *   The job.
   */
  public function getJob(): Job
  {
    return $this->jobItem->getJob();
  }

  /**
   * Get the WordsOnline job ID.
   *
   * @return int
   *   The WordsOnline job ID.
   */
  public function getWolJobId()
  {
    return $this->wolJobItem->wol_job_id;
  }

  /**
   * Get the WordsOnline request GUID.
   *
   * @return string
   *  The WordsOnline request GUID.
   */
  public function getRequestGuid(): string|bool|null
  {
    $result = \Drupal::database()->select(WordsOnlineConst::JOB_TABLE, 'j')
      ->fields('j', ['request_guid'])
      ->condition('id', $this->wolJobItem->wol_job_id)
      ->execute();
    return $result->fetchField();
  }

  /**
   * Get the WordsOnline file name.
   *
   * @return string
   *   The WordsOnline file name.
   */
  public function getFileName(): string
  {
    return $this->wolJobItem->file_name;
  }

  /**
   * Get the job item state.
   *
   * @return string
   *   The job item state.
   */
  public function getState(): string
  {
    return $this->jobItem->getState();
  }

  /**
   * Load a job item by job item id (tmgmt).
   */

  public static function load($item_id): bool|self
  {
    $job_item = JobItem::load($item_id);
    $wol_job_item = \Drupal::service('plugin.manager.tmgmt.wordsonline')->getWolJobItem($item_id);
    if ($job_item && $wol_job_item) {
      return new self($job_item, $wol_job_item);
    }
    return FALSE;
  }

  /**
   * Load multiple job items by job item ids (tmgmt).
   *
   * @param array $item_ids
   *   An array of job item ids.
   *
   * @return array
   *   An array of AggregatedJobItem objects.
   */
  public static function loadMultiple(array $item_ids): array
  {
    $job_items = JobItem::loadMultiple($item_ids);
    $wol_job_items = \Drupal::service('plugin.manager.tmgmt.wordsonline')->getWolJobItems($item_ids);
    $wol_job_items = array_reduce($wol_job_items, function ($carry, $obj) {
      if ($obj->job_Item_id) {
        $carry[$obj->job_Item_id] = $obj;
      }
      return $carry;
    }, []);
    $result = [];
    foreach ($item_ids as $item_id) {
      if (isset($job_items[$item_id]) && isset($wol_job_items[$item_id])) {
        $result[$item_id] = new self($job_items[$item_id], $wol_job_items[$item_id]);
      } else {
        $result[$item_id] = FALSE;
      }
    }
    return $result;
  }

}
