<?php


namespace Drupal\wordsonline_connector\EventSubscriber;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\tmgmt\Events\ContinuousEvents;
use Drupal\tmgmt\Events\ShouldCreateJobEvent;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\wordsonline_connector\Plugin\tmgmt\Translator\WordsOnlineTranslator;
use Drupal\wordsonline_connector\WordsOnlineConnectorManager;
use Drupal\wordsonline_connector\WordsOnlineCronJobManager;
use Drupal\wordsonline_connector\WordsOnlineTMUpdateStatus;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Class ShouldCreateJobEventsSubscriber.
 *
 * @package Drupal\wordsonline_connector\EventSubscriber
 */
class ShouldCreateJobEventsSubscriber implements EventSubscriberInterface
{

  /**
   * {@inheritdoc}
   *
   * @return array
   *   The event names to listen for, and the methods that should be executed.
   */
  public static function getSubscribedEvents()
  {
    return [
      ContinuousEvents::SHOULD_CREATE_JOB => 'shouldCreateJob',
    ];
  }

  /**
   * @var \Drupal\wordsonline_connector\WordsOnlineConnectorManager
   *  The WordsOnline connector manager.
   */
  protected WordsOnlineConnectorManager $wolManager;

  /**
   * @var \Drupal\wordsonline_connector\WordsOnlineCronJobManager
   *  The WordsOnline cron job manager.
   */
  protected WordsOnlineCronJobManager $wolCronJobManager;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The constructor.
   *
   * @param \Drupal\wordsonline_connector\WordsOnlineConnectorManager $wolManager
   *  The WordsOnline connector manager.
   *
   * @param \Drupal\wordsonline_connector\WordsOnlineCronJobManager $wolCronJobManager
   * The WordsOnline cron job manager.
   */

  public function __construct(
    WordsOnlineConnectorManager $wol_manager,
    WordsOnlineCronJobManager   $wol_cronjob_manager,
    LoggerChannelInterface      $logger

  )
  {
    $this->wolManager = $wol_manager;
    $this->wolCronJobManager = $wol_cronjob_manager;
    $this->logger = $logger;
  }

  /**
   * React to a continuous job whether should be created.
   *
   * @param \Drupal\tmgmt\Events\ShouldCreateJobEvent $event
   *   Should create job event.
   */
  public function shouldCreateJob(ShouldCreateJobEvent $event)
  {
    // $config = $event->getConfig();
    // \Drupal::messenger()->addStatus('Saved config: ' . $config->getName());http://localhost:8011/admin/tmgmt/jobs
    $job = $event->getJob();

    // Check if the job is a WordsOnline job.
    if (!$job->getTranslator()->getPlugin() instanceof WordsOnlineTranslator) {
      return;
    }

    $plugin = $event->getPlugin();
    $item_type = $event->getItemType();
    $item_id = $event->getItemId();
    $job_id = $job->id();
    $item = $this->wolCronJobManager->getPreviousAggregatedItem($plugin, $item_type, $item_id, $job->getSourceLanguage()->getId(), $job->getTargetLanguage()->getId());
    $most_recent_item = $this->wolCronJobManager->getMostRecentItem($job_id, $plugin, $item_type, $item_id);
    //$event->shouldCreateItem() => translation enabled && translation is outdated
    if ($event->shouldCreateItem()) {
      $entity = \Drupal::service('plugin.manager.tmgmt.source')->createInstance($plugin)->load($item_type, $item_id);
      if ($entity->revision_translation_affected && $entity->revision_translation_affected->value) {
        if ($most_recent_item && $most_recent_item->getState() != JobItemInterface::STATE_INACTIVE) {
          $event->setShouldCreateItem(FALSE);
          $this->wolCronJobManager->createPendingJobItem($most_recent_item);
        } else {
          // Check if last request finished tm update
          $previous_request_guid = $item ? $item->getRequestGuid() : null;
          $tm_update_status = $previous_request_guid ? wordsonline_connector_get_tm_update_state($previous_request_guid) : null;
          if ($tm_update_status == null) return;
          switch ($tm_update_status) {
            case WordsOnlineTMUpdateStatus::COMPLETED:
              $event->setShouldCreateItem(TRUE);
              break;
            case WordsOnlineTMUpdateStatus::IN_PROGRESS:
              $event->setShouldCreateItem(FALSE);
              $err_msg = 'Error creating a continuous job item because the previous request is updating TM';
              $this->logger->error($err_msg) .
              \Drupal::messenger()->addError($err_msg);
              $this->wolCronJobManager->createPendingJobItemById($job_id, $plugin, $item_type, $item_id);
              break;
            case WordsOnlineTMUpdateStatus::FAILED:
              $err_msg = 'Error creating a continuous job item because the previous request failed to update TM, please contact WordsOnline support team for assistance';
              $this->logger->error($err_msg) .
              \Drupal::messenger()->addError($err_msg);
              $event->setShouldCreateItem(FALSE);
              break;
            case WordsOnlineTMUpdateStatus::TIMEOUT:
              $err_msg = 'Error creating a continuous job item because the previous TM update operation of this request is timeout, please contact WordsOnline support team for assistance';
              $this->logger->error($err_msg) .
              \Drupal::messenger()->addError($err_msg);
              $event->setShouldCreateItem(FALSE);
              break;
          }

        }
      } else {
        $event->setShouldCreateItem(FALSE);
      }
    }
  }

}
