<?php

namespace Drupal\wordsonline_connector\Plugin\tmgmt\Translator;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\TranslatorPluginBase;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\ClientInterface;
use Drupal\tmgmt\JobInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Drupal\wordsonline_connector\WordsOnlineConst;
use Drupal\wordsonline_connector\WordsOnlineMessage;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\node\Entity\Node;
use Drupal\wordsonline_connector\WordsOnlineConnectorManager;

/**
 * WordsOnline translation plugin controller.
 *
 * @TranslatorPlugin(
 *   id = "wordsonline",
 *   label = @Translation("WordsOnline"),
 *   description = @Translation("WordsOnline translator service."),
 *   ui = "Drupal\wordsonline_connector\WordsOnlineTranslatorUi",
 *   logo = "icons/wordsonline-startup.png",
 * )
 */
class WordsOnlineTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface
{

  use StringTranslationTrait;

  /**
   * Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * Database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * WordsOnline Connector Manager.
   *
   * @var \Drupal\wordsonline_connector\WordsOnlineConnectorManager
   */
  protected $wo_manager;

  /**
   * Constructs a WordsOnlineTranslator object.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   The Guzzle HTTP client.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   Messenger.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\wordsonline_connector\WordsOnlineConnectorManager $wo_manager
   * WordsOnline Connector Manager.
   */
  public function __construct(ClientInterface $client, Connection $database, Messenger $messenger, LoggerInterface $logger, array $configuration, $plugin_id, array $plugin_definition, WordsOnlineConnectorManager $wo_manager)
  {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->client = $client;
    $this->database = $database;
    $this->messenger = $messenger;
    $this->logger = $logger;
    $this->wo_manager = $wo_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    $client = $container->get('http_client');
    $database = $container->get('database');
    $messenger = $container->get('messenger');
    $logger = $container->get('logger.factory')
      ->get("wordsonline_connector");
    $wo_manager = $container->get('plugin.manager.tmgmt.wordsonline');
    return new static(
      $client,
      $database,
      $messenger,
      $logger,
      $configuration,
      $plugin_id,
      $plugin_definition,
      $wo_manager
    );
  }

  /**
   * {@inheritdoc}
   *
   * @param array $job_items
   *   List job item.
   */
  public function requestJobItemsTranslation(array $job_items)
  {
    /** @var \Drupal\tmgmt\Entity\Job $job */
    $job = reset($job_items)->getJob();
    if ($job->isContinuous()) {
      $this->wo_manager->makeRequest($job_items, $job->id());
      $submit_message = $this->getSubmitMessage();
      $job->submitted($submit_message);

    }
    return $job;
  }

  /**
   * Get submit message.
   */
  public function getSubmitMessage()
  {
    $submit_message = 'The translation job has been submitted. Please go to <a href="' .
      base_path() .
      'admin/tmgmt/wordsonline/requests' .
      '"> WordsOnline Translation Request</a> to see your quote.';
    return $submit_message;
  }

  /**
   * {@inheritdoc}
   *
   * @param Drupal\tmgmt\JobInterface $job
   *   Job.
   */
  public function requestTranslation(JobInterface $job)
  {
    try {
      $this->wo_manager->makeNewRequest($job);
      $submit_message = $this->getSubmitMessage();
      $job->submitted($submit_message);
    } catch (TMGMTException $e) {
      \Drupal::messenger()->addError('Error creating new request: '.$e->getMessage());
    }
  }


}
