<?php

namespace Drupal\workflow\Element;

use Drupal\Component\Utility\Html;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\workflow\Entity\Workflow;

/**
 * Provides a form element for the WorkflowTransitionForm and ~Widget.
 *
 * @see \Drupal\Core\Render\Element\FormElement
 * @see https://www.drupal.org/node/169815 "Creating Custom Elements"
 *
 * @FormElement("workflow_transition_timestamp")
 */
class WorkflowTransitionTimestamp extends FormElementBase {
  // @todo Extends FormElementBase

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = static::class;
    return [
      '#input' => TRUE,
      '#process' => [
        [$class, 'processTimestamp'],
        [$class, 'processAjaxForm'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $element['#default_value'];
    $timestamp = $transition->getTimestamp();

    if (!$input || !is_array($input)) {
      // Massage, normalize value after pressing Form button.
      // $element is also updated via reference.
      // Get the time from the default transition data.
      return $timestamp;
    }

    if ($transition->isExecuted()) {
      // Updating (comments of) existing transition (on Workflow History page).
      return $timestamp;
    }

    $is_scheduled = (bool) ($input['scheduled'] ?? FALSE);
    if (!$is_scheduled) {
      // Fetch current timestamp for non-scheduled transitions.
      $timestamp = $transition->getDefaultRequestTime();
      return $timestamp;
    }

    // Fetch $timestamp from widget for scheduled transitions.
    $date_time = $input['scheduled_datetime'] ?? [];
    $timestamp = $date_time['datetime'] ?? '';
    if (is_array($timestamp)) {
      $timestamp = implode(' ', $timestamp);
      $timestamp = strtotime($timestamp);
    }
    elseif ($timestamp instanceof DrupalDateTime) {
      // Field was hidden on widget.
      $timestamp = strtotime($timestamp);
    }

    $old_timezone = date_default_timezone_get();
    $new_timezone = $date_time['timezone'] ?? date_default_timezone_get();
    $new_timezone = is_array($new_timezone) ? reset($new_timezone) : $new_timezone;
    if ($new_timezone === $old_timezone) {
      return $timestamp;
    }

    // Convert between timezones. @todo Use OOP.
    // $date = new DrupalDateTime($timestamp, new DateTimeZone($new_timezone));
    // $date = new DrupalDateTime($timestamp, $new_timezone);
    // $text = $date->format('Y-m-d H:i:s');
    // $date->setTimezone(new DateTimeZone($old_timezone));
    // $text = $date->format('Y-m-d H:i:s');
    // $timestamp = $date->getTimestamp();
    date_default_timezone_set($new_timezone);
    $timestamp = strtotime($timestamp);
    date_default_timezone_set($old_timezone);
    if (!$timestamp) {
      // Time should have been validated in form/widget.
      $timestamp = $transition->getDefaultRequestTime();
    }
    return $timestamp;
  }

  /**
   * Generate an element.
   *
   * This function is referenced in the Annotation for this class.
   *
   * @param array $element
   *   The element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The form.
   *
   * @return array
   *   The Workflow element.
   */
  public static function processTimestamp(array &$element, FormStateInterface $form_state, array &$complete_form) {

    /*
     * Input.
     */

    // A Transition object must have been set explicitly.
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $element['#default_value'];
    $user = $transition->getOwner();

    /*
     * Derived input.
     */
    $field_name = $transition->getFieldName();
    // Workflow might be empty on Action/VBO configuration.
    $wid = $transition->getWorkflowId();
    $workflow = $transition->getWorkflow();
    $workflow_settings = $workflow?->getSettings() ?? Workflow::defaultSettings();

    // Display scheduling form if user has permission.
    // Not shown on new entity (not supported by workflow module, because that
    // leaves the entity in the (creation) state until scheduling time.)
    // Not shown when editing existing transition.
    $add_schedule = $workflow_settings['schedule_enable'];
    if ($add_schedule
      && !$transition->isExecuted()
      && $user->hasPermission("schedule $wid workflow_transition")
    ) {
      $step = $form_state?->getValue('step')[0] ?? '';
      switch ($step) {
        case 'views_bulk_operations_config_form':
          // @todo test D8: On VBO Bulk 'modify entity values' form,
          // workflow_debug(__FILE__, __FUNCTION__, __LINE__);
          // Leave field settings.
          $add_schedule = TRUE;
          break;

        default:
          // ... and cannot be shown on a Content add page (no $entity_id),
          // ... but can be shown on a VBO 'set workflow state to..' page
          // when there is no entity.
          $entity = $transition->getTargetEntity();
          $add_schedule = !$entity?->isNew();
          break;
      }
    }

    /*
     * Output: generate the element.
     */

    // Display scheduling form under certain conditions.
    if ($add_schedule) {
      $add_timezone = $workflow_settings['schedule_timezone'];
      $timezone = $user->getTimeZone();
      if (empty($timezone)) {
        $timezone = \Drupal::config('system.date')->get('timezone.default');
      }

      $timezone_options = array_combine(timezone_identifiers_list(), timezone_identifiers_list());
      $is_scheduled = $transition->isScheduled();
      $timestamp = $transition->getTimestamp();
      // Round timestamp to previous minute.
      $timestamp = floor($timestamp / 60) * 60;
      // Convert for use in formElement.
      $timestamp = DrupalDateTime::createFromTimestamp($timestamp);

      // Define class for '#states' behavior.
      // Fetch the form ID. This is unique for each entity,
      // to allow multiple form per page (Views, etc.).
      // Make it uniquer by adding the field name, or else the scheduling of
      // multiple workflow_fields is not independent of each other.
      // If we are indeed on a Transition form (so, not a Node Form with widget)
      // then change the form id, too.
      $form_id = $form_state?->getFormObject()->getFormId()
        ?? WorkflowTransitionElement::getFormId();
      $form_uid = Html::getUniqueId($form_id);
      // @todo Align with WorkflowTransitionForm->getFormId().
      $class_identifier = Html::getClass("scheduled_{$form_uid}-{$field_name}");

      $element['scheduled'] = [
        '#type' => 'radios',
        '#title' => t('Schedule'),
        '#options' => [
          '0' => t('Immediately'),
          '1' => t('Schedule for state change'),
        ],
        '#default_value' => (string) $is_scheduled,
        '#attributes' => [
          // 'id' => "scheduled_{$form_id}",
          'class' => [$class_identifier],
        ],
      ];
      $element['scheduled_datetime'] = [
        '#type' => 'container',
        // '#open' Controls HTML5 'details' 'open' attribute. Defaults to FALSE.
        '#open' => TRUE,
        '#attributes' => ['class' => ['container-inline']],
        '#states' => [
          'visible' => ["input.$class_identifier" => ['value' => '1']],
        ],
      ];
      $element['scheduled_datetime']['datetime'] = [
        '#type' => 'datetime',
        '#prefix' => t('At') . ' ',
        '#default_value' => $timestamp,
      ];
      $element['scheduled_datetime']['timezone'] = [
        '#type' => $add_timezone ? 'select' : 'hidden',
        '#options' => $timezone_options,
        '#default_value' => [$timezone => $timezone],
      ];
    }

    return $element;
  }

}
