<?php

namespace Drupal\workflow\Element;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\workflow\Entity\Workflow;
use Drupal\workflow\Entity\WorkflowTransitionInterface;

/**
 * Provides a form element for the WorkflowTransitionForm and ~Widget.
 *
 * Properties:
 * - #return_value: The value to return when the checkbox is checked.
 *
 * @see \Drupal\Core\Render\Element\FormElement
 * @see https://www.drupal.org/node/169815 "Creating Custom Elements"
 *
 * @FormElement("workflow_transition")
 */
class WorkflowTransitionElement extends FormElementBase {
  // @todo Extends FormElementBase

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = static::class;
    return [
      '#input' => TRUE,
      '#return_value' => 1,
      '#process' => [
        [$class, 'processTransition'],
        [$class, 'processAjaxForm'],
        // [$class, 'processGroup'],
      ],
      '#element_validate' => [
        [$class, 'validateTransition'],
      ],
      // @todo D11 removed #pre_render callback array{class-string<static(Drupal\workflow\Element\WorkflowTransitionElement)>, 'preRenderTransition'} at key '0' is not callable.
      // '#pre_render' => [
      // [$class, 'preRenderTransition'],
      // ],
      // '#theme' => 'input__checkbox',
      // '#theme' => 'input__textfield',
      '#theme_wrappers' => ['form_element'],
      // '#title_display' => 'after',
    ];
  }

  /**
   * Generate an element.
   *
   * This function is referenced in the Annotation for this class.
   *
   * @param array $element
   *   The element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The form.
   *
   * @return array
   *   The Workflow element
   */
  public static function processTransition(array &$element, FormStateInterface $form_state, array &$complete_form) {
    workflow_debug(__FILE__, __FUNCTION__, __LINE__); // @todo D8: test this snippet.
    return self::transitionElement($element, $form_state, $complete_form);
  }

  /**
   * Generate an element.
   *
   * This function is an internal function, to be reused in:
   * - TransitionElement,
   * - TransitionDefaultWidget.
   *
   * @param array $element
   *   Reference to the form element.
   * @param \Drupal\Core\Form\FormStateInterface|null $form_state
   *   The form state.
   * @param array $complete_form
   *   The form.
   *
   * @return array
   *   The form element $element.
   *
   * @usage:
   *   @example $element['#default_value'] = $transition;
   *   @example $element += WorkflowTransitionElement::transitionElement($element, $form_state, $form);
   */
  public static function transitionElement(array &$element, FormStateInterface|NULL $form_state, array &$complete_form) {

    /*
     * Input.
     */
    // A Transition object must have been set explicitly.
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $element['#default_value'];

    /*
     * Derived input.
     */
    $field_name = $transition->getFieldName();
    // Workflow might be empty on Action/VBO configuration.
    $wid = $transition->getWorkflowId();
    $workflow = $transition->getWorkflow();
    $workflow_settings = $workflow?->getSettings() ?? Workflow::defaultSettings();
    $field_label = $transition->getFieldLabel();
    $force = $transition->isForced();

    // The help text is not available for container. Let's add it to the
    // 'to_sid' widget. N.B. it is empty on Workflow Tab, Node View page.
    // @see www.drupal.org/project/workflow/issues/3217214
    $help_text = $element['#description'] ?? '';
    unset($element['#description']);

    // Prepare/manipulate the presence of the 'to_sid' widget.
    $show_widget = WorkflowTransitionElement::showWidget($transition);
    $widget_has_one_option = 1 == count($element['to_sid']['widget']['#options']);
    // Avoid error with grouped options when workflow not set.
    $options_type = $wid ? $workflow_settings['options'] : 'select';
    // Suppress 'buttons' on 'edit executed transition'.
    $options_type = $show_widget && !$widget_has_one_option ? $options_type : 'radios';

    /*
     * Output: generate the element.
     */

    $element['#tree'] = TRUE;
    // Add class following node-form pattern (both on form and container).
    $element['#attributes']['class'][] = "workflow-transition-{$wid}-container";
    $element['#attributes']['class'][] = "workflow-transition-container";

    // Prepare a UI wrapper. It might be a (collapsible) fieldset.
    // Note: It will be overridden in WorkflowTransitionForm.
    $title = NULL;
    $element = WorkflowTransitionElement::addWrapper($element, $title);

    // Start overriding BaseFieldDefinitions.
    // @see WorkflowTransition::baseFieldDefinitions()
    $attribute_name = 'field_name';
    $attribute_key = 'widget';
    $widget = [
      // Only show field_name on VBO/Actions screen.
      '#access' => FALSE,
    ];
    self::updateWidget($element[$attribute_name], $attribute_key, $widget);

    $attribute_name = 'from_sid';
    $attribute_key = 'widget';
    $widget['#access'] = FALSE;
    // Decide if we show either a widget or a formatter.
    // Add a state formatter before the rest of the form,
    // when transition is scheduled or widget is hidden.
    // Also no widget if the only option is the current sid.
    if ($transition->isScheduled() || $transition->isExecuted() || !$show_widget) {
      $entity = $transition->getTargetEntity();
      $from_sid = $element[$attribute_name][$attribute_key]['#default_value'][0];
      $widget = workflow_state_formatter($entity, $field_name, $from_sid);
      $widget['#title'] = t('From');
      $widget['#label_display'] = 'before'; // 'above', 'hidden'.
      $widget['#access'] = TRUE;
    }
    $element[$attribute_name] = $widget;

    // Add the 'options' widget.
    // It may be replaced later if 'Action buttons' are chosen.
    $attribute_name = 'to_sid';
    $attribute_key = 'widget';
    // Resetting $to_sid needed for radios, that have/need 1 value, not array.
    // Default value may be empty when field is added for existing entities.
    $to_sid = $element[$attribute_name][$attribute_key]['#default_value']
      ?? $element[$attribute_name][0][$attribute_key]['#default_value'] ?? '';
    $to_sid = is_array($to_sid) ? reset($to_sid) : $to_sid;
    $widget = [
      '#type' => $options_type,
      '#title' => t('Change @name', ['@name' => $field_label]),
      '#description' => $help_text,
      // When not $show_widget, the 'from_sid' is displayed.
      '#access' => $show_widget,
      // @todo This is only needed for 'radios'. Why?
      '#default_value' => $to_sid,
    ];
    // Subfield is NEVER disabled in Workflow 'Manage form display' settings.
    // @see WorkflowTypeFormHooks class.
    self::updateWidget($element[$attribute_name], $attribute_key, $widget);

    // Note: we SET the button type here in a static variable.
    if (WorkflowTransitionButtons::useActionButtons($options_type)) {
      // In WorkflowTransitionForm, a default 'Submit' button is added there.
      // In Entity Form, workflow_form_alter() adds button per permitted state.
      // Performance: inform workflow_form_alter() to do its job.
      //
      // Make sure the '#type' is not set to the invalid 'buttons' value.
      // It will be replaced by action buttons, but sometimes, the select box
      // is still shown.
      // @see workflow_form_alter().
      $widget = [
        '#type' => 'select',
        '#access' => FALSE,
      ];
      // Subfield is NEVER disabled in Workflow 'Manage form display' settings.
      self::updateWidget($element[$attribute_name], $attribute_key, $widget);
    }

    // Display scheduling form under certain conditions.
    $attribute_name = 'timestamp';
    $attribute_key = 'value';

    // @todo Use properties from BaseField settings.
    $widget = [
      '#type' => 'workflow_transition_timestamp',
      '#access' => $show_widget && !$transition->isExecuted(),
      '#default_value' => $transition,
    ];
    // Subfield may be disabled in Workflow 'Manage form display' settings.
    if (isset($element[$attribute_name])) {
      self::updateWidget($element[$attribute_name]['widget'], $attribute_key, $widget);
    }

    // Show comment, when both Field and Instance allow this.
    // This overrides BaseFieldDefinition.
    // @todo Use all settings in Workflow 'Manage form display' settings.
    // @see https://www.drupal.org/node/2100015 'Comment settings are a field'.
    $attribute_name = 'comment';
    $attribute_key = 'value';
    // @todo Use properties from BaseField settings.
    $widget = [
      '#type' => 'textarea',
      '#access' => ($workflow_settings['comment_log_node'] != '0'),
      '#default_value' => $transition->getComment(),
      '#required' => $workflow_settings['comment_log_node'] == '2',
    ];
    // Subfield may be disabled in Workflow 'Manage form display' settings.
    if (isset($element[$attribute_name])) {
      self::updateWidget($element[$attribute_name]['widget'], $attribute_key, $widget);
    }

    // Let user/system enforce the transition.
    $attribute_name = 'force';
    $attribute_key = 'value';
    $widget = [
      // Only show 'force' parameter on VBO/Actions screen.
      '#access' => FALSE,
      '#default_value' => $force,
      '#weight' => 10,
    ];
    // Subfield may be disabled in Workflow 'Manage form display' settings.
    if (isset($element[$attribute_name])) {
      self::updateWidget($element[$attribute_name]['widget'], $attribute_key, $widget);
    }

    return $element;
  }

  /**
   * Internal function to generate a wrapper with title for an element.
   *
   * @param array $element
   *   The form element to be altered.
   * @param array|null $title
   *   A title, must be NULL sometimes.
   *
   * @return array
   *   The form element $element.
   *
   * @usage:
   *   @example $element = WorkflowTransitionElement::addWrapper($element, $title);
   */
  public static function addWrapper(array $element, ?string $title = NULL) {
    $transition = $element['#default_value'];
    $workflow = $transition->getWorkflow();
    $workflow_settings = $workflow?->getSettings() ?? Workflow::defaultSettings();

    // Prepare a UI wrapper. It might be a (collapsible) fieldset.
    // Note: It will be overridden in WorkflowTransitionForm.
    $element = [
      '#type' => ($workflow_settings['fieldset'] != 0) ? 'details' : 'container',
      // Title may be NULL, since it will overwrite the 'History' page.
      '#title' => $workflow_settings['name_as_title'] ? $title : NULL,
      '#collapsible' => ($workflow_settings['fieldset'] != 0),
      '#open' => ($workflow_settings['fieldset'] != 2),
    ] + $element;

    return $element;
  }

  /**
   * Adds the workflow attributes to the standard attribute of each widget.
   *
   * For some reason, the widgets are in another level when the entity form page
   * is presented, then when the entity form page is submitted.
   *
   * @param array $haystack
   *   The array in which the widget is hidden.
   * @param string $attribute_key
   *   The widget key.
   * @param array $data
   *   The additional workflow data for the widget.
   */
  protected static function updateWidget(array &$haystack, string $attribute_key, array $data) {
    if (isset($haystack[0][$attribute_key])) {
      $haystack[0][$attribute_key] = $data + $haystack[0][$attribute_key];
    }
    elseif (!empty($haystack[$attribute_key])) {
      $haystack[$attribute_key] = $data + $haystack[$attribute_key];
    }
    else {
      // Subfield is disabled in Workflow 'Manage form display' settings.
      // Do not add our data.
    }
  }

  /**
   * Determines if the Workflow Transition Form must be shown.
   *
   * If not, a formatter must be shown, since there are no valid options.
   * Only the comment field may be displayed.
   *
   * @param \Drupal\workflow\Entity\WorkflowTransitionInterface $transition
   *   The transition at hand.
   *
   * @return bool
   *   A boolean indicator to display a widget or formatter.
   *   TRUE = a form (a.k.a. widget) must be shown;
   *   FALSE = no form, a formatter must be shown instead.
   */
  public static function showWidget(WorkflowTransitionInterface $transition) {

    $entity = $transition->getTargetEntity();
    $field_name = $transition->getFieldName();
    $account = $transition->getOwner();

    if ($entity?->in_preview) {
      // Avoid having the form in preview, since it has action buttons.
      // In preview, you can only go back to original, user cannot save data.
      return FALSE;
    }

    if ($transition->isExecuted()) {
      // We are editing an existing/executed/not-scheduled transition.
      // We need the widget to edit the comment.
      // Only the comments may be changed!
      // The states may not be changed anymore.
      return TRUE;
    }

    if (!$transition->getFromSid()) {
      // On Actions, where no entity exists.
      return TRUE;
    }

    // $options = $transition->getSettableOptions(NULL, 'to_sid');
    $options = $transition->getFromState()->getOptions($entity, $field_name, $account);
    $count = count($options);
    // The easiest case first: more then one option: always show form.
    if ($count > 1) {
      return TRUE;
    }

    // #2226451: Even in Creation state,
    // we must have 2 visible states to show the widget.
    // // Only when in creation phase, one option is sufficient,
    // // since the '(creation)' option is not included in $options.
    // // When in creation state,
    // if ($this->isCreationState()) {
    // return TRUE;
    // }
    return FALSE;
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The form ID.
   *
   * @usage Do not change name lightly.
   *   It is also used in hook_form_FORM_ID_alter().
   */
  public static function getFormId() {
    return 'workflow_transition_form';
  }

  /**
   * Implements ContentEntityForm::copyFormValuesToEntity().
   *
   * This is called from:
   * - WorkflowTransitionForm::copyFormValuesToEntity(),
   * - WorkflowDefaultWidget.
   *
   * N.B. in contrary to ContentEntityForm::copyFormValuesToEntity(),
   * - param 1 is returned as result, for returning the updated transition.
   * - param 3 is not $form_state (from Form), but $item array (from Widget).
   *
   * @param \Drupal\Core\Entity\EntityInterface $transition
   *   The transition object.
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $values
   *   The field item.
   *
   * @return \Drupal\workflow\Entity\WorkflowTransitionInterface
   *   The Transition object.
   */
  public static function copyFormValuesToTransition(EntityInterface $transition, array $form, FormStateInterface $form_state, array $values) {
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */

    if ($debug = FALSE) {
      // For debugging/testing, toggle above value,
      // so you can compare the values from transition vs. widget.
      // The transition may already be OK by core's copyFormValuesToEntity().
      $to_sid = $transition->getFromSid();
      $timestamp = $transition->getTimestamp();
      $comment = $transition->getComment();
      $force = $transition->isForced();
    }

    // Get new SID, taking into account action buttons vs. options.
    $action_values = WorkflowTransitionButtons::getTriggeringButton($transition, $form_state, $values);
    $to_sid = $action_values['to_sid'];

    // For reading $timestamp, use lots of fallbacks. :-/ .
    $timestamp = NULL;
    $complete_form = $form_state->getCompleteForm();
    if (!$timestamp) {
      // In Workflow History page, in Block,
      // $timestamp is set by WorkflowTransitionTimestamp::valueCallback(),
      $timestamp = $complete_form['timestamp']['widget'][0]['value']['#value'] ?? NULL;
    }
    if (!$timestamp) {
      // Used in Node Edit form item, not in History page, not in Block.
      $field_name = $transition->getFieldName();
      $timestamp = $complete_form[$field_name]['widget'][0]['timestamp']['widget'][0]['value']['#value'] ?? NULL;
    }
    if (!$timestamp) {
      // Restore lost transition for fetching timestamp
      // in more complex cases with nested arrays.
      $values['#default_value'] = $transition;
    }
    if (!$timestamp) {
      $input = $values;
      $timestamp_input = $input['timestamp'][0]['value'] ?? ['scheduled' => FALSE];
      $timestamp = WorkflowTransitionTimestamp::valueCallback($values, $timestamp_input, $form_state);
    }
    if (!$timestamp) {
      // Fallback to the raw user post. A workaround for AJAX submissions.
      $input = $form_state->getUserInput();
      $timestamp_input = $input['timestamp'][0]['value'] ?? ['scheduled' => FALSE];
      $timestamp = WorkflowTransitionTimestamp::valueCallback($values, $timestamp_input, $form_state);
    }

    // Note: when editing existing Transition, user may still change comments.
    // Note: subfields might be disabled, and not exist in formState.
    // Note: subfields are already set by core.
    // This is only needed on Node edit widget, not on Node view/History page.
    $comment = $values['comment'][0]['value'] ?? '';

    // Attribute 'force' is not set if '#access' is false.
    $force = (bool) ($values['force']['value'] ?? FALSE);

    $transition->setValues($to_sid);
    // Timestamp also determines $transition::is_scheduled();
    $transition->setTimestamp($timestamp);
    $transition->setComment($comment);
    $transition->force($force);
    $transition->copyAttachedFields($form, $form_state, $values);

    // Remove values, since
    // CommentWithWorkflow's 'field_name' overwrites Comment's 'field_name'.
    // This is needed because of the '#tree' problem in
    // DefaultWidget::extractFormValues().
    $form_state->unsetValue(['field_name']);
    $form_state->unsetValue(['to_sid']);
    $form_state->unsetValue(['timestamp']);
    // For some reason, 'comment' must be preserved, or in above code,
    // for Workflow History page, comment is empty. Other fields are OK.
    // This code is not used in Node edit form.
    // $form_state->unsetValue(['comment']); .
    $form_state->unsetValue(['force']);
    $attached_field_definitions = $transition->getAttachedFieldDefinitions();
    foreach ($attached_field_definitions as $field_name => $field) {
      // Remove value,(E.g., 'field_name' ruins Comment's 'field_name').
      $form_state->unsetValue([$field_name]);
    }

    return $transition;
  }

}
