<?php

namespace Drupal\workflow\Entity;

use Drupal\comment\CommentInterface;
use Drupal\Core\Entity\EntityInterface;

/**
 * Provides a wrapper/ decorator for the $transition->getTargetEntity().
 *
 * @todo Change all static functions to non static.
 * But a decorator requires duplicating many functions...
 */
class WorkflowTargetEntity {

  /**
   * The entity a transition points to.
   */
  protected ?EntityInterface $targetEntity;

  /**
   * {@inheritdoc}
   */
  public function __construct(EntityInterface $entity) {
    $this->targetEntity = $entity;
  }

  /**
   * Returns the original unchanged entity.
   *
   * @return \Drupal\Core\Entity\RevisionableInterface|null
   *   The original entity.
   *
   * @see EntityInterface::getOriginal()
   * @see https://www.drupal.org/node/3295826
   */
  public static function getOriginal(EntityInterface $entity) {
    return method_exists($entity, 'getOriginal')
      ? $entity->getOriginal()
      : $entity->original ?? NULL;
  }

  /**
   * Determines the Workflow field_name of an entity.
   *
   * If an entity has multiple workflows, only returns the first one.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity at hand.
   * @param string $field_name
   *   The field name. If given, will be returned unchanged.
   *
   * @return string
   *   The field name of the first workflow field.
   */
  public static function getFieldName(EntityInterface $entity, $field_name = '') {
    $field_name = match (TRUE) {

      // Normal case, a field name is given.
      !empty($field_name) => $field_name,

      // $entity may be empty on Entity Add page.
      !$entity => '',

      // Get the first field_name (multiple may exist).
      !empty($fields = workflow_allowed_field_names($entity)) =>
      array_key_first($fields),

      // No Workflow field exists.
      default => '',
    };
    return $field_name;
  }

  /**
   * Gets an Options list of field names.
   *
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   An entity.
   * @param string $entity_type
   *   An entity_type.
   * @param string $entity_bundle
   *   An entity.
   * @param string $field_name
   *   A field name.
   *
   * @return array
   *   An list of field names.
   */
  public static function getPossibleFieldNames(?EntityInterface $entity = NULL, $entity_type = '', $entity_bundle = '', $field_name = '') {
    $result = [];
    $fields = _workflow_info_fields($entity, $entity_type, $entity_bundle, $field_name);
    // @todo get proper field_definition without $entity.
    foreach ($fields as $definition) {
      $field_name = $definition->getName();
      /** @var \Drupal\Core\Entity\FieldableEntityInterface $entity */
      $field_definition = $entity?->getFieldDefinition($field_name);
      $label = $field_definition?->getLabel() ?? $field_name;
      $result[$field_name] = $label;
    }
    return $result;
  }

  /**
   * Gets the creation state for a given $entity and $field_name.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param string $field_name
   *
   * @return \Drupal\workflow\Entity\WorkflowState
   *   The creation State for the Workflow of the field.
   *
   * @see WorkflowTargetEntity::getCurrentStateId()
   */
  public static function getCreationState(EntityInterface $entity, $field_name) {
    $state = NULL;

    $wid = WorkflowTargetEntity::getWorkflowId($entity, $field_name);
    if (!$wid) {
      \Drupal::messenger()->addError(t('Workflow %wid cannot be loaded. Contact your system administrator.', ['%wid' => $wid]));
    }
    else {
      $workflow = Workflow::load($wid);
      $state = $workflow->getCreationState();
    }

    return $state;
  }

  /**
   * Gets the creation state ID for a given $entity and $field_name.
   *
   * Is a helper function for:
   * - workflow_node_current_state()
   * - workflow_node_previous_state()
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param string $field_name
   *
   * @return string
   *   The ID of the creation State for the Workflow of the field.
   */
  public static function getCreationStateId(EntityInterface $entity, $field_name) {
    $state = WorkflowTargetEntity::getCreationState($entity, $field_name);
    return $state?->id() ?? '';
  }

  /**
   * Gets the current state of a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param string $field_name
   *   The name of the field of the entity to check.
   *
   * @return \Drupal\workflow\Entity\WorkflowState
   *   The current state.
   *
   * @see WorkflowTargetEntity::getCurrentStateId()
   */
  public static function getCurrentState(EntityInterface $entity, $field_name = '') {
    $sid = WorkflowTargetEntity::getCurrentStateId($entity, $field_name);
    /** @var \Drupal\workflow\Entity\WorkflowState $state */
    $state = WorkflowState::load($sid);
    return $state;
  }

  /**
   * Gets the current state ID of a given entity.
   *
   * There is no need to use a page cache.
   * The performance is OK, and the cache gives problems when using Rules.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param string $field_name
   *   The name of the field of the entity to check.
   *   If empty, the field_name is determined on the spot. This must be avoided,
   *   since it makes having multiple workflow per entity unpredictable.
   *   The found field_name will be returned in the param.
   *
   * @return string
   *   The ID of the current state.
   */
  public static function getCurrentStateId(EntityInterface $entity, $field_name = '') {
    $sid = '';

    if (!$entity) {
      return $sid;
    }

    $field_name = workflow_get_field_name($entity, $field_name);
    // $items may be empty on node with options widget, or upon initial setting.
    // If Transition is added via CommentWithWorkflow, use the Commented Entity.
    $items = ($entity instanceof CommentInterface)
      ? $entity->getCommentedEntity()->{$field_name} ?? NULL
      : $entity->{$field_name} ?? NULL;

    if (!$items) {
      // Return the initial value.
      return $sid;
    }

    // Normal situation: get the value.
    $sid = $items->first()?->getStateId() ?? '';
    if ($sid) {
      return $sid;
    }

    // Use previous state if Entity is new/in preview/without current state.
    // (E.g., content was created before adding workflow.)
    // When CommentWithWorkflow, node/entity is never new or in preview.
    if ($entity->isNew() || (!empty($entity->in_preview)) || empty($sid)) {
      $sid = self::getPreviousStateId($entity, $field_name);
    }

    // State ID should now be determined.
    // @todo Raise exception if no value found, yet, instead of returning ''.
    return $sid ?? '';
  }

  /**
   * Gets the previous state of a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   *
   * @return \Drupal\workflow\Entity\WorkflowState
   *   The previous state.
   */
  public static function getPreviousState(EntityInterface $entity, $field_name = '') {
    $sid = WorkflowTargetEntity::getPreviousStateId($entity, $field_name);
    /** @var \Drupal\workflow\Entity\WorkflowState $state */
    $state = WorkflowState::load($sid);
    return $state;
  }

  /**
   * Gets the previous state ID of a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   The ID of the previous state.
   */
  public static function getPreviousStateId(EntityInterface $entity, $field_name = '') {
    $sid = '';

    if (!$entity) {
      return $sid;
    }

    $field_name = workflow_get_field_name($entity, $field_name);
    if (!$field_name) {
      // Return the initial value.
      return $sid;
    }

    // Retrieve previous state from the original.
    $original_entity = WorkflowTargetEntity::getOriginal($entity);
    if (!empty($sid = $original_entity?->{$field_name}?->first()?->getStateId())) {
      return $sid;
    }

    // A node may not have a Workflow attached.
    if ($entity->isNew()) {
      return self::getCreationStateId($entity, $field_name);
    }

    // @todo Read history with an explicit langcode(?).
    $langcode = ''; // $entity->language()->getId();
    // @todo D8: #2373383 Add integration with older revisions via Revisioning module.
    $entity_type_id = $entity->getEntityTypeId();
    $last_transition = WorkflowTransition::loadByProperties($entity_type_id, $entity->id(), [], $field_name, $langcode, 'DESC');
    if ($last_transition) {
      $sid = $last_transition->getToSid(); // @see #2637092, #2612702
    }
    if (!$sid) {
      // No history found on an existing entity.
      $sid = self::getCreationStateId($entity, $field_name);
    }

    return $sid;
  }

  /**
   * Gets the Workflow for a given $entity and $field_name.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param string $field_name
   *
   * @return \Drupal\workflow\Entity\Workflow
   *   The Workflow entity of the field.
   *
   * @see WorkflowTargetEntity::getWorkflowId()
   */
  public static function getWorkflow(EntityInterface $entity, $field_name) {
    $workflow = NULL;

    $wid = WorkflowTargetEntity::getWorkflowId($entity, $field_name);
    if (!$wid) {
      \Drupal::messenger()->addError(t('Workflow %wid cannot be loaded. Contact your system administrator.', ['%wid' => $wid]));
    }
    else {
      $workflow = Workflow::load($wid);
    }

    return $workflow;
  }

  /**
   * Gets the Workflow ID for a given $entity and $field_name.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param string $field_name
   *
   * @return \Drupal\workflow\Entity\WorkflowState
   *   The creation State for the Workflow of the field.
   *
   * @see WorkflowTargetEntity::getCurrentStateId()
   */
  public static function getWorkflowId(EntityInterface $entity, $field_name) {
    $wid = NULL;

    /** @var \Drupal\Core\Config\Entity\ConfigEntityBase $entity */
    // Sometimes, no first item is available, so read wid from storage.
    $items = $entity->{$field_name} ?? NULL;
    if ($items) {
      $wid = $entity->{$field_name}?->first()?->getWorkflow()?->id();
    }
    if ($items && !$wid) {
      $definition = $entity->{$field_name}?->getFieldDefinition();
      $field_storage_definition = $definition->getFieldStorageDefinition();
      $wid = $field_storage_definition->getSetting('workflow_type');
    }
    return $wid;
  }

  /**
   * {@inheritdoc}
   *
   * Gets the initial/resulting Transition of a workflow form/widget.
   */
  public static function getDefaultTransition(EntityInterface $entity, $field_name): ?WorkflowTransitionInterface {

    if ($entity->isNew()) {
      // Do not read when editing existing CommentWithWorkflow.
      if ($entity instanceof CommentInterface) {
        // On CommentWithWorkflow, for new comments,
        // always read scheduled transition of entity, not comment,
        // since it is unknown how the scheduled transition was created.
        /** @var \Drupal\comment\CommentInterface $entity */
        $commented_entity = $entity->getCommentedEntity();
        $transition = WorkflowScheduledTransition::loadByProperties(
          $commented_entity->getEntityTypeId(),
          $commented_entity->id(),
          [],
          $field_name
        );
        // But yes, convert to a transition on comment.
        $transition?->setTargetEntity($commented_entity);
      }
    }
    else {
      // Do not read when editing existing CommentWithWorkflow.
      if ($entity instanceof CommentInterface) {
        $transition = WorkflowTransition::loadByProperties(
          $entity->getEntityTypeId(),
          $entity->id(),
          [],
          $field_name
        );
      }
      else {
        // Only 1 scheduled transition can be found, but multiple executed ones.
        $transition = WorkflowScheduledTransition::loadByProperties(
          $entity->getEntityTypeId(),
          $entity->id(),
          [],
          $field_name
        );
      }
    }

    // Note: Field is empty if node created before module installation.
    // $transition may be NULL when initially setting the value on node form.
    $items = $entity->{$field_name} ?? NULL;
    if ($items) {
      $transition ??= $entity->{$field_name}?->first()?->getTransition();
      $transition ??= WorkflowTransition::create([
        'entity' => $entity,
        'field_name' => $field_name,
      ]);
    }

    return $transition;
  }

  /**
   * Determine if the entity is Workflow* entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE, if the entity is defined by workflow module.
   *
   * @usage Use it when a function should not operate on Workflow objects.
   */
  public static function isWorkflowEntityType($entity_type_id) {
    return in_array($entity_type_id, [
      'workflow_type',
      'workflow_state',
      'workflow_config_transition',
      'workflow_transition',
      'workflow_scheduled_transition',
    ]);
  }

}
