<?php

namespace Drupal\workflow\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\workflow\Element\WorkflowTransitionElement;
use Drupal\workflow\Entity\WorkflowTargetEntity;
use Drupal\workflow\Entity\WorkflowTransitionInterface;

/**
 * Provides a Transition Form to be used in the Workflow Widget.
 */
class WorkflowTransitionForm extends ContentEntityForm {

  /*************************************************************************
   * Implementation of interface FormInterface.
   */

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    // We need a proprietary Form ID, to identify the unique forms
    // when multiple fields or entities are shown on 1 page.
    // Test this f.i. by checking the 'scheduled' box. It will not unfold.
    // $form_id = parent::getFormId();

    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $this->getEntity();
    $field_name = $transition->getFieldName();

    // Entity may be empty on VBO bulk form.
    // $entity = $transition->getTargetEntity();
    // Compose Form Id from string + Entity Id + Field name.
    // Field ID contains entity_type, bundle, field_name.
    // The Form Id is unique, to allow for multiple forms per page.
    // $workflow_type_id = $transition->getWorkflowId();
    // Field name contains implicit entity_type & bundle (since 1 field per entity)
    // $entity_type = $transition->getTargetEntityTypeId();
    // $entity_id = $transition->getTargetEntityId();;
    $suffix = 'form';
    // Emulate nodeForm convention.
    if ($transition->id()) {
      $suffix = 'edit_form';
    }
    $form_id = implode('_', [
      'workflow_transition',
      $transition->getTargetEntityTypeId(),
      $transition->getTargetEntityId() ?? 'new',
      $field_name,
      $suffix,
    ]);
    // $form_id = Html::getUniqueId($form_id);
    return $form_id;
  }

  /**
   * Gets the TransitionWidget in a form (for e.g., Workflow History Tab)
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   * @param array $form_state_additions
   *   Some additions.
   * @param \Drupal\workflow\Entity\WorkflowTransitionInterface|null $transition
   *   The current transition, if any.
   *
   * @return array
   *   The form.
   *
   * @usage Use WorkflowTransitionForm::createInstance() in WT forms, and
   *   WorkflowTransitionForm::retrieveFormElement() in entity field widgets.
   */
  public static function createInstance(EntityInterface $entity, $field_name, array $form_state_additions = [], ?WorkflowTransitionInterface $transition = NULL) {
    /** @var \Drupal\Core\Entity\EntityFormBuilder $entity_form_builder */
    $entity_form_builder = \Drupal::getContainer()->get('entity.form_builder');

    $transition ??= WorkflowTargetEntity::getDefaultTransition($entity, $field_name);

    $form = $entity_form_builder->getForm($transition, 'add', $form_state_additions);
    return $form;
  }

  /**
   * Gets the TransitionWidget in a form (for e.g., Workflow History Tab)
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   * @param array $form_state_additions
   *   Some additions.
   * @param \Drupal\workflow\Entity\WorkflowTransitionInterface $transition
   *   The current transition.
   *
   * @return array
   *   The form.
   */
  public static function retrieveFormElement(EntityInterface $entity, $field_name, array $form_state_additions, WorkflowTransitionInterface $transition) {
    /** @var \Drupal\Core\Entity\EntityFormBuilder $entity_form_builder */
    // $entity_form_builder = \Drupal::getContainer()->get('entity.form_builder');
    // $form = $entity_form_builder->getForm($transition, 'add', $form_state_additions);

    // Mimic $entity_form_builder->getForm(), but only get the element.
    // Mimic $this->formBuilder->buildForm(), but only get the element.
    $entity_type_manager = \Drupal::entityTypeManager();
    $form_object = $entity_type_manager->getFormObject($transition->getEntityTypeId(), 'add');
    $form_object->setEntity($transition);
    $form_state = (new FormState)->setFormState($form_state_additions);
    $form_state->setFormObject($form_object);

    // $form_id = 'workflow_action_transition_form';
    // $element = $form_builder->retrieveForm($form_id, $form_state);
    $form = [];
    $element = $form_object->buildForm($form, $form_state);
    $element = WorkflowTransitionForm::trimFormElement($element);
    // Overwrite values that are earlier set by Form.
    // Used in Node Edit form, not in History page, not in Block.
    $title = $transition->getFieldLabel();
    $element = WorkflowTransitionElement::addWrapper($element, $title);

    return $element;
  }

  /**
   * Removes elements that are needed for a form, but not for a form element.
   *
   * @param array $workflow_form
   *   The gross form.
   *
   * @return array
   *   The trimmed form.
   */
  public static function trimFormElement(array $workflow_form) {
    $transition = $workflow_form['#default_value'];

    // The following are not in Element::children.
    $attributes = [
      // The container settings.
      '#type',
      '#title',
      '#collapsible',
      '#open',
      // The WorkflowTransaction at hand.
      '#default_value',
    ];
    foreach ($attributes as $attribute_name) {
      $element[$attribute_name] ??= $workflow_form[$attribute_name];
      unset($workflow_form[$attribute_name]);
    }

    // Determine and move the (attached) fields to the form.
    foreach (Element::children($workflow_form) as $attribute_name) {
      if ($transition->hasField($attribute_name)) {
        $element[$attribute_name] ??= $workflow_form[$attribute_name];
        unset($workflow_form[$attribute_name]);
      }
    }

    return $element;
  }

  /* *************************************************************************
   *
   * Implementation of interface EntityFormInterface (extends FormInterface).
   *
   */

  /**
   * This function is called by buildForm().
   *
   * Caveat: !! It is not declared in the EntityFormInterface !!
   *
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    // Call parent to get (extra) fields.
    // This might cause baseFieldDefinitions to appear twice.
    $form = parent::form($form, $form_state);

    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $this->entity;

    $form['#default_value'] = $transition;
    // Overwrite the BaseFields with the custom values.
    $form = WorkflowTransitionElement::transitionElement($form, $form_state, $form);
    // The title from the element overwrites the page title (in Claro theme).
    // This is unwanted.
    $form['#title'] = NULL;

    return $form;
  }

  /**
   * Returns an array of supported actions for the current entity form.
   *
   * Caveat: !! It is not declared in the EntityFormInterface !!
   *
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    // Action buttons are added in common workflow_form_alter(),
    // addActionButtons(), since it will be done in many form_id's.
    // Keep aligned: workflow_form_alter(), WorkflowTransitionForm::actions().
    if (!empty($actions['submit']['#value'])) {
      $actions['submit']['#value'] = $this->t('Update workflow');
    }
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function buildEntity(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\Core\Entity\FieldableEntityInterface $entity */
    $transition = $this->entity;
    // Update the entity.
    $transition = $this->copyFormValuesToEntity($transition, $form, $form_state);
    // Mark the entity as NOT requiring validation. (Used in validateForm().)
    $transition->setValidationRequired(FALSE);

    return $transition;
  }

  /**
   * {@inheritdoc}
   */
  public function copyFormValuesToEntity(EntityInterface $transition, array $form, FormStateInterface $form_state) {
    // The following call to parent should set all data correct to $transition.
    // 20250815 test results:
    // field\case | node widget | WTH create | WTH edit | WTH revert | Action widget |
    // field_name | n/a (4)     | ok idem    | ok idem  |            |               |
    // from_sid   |             | ok idem    | ok idem  |            |               |
    // to_sid     |             | ok update  | ok idem  |            |               |
    // timestamp  |             | NOK (1)    | NOK (1)  |            |               |
    // comment    |             | ok update  | ok update |           |               |
    // scheduled  |             | nok (3)    | ok 'no'  |            |               |
    // forced     |             | ok 'no'    | ok 'no'  |            |               |
    // executed   |             | ok 'yes'   | ok 'yes' |            |               |
    // extra field|             | ok (2)     | ok (2)   |            |               |
    //
    // @todo 20250815 Known issues for WTF:copyFormValuesToEntity():
    // (1): Executed/Scheduled timestamp set to current time (on History page).
    // (2): Extra field on WT~edit form are editable in UI, so 'updated' is ok.
    // -    Extra field for scheduled WT is not supported yet.
    // (3): $is_scheduled wrong, due to (1).
    // (4): Field widget calls copyFormValuesToTransition() directly.
    parent::copyFormValuesToEntity($transition, $form, $form_state);

    if ($jvo = TRUE) {
      // Use own version of copyFormValuesToEntity() to fix missing fields.
      // @todo normalize Widget so that parent can do all stuff.
      // Note: Pay attention use case where WT changes to WST and v.v.
      // @todo This is not needed (anymore) on WFH, only for action buttons.
      $values = $form_state->getValues();
      $transition = WorkflowTransitionElement::copyFormValuesToTransition($transition, $form, $form_state, $values);
    }
    return $transition;
  }

  /**
   * {@inheritdoc}
   *
   * This is called from submitForm().
   */
  public function save(array $form, FormStateInterface $form_state) {
    // Execute transition and update the target entity.
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $this->getEntity();
    return $transition->executeAndUpdateEntity();
  }

}
