<?php

namespace Drupal\workflow\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\workflow\Element\WorkflowTransitionElement;
use Drupal\workflow\Entity\Workflow;
use Drupal\workflow\Entity\WorkflowTargetEntity;
use Drupal\workflow\Entity\WorkflowTransitionInterface;
use Drupal\workflow\Form\WorkflowTransitionForm;

/**
 * Plugin implementation of the 'workflow_default' widget.
 *
 * @FieldWidget(
 *   id = "workflow_default",
 *   label = @Translation("Workflow Transition form"),
 *   field_types = {"workflow"},
 * )
 */
class WorkflowDefaultWidget extends WidgetBase {

  /**
   * Generates a widget.
   *
   * @param \Drupal\workflow\Entity\WorkflowTransitionInterface $transition
   *   A WorkflowTransition.
   *
   * @return array
   *   A render element, with key = field name.
   */
  public static function createInstance(WorkflowTransitionInterface $transition) : array {
    $element = [];

    $entity_type_manager = \Drupal::service('entity_type.manager');
    $entity = $transition->getTargetEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $entity_bundle = $entity->bundle();
    $view_mode = 'default';
    $field_name = $transition->getFieldName();

    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $form_display */
    $entity_form_display = $entity_type_manager->getStorage('entity_form_display');
    $dummy_form['#parents'] = [];
    $form_state = new FormState();
    $form_display = $entity_form_display->load("$entity_type_id.$entity_bundle.$view_mode");
    // $form_state_clone = clone $form_state;
    // $form_state_clone->set('entity', $entity);
    // $form_state_clone->set('form_display', $form_display);
    // $widget_fields = [$field_name];
    // foreach ($form_display->getComponents() as $name => $component) {
    // if (in_array($name, $widget_fields)) {
    /** @var \Drupal\Core\Entity\FieldableEntityInterface $entity */
    if ($widget = $form_display->getRenderer($field_name)) {
      $items = $entity->get($field_name);
      $items->filterEmptyItems();
      $element[$field_name] = $widget->form($items, $dummy_form, $form_state);
    }
    // }
    // }
    return $element;
  }

  /**
   * {@inheritdoc}
   *
   * Be careful: Widget may be shown in very different places. Test carefully!!
   *  - On a entity add/edit page;
   *  - On a entity preview page;
   *  - On a entity view page;
   *  - On a entity 'workflow history' tab;
   *  - On a comment display, in the comment history;
   *  - On a comment form, below the comment history.
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {

    $wid = $this->getFieldSetting('workflow_type');
    if (!$workflow = Workflow::load($wid)) {
      // @todo Add error message.
      return $element;
    }

    if ($this->isDefaultValueWidget($form_state)) {
      // On the Field settings page, User may not set a default value
      // (this is done by the Workflow module).
      return [];
    }

    $entity = $items->getEntity();
    $field_name = $items->getName();
    // To prepare the widget, use the Form, in order to get extra fields.
    $form_state_additions = [
      'input' => $form_state->getUserInput(),
      'values' => $form_state->getValues(),
      'triggering_element' => $form_state->getTriggeringElement(),
    ];

    // @todo This transition should be known from parameters.
    $transition = WorkflowTargetEntity::getDefaultTransition($entity, $field_name);
    $workflow_form = WorkflowTransitionForm::retrieveFormElement($entity, $field_name, $form_state_additions, $transition);
    $element = WorkflowTransitionForm::trimFormElement($workflow_form);
    // Overwrite values that are earlier set by Form.
    // Used in Node Edit form, not in History page, not in Block.
    $title = $transition->getFieldLabel();
    $element = WorkflowTransitionElement::addWrapper($element, $title);

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();

    // Override WidgetBase::extractFormValues() since it extracts field values
    // without respecting #tree = TRUE.
    // So, the following function massageFormValues has nothing to do.
    // @todo Why are the subfields not in their own array sub-element?
    // parent::extractFormValues($items, $form, $form_state);
    $values = [
      0 => $form_state->getUserInput(),
      // 0 => $form_state->getValues(),
    ];

    // Let the widget massage the submitted values.
    // And also update the above created $transition.
    $values = $this->massageFormValues($values, $form, $form_state);
    /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
    $transition = $values[0];

    // Refresh the target entity, since multiple versions are lingering around.
    // This is at least necessary for 'entity_create' form.
    $entity = $items->getEntity();
    $transition->setTargetEntity($entity);
    // Update targetEntity's itemList with the transition.
    // Note: This is a duplicate of below $items->setValue($values);
    $transition->setEntityWorkflowField();

    // BEGIN copy parent
    // Assign the values and remove the empty ones.
    $items->setValue($values);
    $items->filterEmptyItems();

    // Put delta mapping in $form_state, so that flagErrors() can use it.
    $field_state = static::getWidgetState($form['#parents'], $field_name, $form_state);
    foreach ($items as $delta => $item) {
      $field_state['original_deltas'][$delta] = $item->_original_delta ?? $delta;
      unset($item->_original_delta, $item->_weight, $item->_actions);
    }
    static::setWidgetState($form['#parents'], $field_name, $form_state, $field_state);
    // END copy parent.
  }

  /**
   * {@inheritdoc}
   *
   * Implements workflow_transition() -> WorkflowDefaultWidget::submit().
   *
   * This is called from function _workflow_form_submit($form, &$form_state)
   * It is a replacement of function workflow_transition($entity, $to_sid, $force, $field)
   * It performs the following actions;
   * - save a scheduled action
   * - update history
   * - restore the normal $items for the field.
   *
   * @todo Remove update of {node_form} table. (separate task, because it has features, too.)
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $field_name = $this->fieldDefinition->getName();

    // Set the new value.
    // Beware: We presume WorkflowItem::cardinality = 1 !!
    // The widget form element type has transformed the value to a
    // WorkflowTransition object at this point. We need to convert it
    // back to the regular 'value' string format.
    foreach ($values as $index => &$item) {
      if (!empty($item)) {
        /** @var \Drupal\workflow\Entity\WorkflowTransitionInterface $transition */
        $transition = $form[$field_name]['widget'][$index]['#default_value'];

        // Use a proprietary version of copyFormValuesToEntity().
        // We want to adapt the current WT, keeping id(), not create a new WT.
        $transition = WorkflowTransitionElement::copyFormValuesToTransition($transition, $form, $form_state, $item);

        // Try to execute the transition. Return $from_sid when error.
        $to_sid = $from_sid = '';
        if (!$transition) {
          // This should not be possible (perhaps when testing/developing).
          $this->messenger()->addError($this->t('Error: the transition from %from_sid to %to_sid could not be generated.'));
        }
        else {
          // The transition may be scheduled or not. Save the result, and
          // rely upon hook workflow_entity_insert/update($entity) in
          // file workflow.module to save/execute the transition.
          // - validate option;
          // - add hook to let other modules change comment;
          // - add to history;
          // - add message to logger;
          // - return the new State ID. (Execution may fail and return old SID.)
          // @todo #2287057: verify if submit() really is only used for UI.
          // If not, $user must be passed.
          $user = $transition->getOwner();
          $from_sid = $transition->getFromSid();
          $force = $transition->isForced();
          // If Entity is inserted, the Id is not yet known.
          // So we can't yet save the transition right now, but must rely on
          // function/hook workflow_entity_insert($entity) in file workflow.module.
          // $to_sid = $transition->force($force)->execute();
          // If Entity is updated, to stay in sync with insert, we rely on
          // function/hook workflow_entity_update($entity) in file workflow.module.
          // $to_sid = $transition->force($force)->execute();
          $to_sid = $transition->isAllowed($user, $force)
            ? $transition->getToSid()
            : $from_sid;
        }
      }
    }
    return [$transition];
  }

}
