<?php

namespace Drupal\workflow;

use Drupal\workflow\Entity\Workflow;
use Drupal\workflow\Entity\WorkflowInterface;
use Drupal\workflow\Plugin\Field\FieldType\WorkflowItem;

/**
 * Wrapper methods for Workflow* objects.
 *
 * This adds getWorkflow(), getWorkflowId(), setWorkflow(), setWorkflowId()
 * methods to the class.
 *
 * @ingroup workflow
 */
trait WorkflowTypeAttributeTrait {

  /**
   * The machine_name of the attached Workflow.
   *
   * @var string
   */
  protected $wid = '';

  /**
   * The attached Workflow.
   *
   * It must explicitly be defined, and not be public, to avoid errors
   * when exporting with json_encode().
   *
   * @var \Drupal\workflow\Entity\Workflow
   */
  protected $workflow = NULL;

  /**
   * Sets the Workflow.
   *
   * @param \Drupal\workflow\Entity\WorkflowInterface|null $workflow
   *   The Workflow object.
   */
  public function setWorkflow(?WorkflowInterface $workflow = NULL) {
    $this->wid = '';
    $this->workflow = NULL;
    if ($workflow) {
      $this->wid = $workflow->id();
      $this->workflow = $workflow;
    }
  }

  /**
   * Returns the Workflow object of this object.
   *
   * @return \Drupal\workflow\Entity\Workflow
   *   Workflow object.
   */
  public function getWorkflow() {
    if ($this->workflow) {
      return $this->workflow;
    }

    /* @noinspection PhpAssignmentInConditionInspection */
    if ($wid = $this->getWorkflowId()) {
      $this->workflow = Workflow::load($wid);
    }
    return $this->workflow;
  }

  /**
   * Sets the Workflow ID of this object.
   *
   * @param string $wid
   *   The Workflow ID.
   *
   * @return object
   *   The Workflow object.
   */
  public function setWorkflowId($wid) {
    $this->wid = $wid;
    $this->workflow = NULL;
    return $this;
  }

  /**
   * Returns the Workflow ID of this object.
   *
   * @return string
   *   Workflow ID.
   */
  public function getWorkflowId() {
    /** @var \Drupal\Core\Entity\ContentEntityBase $this */
    if (!empty($this->wid)) {
      // WorkflowState.
      return $this->wid;
    }

    if ($this instanceof WorkflowItem) {
      try {
        // workflow_type may be already set in WorkflowItem::__construct().
        // Or is not set yet upon field creation.
        $this->wid = $this->getSetting('workflow_type');
        return $this->wid;
      }
      catch (\Error $e) {
        // Error: Call to undefined method
        // Drupal\workflow\Entity\WorkflowTransition::getSetting().
      }
    }

    try {
      // WorkflowTransition.
      $value = $this->get('wid');
      $this->wid = match (TRUE) {
        // 'entity_reference' in WorkflowTransition.
        is_object($value) => $value->{'target_id'} ?? '',
        // 'list_string' in WorkflowTransition.
        is_string($value) => $value,
      };
    }
    catch (\UnhandledMatchError $e) {
      workflow_debug(__FILE__, __FUNCTION__, __LINE__, '', '');
    }
    return $this->wid;
  }

}
