<?php

namespace Drupal\workflow\Form;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\workflow\Entity\WorkflowInterface;
use Drupal\workflow\Entity\WorkflowRole;

/**
 * Defines a class to build a listing of Workflow Config Transitions entities.
 *
 * @see \Drupal\workflow\Entity\WorkflowConfigTransition
 */
class WorkflowConfigTransitionRoleForm extends WorkflowConfigTransitionFormBase {

  /**
   * {@inheritdoc}
   */
  protected $entitiesKey = 'workflow_state';

  /**
   * {@inheritdoc}
   */
  protected $type = 'permission';

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = [];

    $workflow = $this->workflow;
    $states = $workflow->getStates(WorkflowInterface::ACTIVE_CREATION_STATES);
    if ($states) {
      $header['label_new'] = $this->t('From \ To');

      /** @var \Drupal\workflow\Entity\WorkflowState $state */
      foreach ($states as $state) {
        // Don't allow transition TO (creation).
        if (!$state->isCreationState()) {
          $header[$state->id()] = $this->t('@label', ['@label' => $state->label()]);
        }
      }
    }
    return $header;
  }

  /**
   * {@inheritdoc}
   *
   * Builds a row for the following table:
   *   Transitions, for example:
   *     18 => [
   *       20 => [
   *         'author' => 1,
   *         1        => 0,
   *         2        => 1,
   *       ]
   *     ]
   *   means the transition From state 18 to state 20 can be executed by
   *   the content author or a user in role 2. The $transitions array should
   *   contain ALL transitions for the workflow.
   */
  public function buildRow(EntityInterface $entity) {
    $row = [];

    $workflow = $this->workflow;
    if ($workflow) {
      // Each $entity is a from-state.
      /** @var \Drupal\workflow\Entity\WorkflowState $from_state */
      $from_state = $entity;
      $from_sid = $from_state->id();

      $states = $workflow->getStates(WorkflowInterface::ACTIVE_CREATION_STATES);
      if ($states) {
        // Only get the roles with proper permission + 'author' role.
        $type_id = $workflow->id();
        $roles = workflow_allowed_user_role_names("create $type_id workflow_transition");
        // Prepare default value for 'stay_on_this_state'.
        // array_combine(array_keys($roles), array_keys($roles));
        $allow_all_roles = [];

        foreach ($states as $state) {
          $row['to'] = [
            '#type' => 'markup',
            '#markup' => $this->t('@label', ['@label' => $from_state->label()]),
          ];

          foreach ($states as $to_state) {
            // Don't allow transition TO (creation).
            if ($to_state->isCreationState()) {
              continue;
            }
            // Only allow transitions from $from_state.
            if ($state->id() !== $from_state->id()) {
              continue;
            }
            $to_sid = $to_state->id();

            // Load existing config_transitions. Create if not found.
            $config_transitions = $workflow->getTransitionsByStateId($from_sid, $to_sid);
            if (!$config_transition = reset($config_transitions)) {
              $config_transition = $workflow->createTransition($from_sid, $to_sid);
            }
            $stay_on_this_state = !$config_transition->hasStateChange();

            $row[$to_sid]['workflow_config_transition'] = [
              '#type' => 'value',
              '#value' => $config_transition,
            ];
            $row[$to_sid]['roles'] = [
              '#type' => 'checkboxes',
              '#options' => $stay_on_this_state ? [] : $roles,
              '#disabled' => $stay_on_this_state,
              '#default_value' => $stay_on_this_state ? $allow_all_roles : $config_transition->roles,
            ];
          }
        }
      }
    }
    return $row;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // 'Creation' state may not be the only state.
    if (count($form_state->getValue($this->entitiesKey)) < 2) {
      $form_state->setErrorByName('id', $this->t('Please create at least one other state.'));
      return;
    }

    // Make sure 'author' is checked for (creation) -> [something].
    $creation_state = $this->getWorkflow()->getCreationState();
    $creation_state_id = $creation_state->id();
    $author_has_permission = FALSE;
    $values = $form_state->getValue($this->entitiesKey)[$creation_state_id];
    foreach ($values as $to_sid => $transition_data) {
      if ($to_sid == $creation_state_id) {
        // This should never be the case, but test anyway.
        continue;
      }

      // Get list of roles that are allowed for this transition.
      $roles = $transition_data['roles'];
      $roles = array_filter($roles);
      // 'author' role is not required anymore, but 'any' role is required.
      // @see https://www.drupal.org/project/workflow/issues/359834
      if (isset($roles[WorkflowRole::AUTHOR_RID])) {
        // continue; .
      }
      if (!empty($roles)) {
        $author_has_permission = TRUE;
      }
    }
    if (!$author_has_permission) {
      $form_state->setErrorByName('id', $this->t('At least one role must have permission to go from %creation to another state.',
        ['%creation' => $creation_state->label()]));
    }

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    foreach ($form_state->getValue($this->entitiesKey) as $from_sid => $to_data) {
      foreach ($to_data as $transition_data) {
        if (isset($transition_data['workflow_config_transition'])) {
          /** @var \Drupal\workflow\Entity\WorkflowConfigTransition $config_transition */
          $config_transition = $transition_data['workflow_config_transition'];
          $config_transition->roles = $transition_data['roles'];
          $config_transition->save();
        }
        else {
          // Should not be possible.
        }
      }
    }

    $this->messenger()->addStatus($this->t('The workflow was updated.'));
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    /** @var \Drupal\workflow\Entity\WorkflowConfigTransition[] $workflow_transitions */
    $workflow_transitions = $this->workflow->getTransitions();

    $config_names = [];
    foreach ($workflow_transitions as $transition) {
      $config_names[] = $transition->getConfigDependencyName();
    }
    return $config_names;
  }

}
