<?php

namespace Drupal\workflow_notifications\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\user\Entity\User;
use Drupal\workflow\Entity\WorkflowRole;
use Drupal\workflow\Entity\WorkflowTransition;
use Drupal\workflow\Entity\WorkflowTransitionInterface;
use Drupal\workflow\WorkflowTypeAttributeTrait;

/**
 * Defines a Workflow Notification entity'. Must be overwritten.
 */
abstract class WorkflowAbstractNotification extends ConfigEntityBase implements WorkflowNotificationInterface {

  use WorkflowTypeAttributeTrait;

  /**
   * The machine_name of this Notification.
   *
   * @var string
   */
  public $id;

  /**
   * The label of this Notification.
   *
   * @var string
   */
  public $label;

  /**
   * The Notification will be sent upon this 'from state ID'.
   *
   * @var string
   */
  public $from_sid = '';

  /**
   * The Notification will be sent upon this 'to state ID'.
   *
   * @var string
   */
  public $to_sid = '';

  /**
   * Indicator when to send a message.
   *
   * @var string
   */
  public $when_to_trigger = 'on_state_change';

  /**
   * Number of days a message sent before a scheduled transition will happen.
   *
   * @var int
   */
  public $days = 0;

  /**
   * Indicator to restrict to only participating receivers.
   *
   * @var bool
   */
  public $participate = FALSE;

  /**
   * The list of user roles to receive a message.
   *
   * @var array
   */
  public $roles = [];

  /**
   * The message subject to be sent when state change happens.
   *
   * @var string
   */
  public $subject;

  /**
   * The message to be sent when state change happens.
   *
   * @var array
   */
  public $message = ['value' => '', 'format' => 'basic_html'];

  /**
   * Get the list of ID's as set by administrator on Notification form.
   *
   * @return string
   *   An list of ID's (mail addresses, phone numbers), separated by '\r\n'.
   */
  abstract protected function getReceiverIds(): string;

  /**
   * {@inheritdoc}
   *
   * Note: The entity type must explicitly set by the inheriting subclass.
   */
  public static function loadMultipleByProperties($from_sid, $to_sid, $wid, $trigger, $days, $entity_type = 'notification_type') {
    $result = \Drupal::entityQuery($entity_type);
    if (!empty($from_sid) && !empty($to_sid)) {
      $from_sid = $result->orConditionGroup()
        ->condition('from_sid', $from_sid, '=')
        ->condition('from_sid', 'all', '=');
      $to_sid = $result->orConditionGroup()
        ->condition('to_sid', $to_sid, '=')
        ->condition('to_sid', 'all', '=');
      $result->condition($from_sid)
        ->condition($to_sid);
    }
    if (!empty($wid)) {
      $result->condition('wid', $wid, '=');
    }
    if (!empty($trigger)) {
      $trigger = is_array($trigger) ? $trigger : [$trigger];
      $result->condition('when_to_trigger', $trigger, 'IN');
    }
    if (!empty($days)) {
      $result->condition('days', $days, '=');
    }

    $ids = $result
      ->accessCheck(FALSE)
      ->execute();

    $notifications = self::loadMultiple($ids);
    return $notifications;
  }

  /**
   * {@inheritdoc}
   */
  public function save() {
    if ($this->when_to_trigger == 'on_state_change') {
      $this->days = 0;
    }
    // Avoid configuration_inspector errors because of '0'-values.
    $this->roles = array_filter($this->roles);
    return parent::save();
  }

  /**
   * {@inheritdoc}
   */
  public function sendMessages(string $trigger, WorkflowTransitionInterface $transition) {
    $from_sid = $transition->getFromSid();
    $to_sid = $transition->getToSid();
    // Support 'any' from/to State.
    if ($this->from_sid !== 'all' && $this->from_sid !== $from_sid) {
      return;
    }
    if ($this->to_sid !== 'all' && $this->to_sid !== $to_sid) {
      return;
    }

    $entity = $transition->getTargetEntity();
    $tkn_rpl_val = $this->replaceTokens($this->getReceiverIds(), $this->message['value'], $this->subject ?? '', $entity, $transition);

    // Collect the list of receiver id's (mail address, sms phone number, etc.)
    $ids = WorkflowNotification::convertToArray($tkn_rpl_val['ids']);
    // Get list of roles that are allowed for this transition.
    $roles = $this->roles;
    // Collect ID from owner.
    $roles = array_filter($roles);
    if (isset($roles[WorkflowRole::AUTHOR_RID])) {
      /** @var \Drupal\user\EntityOwnerInterface $entity */
      $ids[] = $this->getReceiverIdFromUser($entity->getOwner());
      unset($roles[WorkflowRole::AUTHOR_RID]);
    }
    $ids = array_merge($ids, $this->collectReceiverIdsFromRoles($roles));
    $ids = $this->filterReceiverIdsByEntity($roles, $ids, $entity);
    $params['subject'] = $tkn_rpl_val['subject'];
    $params['message'] = $tkn_rpl_val['message'];
    // Defer functionality to each subclass.
    $this->send($ids, $params, $trigger);
  }

  /**
   * Send message to all determined users.
   *
   * @param array $to
   *   A list of receiver ID's (mail, sms).
   * @param array $params
   *   The message details.
   * @param string $trigger
   *   Determines the triggering moment.
   */
  abstract protected function send(array $to, array $params, string $trigger);

  /**
   * Returns the default trigger ID for this Notification type.
   *
   * @return string
   *   The default trigger ID for this Notification type.
   */
  abstract public function getDefaultTriggerId();

  /**
   * Convert a string with id's into array.
   *
   * @param string $ids
   *   An list of ID's (mail addresses, phone numbers), separated by '\r\n'.
   *
   * @return array
   *   The converted list.
   */
  public static function convertToArray(string $ids) {
    $ids = "\r\n$ids";
    return array_filter(preg_split('/\r\n|[\r\n]/', $ids));
  }

  /**
   * Implements token replace.
   *
   * @param string $ids
   *   The list of receiver IDs.
   * @param string $message
   *   The message body.
   * @param string $subject
   *   The message header.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The (to be) changed entity that triggers the Notification.
   * @param \Drupal\workflow\Entity\WorkflowTransitionInterface $transition
   *   The Transition that triggers the Notification.
   *
   * @return array
   *   The replaced tokes.
   *
   * @todo This should be moved into Workflow service.
   */
  protected function replaceTokens($ids, string $message, string $subject, EntityInterface $entity, WorkflowTransitionInterface $transition): array {
    $token_service = \Drupal::token();

    $tkn_rpl_val = [];
    $tkn_rpl_val['ids'] = $token_service->replace($ids, [
      $entity->getEntityTypeId() => $entity,
      'workflow_transition' => $transition,
      'workflow_scheduled_transition' => $transition,
    ], ['clear' => TRUE]);

    $tkn_rpl_val['message'] = $token_service->replace($message, [
      $entity->getEntityTypeId() => $entity,
      'workflow_transition' => $transition,
      'workflow_scheduled_transition' => $transition,
    ], ['clear' => TRUE]);

    $tkn_rpl_val['subject'] = $token_service->replace($subject, [
      $entity->getEntityTypeId() => $entity,
      'workflow_transition' => $transition,
      'workflow_scheduled_transition' => $transition,
    ], ['clear' => TRUE]);

    return $tkn_rpl_val;
  }

  /**
   * Get role-defined receivers to the list of user-specified receivers.
   *
   * @param array $roles
   *   A list of user role id's.
   *
   * @return array
   *   A list of receivers (mail addresses, phone numbers).
   *
   * @todo Move empty function to abstract parent class.
   * Replaces _workflow_sms_notify_collect_phone_num().
   * Replaces _workflow_notifications_collect_mail_ids().
   */
  protected function collectReceiverIdsFromRoles(array $roles) {
    $receiver_ids = [];

    // Remove unchecked roles.
    $roles = array_filter($roles);
    if (!$roles) {
      return $receiver_ids;
    }

    // Note: Role 'authenticated' does not include other roles.
    $user_ids = \Drupal::entityQuery('user')
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->condition('roles', $roles, 'IN')
      ->execute();

    if ($user_ids) {
      /** @var \Drupal\user\UserInterface[] $users */
      $users = User::loadMultiple($user_ids);
      foreach ($users as $key => $user) {
        $receiver_ids[] = $this->getReceiverIdFromUser($user);
      }
    }

    return $receiver_ids;
  }

  /**
   * Returns only the participating users from the given receivers.
   *
   * @param array $roles
   *   A list of user role id's.
   * @param array $receiver_ids
   *   A list of receivers (mail addresses, phone numbers).
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The (to be) changed entity that triggers the Notification.
   *
   * @return array
   *   A filtered list of receivers (mail addresses, phone numbers).
   *
   * @todo Move empty function to abstract parent class.
   */
  protected function filterReceiverIdsByEntity(array $roles, array $receiver_ids, $entity) {

    if (!$this->participate) {
      return $receiver_ids;
    }

    if (!$receiver_ids) {
      return $receiver_ids;
    }

    if (!$entity) {
      return $receiver_ids;
    }

    $participating_ids = [];
    $nid = $entity->id();
    $entity_type_id = $entity->getEntityTypeId();
    $transitions = WorkflowTransition::loadMultipleByProperties($entity_type_id, [$nid]);
    foreach ($transitions as $transition) {
      $user = $transition->getOwner();
      $uid = $transition->getOwnerId();
      foreach ($roles as $rid => $role_name) {
        if ($user->hasRole($rid)) {
          $participating_ids[$uid] = $this->getReceiverIdFromUser($user);
        }
      }
    }

    if ($participating_ids) {
      $filtered_ids = array_intersect($receiver_ids, $participating_ids);
      $receiver_ids = $filtered_ids;
    }

      return $receiver_ids;
  }

  /**
   * Returns the receiver ID from the user.
   *
   * @param \Drupal\user\UserInterface $user
   *   A user object.
   *
   * @return string
   *   A receiver (mail address, phone number).
   *
   * @todo Move empty function to abstract parent class.
   */
  abstract protected function getReceiverIdFromUser($user);

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = 'canonical', array $options = []) {
    /** @var \Drupal\Core\Url $url */
    $url = parent::toUrl($rel, $options);
    // The parameters are used in routing.yml file.
    $url->setRouteParameter('workflow_type', $this->getWorkflowId());
    $url->setRouteParameter('workflow_notify', $this->id());
    return $url;
  }

}
