<?php

namespace Drupal\workflow_sms_notify\Entity;

use Drupal\workflow_notifications\Entity\WorkflowAbstractNotification;
use Drupal\workflow_notifications\Entity\WorkflowNotificationInterface;
use Drupal\sms\Entity\SmsMessage;
use Drupal\sms\Direction;

/**
 * Defines a Workflow Notification entity'.
 *
 * @ConfigEntityType(
 *   id = "workflow_sms_notify",
 *   label = @Translation("Workflow SMS Notification"),
 *   handlers = {
 *     "access" = "Drupal\workflow_notifications\WorkflowNotificationControlHandler",
 *     "list_builder" = "Drupal\workflow_sms_notify\Controller\WorkflowSmsNotificationListBuilder",
 *     "form" = {
 *       "add" = "Drupal\workflow_sms_notify\Form\WorkflowSmsNotificationForm",
 *       "edit" = "Drupal\workflow_sms_notify\Form\WorkflowSmsNotificationForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "workflow_sms_notify",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "wid" = "wid",
 *     "from_sid" = "from_sid",
 *     "to_sid" = "to_sid",
 *     "when_to_trigger" = "when_to_trigger",
 *     "phone_num" = "phone_num",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "wid",
 *     "from_sid",
 *     "to_sid",
 *     "when_to_trigger",
 *     "days",
 *     "roles",
 *     "participate",
 *     "subject",
 *     "message",
 *     "phone_num",
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/workflow/workflow/{workflow_type}/sms-notifications/{workflow_sms_notify}/edit",
 *     "delete-form" = "/admin/config/workflow/workflow/{workflow_type}/sms-notifications/{workflow_sms_notify}/delete",
 *     "collection" = "/admin/config/workflow/workflow/{workflow_type}/sms-notifications",
 *   },
 * )
 */
class WorkflowSmsNotify extends WorkflowAbstractNotification implements WorkflowNotificationInterface {

  /**
   * The list of user mail addresses/phone numbers to receive the message.
   *
   * @var string
   */
  public $phone_num = '';

  /**
   * {@inheritdoc}
   */
  protected function getReceiverIds(): string {
    return $this->phone_num;
  }

  /**
   * {@inheritdoc}
   *
   * Note: This function sets the entity type explicitly as a parameter.
   */
  public static function loadMultipleByProperties($from_sid, $to_sid, $wid, $trigger, $days, $entity_type = 'workflow_sms_notify') {
    return parent::loadMultipleByProperties($from_sid, $to_sid, $wid, $trigger, $days, $entity_type);
  }

  /**
   * {@inheritdoc}
   */
  protected function send(array $to, array $params, string $trigger) {
    if (empty($to)) {
      return;
    }

    // Remove duplicate and empty entries.
    asort($to);
    $to = array_filter($to);
    $to = array_unique($to);

    $messenger = \Drupal::messenger();
    try {
      $smsProvider = \Drupal::service('sms.provider');
      $sms_message = SmsMessage::create()
        ->setMessage($params['message']);
      // $gateway = SmsGateway::load("sms_gateway");
      $sms_message = SmsMessage::convertFromSmsMessage($sms_message)
        ->addRecipients($to)
        // ->setGateway($gateway)
        ->setDirection(Direction::OUTGOING);
      $phone_number_service = \Drupal::service('sms.provider.default')->queue($sms_message);
      $messenger->addMessage(t('SMS sent successfully'), 'status', FALSE);
    }
    catch (\Exception $e) {
      $messenger->addMessage(t('SMS not sent, please contact site administrator'), 'error', FALSE);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultTriggerId() {
    return 'workflow_notification_before_sms_trigger';
  }

  /**
   * {@inheritdoc}
   */
  protected function getReceiverIdFromUser($user) {
    static $phone_number_service = \Drupal::service('sms.phone_number');
    // Note: Returns only verified numbers, since 2nd param $verified = TRUE.
    $phone_number = $phone_number_service->getPhoneNumbers($user);
    if (empty($phone_number)) {
      return '';
    }

    return $phone_number;
  }

}
