<?php

namespace Drupal\workflow_notifications\Entity;

/**
 * Defines a Workflow Notification entity'.
 *
 * @ConfigEntityType(
 *   id = "workflow_notify",
 *   label = @Translation("Workflow Mail Notification"),
 *   handlers = {
 *     "access" = "Drupal\workflow_notifications\WorkflowNotificationControlHandler",
 *     "list_builder" = "Drupal\workflow_notifications\Controller\WorkflowNotificationListBuilder",
 *     "form" = {
 *       "add" = "Drupal\workflow_notifications\Form\WorkflowNotificationForm",
 *       "edit" = "Drupal\workflow_notifications\Form\WorkflowNotificationForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "workflow_notify",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "wid" = "wid",
 *     "from_sid" = "from_sid",
 *     "to_sid" = "to_sid",
 *     "when_to_trigger" = "when_to_trigger",
 *     "mail_to" = "mail_to",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "wid",
 *     "from_sid",
 *     "to_sid",
 *     "when_to_trigger",
 *     "days",
 *     "roles",
 *     "participate",
 *     "subject",
 *     "message",
 *     "mail_ids",
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/workflow/workflow/{workflow_type}/notifications/{workflow_notify}/edit",
 *     "delete-form" = "/admin/config/workflow/workflow/{workflow_type}/notifications/{workflow_notify}/delete",
 *     "collection" = "/admin/config/workflow/workflow/{workflow_type}/notifications",
 *   },
 * )
 */
class WorkflowNotification extends WorkflowAbstractNotification implements WorkflowNotificationInterface {

  /**
   * The module name.
   *
   * @todo Get module_name dynamically from module or namespace.
   *
   * @const string
   */
  protected const MODULE_NAME = "workflow_notifications";

  /**
   * The list of user mail addresses/phone numbers to receive the message.
   *
   * @var string
   */
  public $mail_ids = '';

  /**
   * {@inheritdoc}
   */
  protected function getReceiverIds(): string {
    return $this->mail_ids;
  }

  /**
   * {@inheritdoc}
   *
   * Note: This function sets the entity type explicitly as a parameter.
   */
  public static function loadMultipleByProperties($from_sid, $to_sid, $wid, $trigger, $days, $entity_type = 'workflow_notify') {
    return parent::loadMultipleByProperties($from_sid, $to_sid, $wid, $trigger, $days, $entity_type);
  }

  /**
   * {@inheritdoc}
   */
  protected function send(array $to, array $params, string $trigger) {
    if (empty($to)) {
      return;
    }

    // Remove duplicate and empty receivers, then convert to string.
    asort($to);
    $to = array_filter($to);
    $to = array_unique($to);
    $to = implode(', ', $to);

    // @todo Get module_name dynamically from module or namespace.
    $module_name = static::MODULE_NAME;
    $langcode = \Drupal::currentUser()->getPreferredLangcode();
    $send = TRUE;

    $logger = \Drupal::logger('workflow_notifications');
    $messenger = \Drupal::messenger();

    $mailManager = \Drupal::service('plugin.manager.mail');
    $result = $mailManager->mail($module_name, $trigger, $to, $langcode, $params, NULL, $send);
    if ($result['result'] !== TRUE) {
      $logger->error("There was a problem sending your email message and it was not sent.");
      $messenger->addMessage(t('There was a problem sending your email message and it was not sent.'), 'error');
    }
    else {
      $logger->notice("Email sent successfully to $to");
      $messenger->addMessage(t('Email sent successfully.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultTriggerId() {
    return 'workflow_notification_before_mail_trigger';
  }

  /**
   * {@inheritdoc}
   */
  protected function getReceiverIdFromUser($user) {
    return $user->getEmail();
  }

}
