<?php

namespace Drupal\workflow_notifications\Form;

use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Token;
use Drupal\workflow\WorkflowTypeAttributeTrait;
use Drupal\workflow_notifications\Entity\WorkflowNotification;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A form with details for the Notification.
 */
class WorkflowNotificationForm extends EntityForm {

  use WorkflowTypeAttributeTrait;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * The email validator.
   *
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected $emailValidator;

  /**
   * Constructor.
   */
  public function __construct(ModuleHandlerInterface $module_handler, Token $token, EmailValidatorInterface $email_validator) {
    $this->moduleHandler = $module_handler;
    $this->token = $token;
    $this->emailValidator = $email_validator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('token'),
      $container->get('email.validator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\workflow_notifications\Entity\WorkflowNotification $notification */
    $notification = $this->entity;
    // Accomodate WorkflowTypeAttributeTrait.
    if ($notification->isNew()) {
      $workflow = workflow_url_get_workflow();
      $notification->setWorkflow($workflow);
    }
    $workflow = $notification->getWorkflow();
    $this->setWorkflow($workflow);

    $role_options = workflow_allowed_user_role_names('');
    unset($role_options['anonymous']);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $notification->label(),
      '#required' => TRUE,
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $notification->id(),
      '#machine_name' => [
        'exists' => [$this, 'exists'],
      ],
      '#disabled' => !$notification->isNew(),
    ];
    $form['wid'] = [
      '#type' => 'hidden',
      '#default_value' => $notification->getWorkflowId(),
    ];
    $form['trigger'] = [
      '#type' => 'details',
      '#title' => $this->t('Trigger'),
      '#collapsible' => TRUE,
      '#open' => TRUE,
    ];
    // @todo This should be moved into Workflow service.
    $state_options = ['all' => $this->t('Any State')];
    $state_options += workflow_allowed_workflow_state_names($workflow->id(), FALSE, 'CREATION');
    $form['trigger']['from_sid'] = [
      '#type' => 'select',
      '#title' => $this->t('From state'),
      '#options' => $state_options,
      '#default_value' => $notification->from_sid,
    ];
    // @todo This should be moved into Workflow service.
    $state_options = ['all' => $this->t('Any State')];
    $state_options += workflow_allowed_workflow_state_names($workflow->id(), FALSE, FALSE);
    $form['trigger']['to_sid'] = [
      '#type' => 'select',
      '#title' => $this->t('To state'),
      '#options' => $state_options,
      '#default_value' => $notification->to_sid,
      '#states' => [
        'required' => [
          'input[name="when_to_trigger"]' => ['value' => 'no_state_change'],
        ],
      ],
    ];
    $form['trigger']['when_to_trigger'] = [
      '#type' => 'radios',
      '#title' => $this->t('When to trigger'),
      '#options' => workflow_notifications_get_trigger_values(),
      '#default_value' => $notification->when_to_trigger,
      '#description' => $this->t('Determine when the message must be sent:<br/>
        a) direct upon a state change;<br/>
        b) some days before a state change (using scheduled transitions);<br/>
        c) some days after last state change.'),
      '#required' => TRUE,
      '#attributes' => [
        'class' => ['when-to-trigger'],
      ],
    ];
    $form['trigger']['days'] = [
      '#type' => 'number',
      '#title' => $this->t('Days'),
      '#min' => 0,
      '#default_value' => $notification->days,
      '#description' => $this->t('Enter the number of days before a transition is scheduled, a message must be sent.'),
      '#attributes' => [
        'class' => ['time'],
      ],
      '#states' => [
        'invisible' => [
          'input[name="when_to_trigger"]' => ['value' => 'on_state_change'],
        ],
      ],
    ];
    $form['receivers'] = [
      '#type' => 'details',
      '#title' => $this->t('Mail To'),
      '#collapsible' => TRUE,
      '#open' => TRUE,
    ];
    $form['receivers']['roles'] = [
      '#type' => 'checkboxes',
      '#options' => $role_options,
      '#title' => $this->t('Roles'),
      '#default_value' => $notification->roles,
      '#required' => TRUE,
      '#description' => $this->t('Check each role that must be informed. Note: the "Authenticated user" does not select all users. Please user other roles.'),
    ];
    // @todo Add validation for email addresses.
    $form['receivers']['mail_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Email Addresses'),
      '#default_value' => $notification->mail_ids ?? "",
      '#description' => $this->t('Enter valid email address, one per line.'),
    ];
    $form['receivers']['participate'] = [
      '#type' => 'checkbox',
      '#title' => t('Send to Participating users only'),
      '#default_value' => $notification->participate,
    ];
    $form['message'] = [
      '#type' => 'details',
      '#title' => $this->t('Message'),
      '#collapsible' => TRUE,
      '#open' => TRUE,
    ];
    $form['message']['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $notification->subject ?? "",
      '#required' => TRUE,
    ];
    $form['message']['message'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Message'),
      '#default_value' => $notification->message['value'],
      '#format' => $notification->message['format'],
      '#required' => TRUE,
    ];
    $form['note'] = [
      '#type' => 'markup',
      '#markup' => $this->t("<b>Note:</b> Token can be available in the listed fields - MailID, Subject, Message"),
    ];

    // Token support.
    if ($this->moduleHandler->moduleExists('token')) {
      $form['tokens'] = [
        '#title' => $this->t('Tokens'),
        '#type' => 'container',
        '#states' => [
          'invisible' => [
            'input[name="use_token"]' => ['checked' => FALSE],
          ],
        ],
      ];
      $form['tokens']['help'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => [
          'node',
          'workflow_transition',
          'workflow_scheduled_transition',
          'term',
          'site',
          'paragraph',
          'comment',
        ],
        // '#token_types' => 'all'
        '#global_types' => FALSE,
        '#dialog' => TRUE,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // @see WorkflowAbstractNotify::save();
    $form_values = $form_state->getValues();

    $ids = $form_values['mail_ids'] ?? '';
    // @todo What happens here? Is there a use case for tokens in receiver IDs?
    $tokens = $this->moduleHandler->moduleExists('token')
      ? $this->token->scan($ids)
      : FALSE;
    // @todo Trim spaces from mail addresses for better UX.
    if (!empty($ids) && empty($tokens)) {
      $ids = WorkflowNotification::convertToArray($ids);
      foreach ($ids as $index => $id) {
        if (!$this->emailValidator->isValid($id)) {
          $form_state->setErrorByName('mail_ids', $this->t('The email address %id is not valid.', ['%id' => $id]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $notification = $this->entity;
    $status = parent::save($form, $form_state);

    if ($status) {
      $this->messenger()->addMessage($this->t('Saved the %label workflow notification.', [
        '%label' => $notification->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('The %label workflow notification was not saved.', [
        '%label' => $notification->label(),
      ]));
    }

    $form_state->setRedirect('entity.workflow_notify.collection', [
      'workflow_type' => $this->getWorkflowId(),
      'workflow_notify' => $notification->id(),
    ]);

    return $status;
  }

  /**
   * Helper function for machine_name element.
   *
   * @param string $id
   *   The given machine name.
   *
   * @return bool
   *   Indicates if the machine name already exists.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function exists(string $id) {
    $type = $this->entity->getEntityTypeId();
    return (bool) $this->entityTypeManager->getStorage($type)->load($id);
  }

}
