<?php

namespace Drupal\workflow_participants\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessCheck;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\Routing\Route;

/**
 * Access check for revisions and revision history.
 */
class RevisionCheck implements AccessInterface {

  /**
   * Entity access permissions that are handled by this access checker.
   *
   * @var String[]
   */
  const REVISION_PERMISSIONS = [
    'node.view all revisions',
    'node_revision.view revision',
  ];

  /**
   * The entity access checker service.
   *
   * @var \Drupal\Core\Entity\EntityAccessCheck
   */
  protected $entityAccessCheck;

  /**
   * The node storage.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  protected $nodeStorage;

  /**
   * The participant storage.
   *
   * @var \Drupal\workflow_participants\WorkflowParticipantsStorageInterface
   */
  protected $participantStorage;

  /**
   * Constructs the access check.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityAccessCheck $entity_access_check
   *   The default node revision access checker.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityAccessCheck $entity_access_check) {
    $this->entityAccessCheck = $entity_access_check;
    $this->nodeStorage = $entity_type_manager->getStorage('node');
    $this->participantStorage = $entity_type_manager->getStorage('workflow_participants');
  }

  /**
   * Checks access to the node revision and revision history.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   * @param \Drupal\node\NodeInterface $node
   *   (optional) A node object. Used for checking access to a node's default
   *   revision when $node_revision is unspecified. Ignored when $node_revision
   *   is specified. If neither $node_revision nor $node are specified, then
   *   access is denied.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account, ?NodeInterface $node = NULL) {
    $operation = $route->getRequirement('_workflow_participants_revision');
    $participants = $this->participantStorage->loadForModeratedEntity($node);

    // Since the default entity access check has been removed, it is added here
    // for the access checker to verify.
    // @see \Drupal\workflow_participants\Routing\RouteSubscriber::applyRevisionsCheck()
    $route->setRequirement('_entity_access', $operation);

    return $this->entityAccessCheck->access($route, $route_match, $account)
      ->orIf(
        AccessResult::allowedIf(
          $node
          // There should be at least 2 revisions.
          && ($this->nodeStorage->countDefaultLanguageRevisions($node) > 1)
          && in_array($operation, static::REVISION_PERMISSIONS)
          && ($participants->isEditor($account) || $participants->isReviewer($account))
        ))->addCacheableDependency($participants);
  }

}
