<?php

namespace Drupal\workflow_participants\Routing;

use Drupal\Core\Routing\RouteBuildEvent;
use Drupal\Core\Routing\RoutingEvents;
use Drupal\workflow_participants\Access\RevisionCheck;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Routing\Route;

/**
 * Route subscriber to alter access for content moderation routes.
 */
class RouteSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[RoutingEvents::ALTER] = ['alterRoutes', 50];
    return $events;
  }

  /**
   * Alter content moderation route permissions callbacks.
   *
   * @param \Drupal\Core\Routing\RouteBuildEvent $event
   *   The route build event.
   */
  public function alterRoutes(RouteBuildEvent $event) {
    // @todo This is hard-coded for nodes.
    // @see https://www.drupal.org/node/2922348
    $collection = $event->getRouteCollection();

    // Revision history tab.
    if ($route = $collection->get('entity.node.version_history')) {
      $this->applyRevisionsCheck($route);
    }

    // Node revision.
    if ($route = $collection->get('entity.node.revision')) {
      $this->applyRevisionsCheck($route);
    }

    // Diff module's version comparison.
    if ($route = $collection->get('diff.revisions_diff')) {
      $this->applyRevisionsCheck($route);
    }
  }

  /**
   * Adds the revision access check.
   *
   * This also removes the `_access_node_revision` check, and that is checked
   * by the `_workflow_participants_revision` check.
   *
   * @todo Investigate decorating the node access check instead.
   */
  protected function applyRevisionsCheck(Route $route) {
    // Remove the `_entity_access` check for specific permissions.
    $requirements = $route->getRequirements();
    if (isset($requirements['_entity_access']) && in_array($requirements['_entity_access'], RevisionCheck::REVISION_PERMISSIONS)) {
      $requirements['_workflow_participants_revision'] = $requirements['_entity_access'];
      // This check is added back in the revision check.
      // @see \Drupal\workflow_participants\Access\RevisionCheck::access()
      unset($requirements['_entity_access']);
      $route->setRequirements($requirements);
    }
  }

}
