<?php

namespace Drupal\workflow_participants_auto\Hook;

use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Hook implementations for Workflow Participants Auto.
 */
class WorkflowParticipantsAutoHooks {

  /**
   * The moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformationInterface
   */
  private ModerationInformationInterface $moderationInformation;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the hook object.
   */
  public function __construct(
    ModerationInformationInterface $moderationInformation,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->moderationInformation = $moderationInformation;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Implements hook_entity_type_build().
   */
  #[Hook('entity_type_build')]
  public function entityTypeBuild(array &$entityTypes): void {
    /** @var \Drupal\Core\Entity\EntityTypeInterface[] $entityTypes */
    $entityTypes['workflow']
      ->setFormClass('workflow_participants', 'Drupal\workflow_participants_auto\Form\WorkflowParticipantsForm')
      ->setLinkTemplate('workflow-participants-form', '/admin/config/workflow/workflows/manage/{workflow}/participants');
  }

  /**
   * Implements hook_entity_insert().
   */
  #[Hook('entity_insert')]
  public function entityInsert(EntityInterface $entity): void {
    // Add participants automatically if they are configured for the workflow.
    if ($entity instanceof ContentEntityInterface) {
      if ($workflow = $this->moderationInformation->getWorkflowForEntity($entity)) {
        $participants = $workflow->getThirdPartySettings('workflow_participants_auto');
        if (!empty($participants['reviewers']) || !empty($participants['editors'])) {
          $userStorage = $this->entityTypeManager->getStorage('user');
          foreach ($participants as $key => $uids) {
            if (!empty($uids)) {
              /** @var \Drupal\user\UserInterface[] $users */
              $users = $userStorage->loadMultiple($uids);
              foreach ($participants[$key] as $index => $uid) {
                if (empty($users[$uid]) || !$users[$uid]->isActive()) {
                  unset($participants[$key][$index]);
                }
              }
            }
          }
          $this->entityTypeManager->getStorage('workflow_participants')->create([
            'moderated_entity' => $entity,
            'reviewers' => $participants['reviewers'] ?? [],
            'editors' => $participants['editors'] ?? [],
          ])->save();
        }
      }
    }
  }

}
