<?php

namespace Drupal\workflow_participants\Hook;

use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\workflow_participants\Access\WorkflowParticipantsAccessChecker;
use Drupal\workflow_participants\Entity\WorkflowParticipantsInterface;
use Drupal\workflow_participants\ParticipantNotifierInterface;

/**
 * Entity hook implementations for the Workflow Participants module.
 */
class WorkflowParticipantsEntityHooks {

  use StringTranslationTrait;

  /**
   * The access checker.
   *
   * @var \Drupal\workflow_participants\Access\WorkflowParticipantsAccessChecker
   */
  private WorkflowParticipantsAccessChecker $accessChecker;

  /**
   * The moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformationInterface
   */
  private ModerationInformationInterface $moderationInformation;

  /**
   * The participant notifier.
   *
   * @var \Drupal\workflow_participants\ParticipantNotifierInterface
   */
  private ParticipantNotifierInterface $notifier;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the hook object.
   */
  public function __construct(
    WorkflowParticipantsAccessChecker $accessChecker,
    ModerationInformationInterface $moderationInformation,
    ParticipantNotifierInterface $notifier,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->accessChecker = $accessChecker;
    $this->moderationInformation = $moderationInformation;
    $this->notifier = $notifier;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Implements hook_entity_access().
   */
  #[Hook('entity_access')]
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {
    if (!$entity instanceof ContentEntityInterface || !$entity->id()) {
      return AccessResult::neutral();
    }

    return $this->accessChecker->hasEntityAccess($entity, $operation, $account);
  }

  /**
   * Implements hook_entity_operation().
   */
  #[Hook('entity_operation')]
  public function entityOperation(EntityInterface $entity): array {
    $operations = [];
    if ($entity->hasLinkTemplate('workflow-participants') && $this->moderationInformation->isModeratedEntity($entity)) {
      $operations['workflow-participants'] = [
        'title' => $this->t('Workflow participants'),
        'url' => $entity->toUrl('workflow-participants'),
        'weight' => 27,
      ];
    }
    return $operations;
  }

  /**
   * Implements hook_ENTITY_TYPE_insert() and hook_ENTITY_TYPE_update().
   */
  #[Hook('workflow_participants_insert')]
  #[Hook('workflow_participants_update')]
  public function processNotifications(WorkflowParticipantsInterface $entity): void {
    $this->notifier->processNotifications($entity);
  }

  /**
   * Implements hook_entity_delete().
   */
  #[Hook('entity_delete')]
  public function entityDelete(EntityInterface $entity): void {
    if ($entity instanceof ContentEntityInterface) {
      // Remove corresponding participants entity when the moderated entity is
      // deleted.
      /** @var \Drupal\workflow_participants\Entity\WorkflowParticipantsInterface $participants */
      $participants = $this->entityTypeManager->getStorage('workflow_participants')->loadForModeratedEntity($entity);
      if (!$participants->isNew()) {
        $participants->delete();
      }
    }
  }

}
