<?php

namespace Drupal\workflow_participants\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\workflow_participants\EntityOperations;

/**
 * Standard hook implementations for the Workflow Participants module.
 */
class WorkflowParticipantsHooks {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the hook object.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $routeName) {
    if ($routeName == 'help.page.workflow_participants') {
      $output = '';
      $output .= '<h3>' . $this->t('About') . '</h3>';
      $output .= $this->t('This module allows per-entity workflow participants to be configured. These participants can either be editors or reviewers. Content moderation states can be configured to allow editors, reviewers, or both, to make transitions. Reviewers cannot edit the content, only moderate. Editors can moderate and make changes.');
      $output .= '<h3>' . $this->t('Setup and usage') . '</h3>';
      $output .= '<dl>';
      $output .= '<dt>' . $this->t('Permissions') . '</dt>';
      $output .= '<dd>' . $this->t('Optionally configure roles with the <em>Allowed to be a workflow editor or reviewer</em> permission. If this is skipped, any active user can be an editor or reviewer.') . '</dd>';
      $output .= '<dt>' . $this->t('Moderation state transitions') . '</dt>';
      $output .= '<dd>' . $this->t('Edit <a href=":workflows">moderation states</a> and check the <em>Allow editors</em> and <em>Allow reviewers</em> as needed.', [
        ':workflows' => Url::fromRoute('entity.workflow.collection')->toString(),
      ]) . '</dd>';
      $output .= '</dl>';
      return $output;
    }
  }

  /**
   * Implements hook_form_BASE_ID_alter().
   */
  #[Hook('form_workflow_transition_add_form_alter')]
  #[Hook('form_workflow_transition_edit_form_alter')]
  public function formAlterWorkflowTransitionForm(array &$form, FormStateInterface $formState): void {
    // Add editor and reviewer checkboxes.
    /** @var \Drupal\workflows\WorkflowInterface $workflow */
    $workflow = $formState->getFormObject()->getEntity();
    $transitionId = $form['id']['#value'] ?? '';
    $form['editor_transitions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow editors to make this transition'),
      '#default_value' => in_array($transitionId, $workflow->getThirdPartySetting('workflow_participants', 'editor_transitions', [])),
    ];
    $form['reviewer_transitions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow reviewers to make this transition'),
      '#default_value' => in_array($transitionId, $workflow->getThirdPartySetting('workflow_participants', 'reviewer_transitions', [])),
    ];
    $form['#entity_builders'][] = EntityOperations::class . '::workflowTransitionsFormBuilder';
  }

  /**
   * Implements hook_form_BASE_ID_alter().
   */
  #[Hook('form_content_moderation_notification_form_alter')]
  public function formAlterContentModerationNotificationForm(array &$form, FormStateInterface $formState): void {
    /** @var \Drupal\content_moderation_notifications\ContentModerationNotificationInterface $contentModerationNotification */
    $contentModerationNotification = $formState->getFormObject()->getEntity();

    // Add participant selection.
    $defaults = [
      'editors' => $contentModerationNotification->getThirdPartySetting('workflow_participants', 'editors', FALSE),
      'reviewers' => $contentModerationNotification->getThirdPartySetting('workflow_participants', 'reviewers', FALSE),
    ];
    $form['workflow_participants'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Workflow participants'),
      '#description' => $this->t('Send this notification to workflow participants of the selected type(s).'),
      '#options' => [
        'editors' => $this->t('Editors'),
        'reviewers' => $this->t('Reviewers'),
      ],
      '#default_value' => array_keys(array_filter($defaults)),
    ];
    $form['#entity_builders'][] = EntityOperations::class . '::notificationFormBuilder';
  }

  /**
   * Implements hook_mail().
   */
  #[Hook('mail')]
  public function mail($key, &$message, $params): void {
    if ($key === 'new_participant') {
      $message['subject'] = $params['subject'];
      $message['body'] = [$params['body']];
    }
  }

  /**
   * Implements hook_content_moderation_notification_mail_data_alter().
   */
  #[Hook('content_moderation_notification_mail_data_alter')]
  public function contentModerationNotificationMailDataAlter(EntityInterface $entity, array &$data): void {
    /** @var \Drupal\content_moderation_notifications\ContentModerationNotificationInterface $notification */
    $notification = $data['notification'];
    if ($notification->getThirdPartySetting('workflow_participants', 'reviewers', FALSE) || $notification->getThirdPartySetting('workflow_participants', 'editors', FALSE)) {
      /** @var \Drupal\workflow_participants\Entity\WorkflowParticipantsInterface $participants */
      $participants = $this->entityTypeManager->getStorage('workflow_participants')->loadForModeratedEntity($entity);
      if ($notification->getThirdPartySetting('workflow_participants', 'editors', FALSE)) {
        foreach ($participants->getEditors() as $account) {
          $data['to'][] = $account->getEmail();
        }
      }
      if ($notification->getThirdPartySetting('workflow_participants', 'reviewers', FALSE)) {
        foreach ($participants->getReviewers() as $account) {
          $data['to'][] = $account->getEmail();
        }
      }
    }
  }

}
