<?php

declare(strict_types=1);

namespace Drupal\workflows_graph\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\workflows\WorkflowInterface;
use Drupal\workflows\WorkflowTypeInterface;
use Drush\Attributes\Argument;
use Drush\Attributes\Command;
use Drush\Attributes\Usage;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for workflows_graph module.
 */
final class WorkflowsGraphCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Builds a new instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager service.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Generates a dot file for a workflow.
   */
  #[Command(name: 'workflows_graph:draw', aliases: ['wg-draw'])]
  #[Argument(name: 'workflow_name', description: 'Workflow machine name.')]
  #[Usage(name: 'workflows_graph:draw foo', description: 'Generates a dot file for "foo" workflow states.')]
  public function generateDot($workflow_name) {
    $workflow = $this->entityTypeManager
      ->getStorage('workflow')
      ->load($workflow_name);
    if (!$workflow instanceof WorkflowInterface) {
      $this->io()->error("Cannot find '$workflow_name' workflow configuration.");
      return static::EXIT_FAILURE;
    }
    $type = $workflow->getTypePlugin();
    if (!$type instanceof WorkflowTypeInterface) {
      $this->io()->error("Invalid workflow type plugin on '$workflow_name' workflow configuration.");
      return static::EXIT_FAILURE;
    }
    $dot_lines = [];
    $dot_lines[] = sprintf('digraph %s {', $workflow_name);
    foreach ($type->getStates() as $state) {
      $state_id = $state->id();
      $state_label = sprintf('%s\n(%s)', $state->label(), $state_id);
      $dot_lines[] = sprintf('%s [label="%s"];', $state_id, $state_label);
    }
    foreach ($type->getTransitions() as $transition) {
      $end_id = $transition->to()->id();
      $transition_label = $transition->label();
      foreach ($transition->from() as $from_state) {
        $start_id = $from_state->id();
        $dot_lines[] = sprintf('%s -> %s [label="%s"];', $start_id, $end_id, $transition_label);
      }
    }
    $dot_lines[] = '}';
    $this->io()->text($dot_lines);
  }

}
