# Workspaces Access

A Drupal module that provides granular permissions for editing content within specific workspaces.

## Overview

The Workspaces Access module extends Drupal's core Workspaces module by providing fine-grained permissions for content within workspaces. Instead of the all-or-nothing approach of the core "bypass entity access own workspace" permission, this module creates individual permissions for each workspace.

## Features

- **Granular Permissions**: Generate operation-specific permissions like "workspace_staging_add_content", "workspace_staging_edit_content", and "workspace_staging_remove_content" for each workspace
- **Selective Access Control**: Allow/block specific operations based on granular workspace permissions
- **Dynamic Permission Generation**: Automatically create permissions when workspaces are created
- **Role-Based Assignment**: Assign workspace permissions to user roles through Drupal's permission interface

## Installation

1. Place the `workspaces_access` module in your `web/modules/custom/` directory
2. Enable the module: `drush en workspaces_access`
3. Clear caches: `drush cr`

## Usage

### Permission Assignment

After installation, new permissions will appear in Drupal's permission administration interface. You can access them in two ways:

1. **Direct Link**: Navigate to `/admin/people/permissions`
2. **Module Link**: Go to Administration → Configuration → People → Workspaces Access Permissions

#### Available Permissions

**Important:** To add or edit content in a workspace, users need BOTH the workspace permission AND the corresponding view permission. For example:
- Add: "Staging - View content" + "Staging - Add content" + "create article content"
- Edit: "Staging - View content" + "Staging - Edit content" + "edit any article content"
- Without view permission, the forms may load but post-save redirects will fail.

**For Non-Live Workspaces:**
- "[workspace_name] - View content" - Role can view content in workspace. Requires relevant permissions (e.g., "view article content") for operations.
- "[workspace_name] - Add content" - Role can create content in workspace. Requires "View" permission for add functionality and relevant permissions (e.g., "create article content") for operations.
- "[workspace_name] - Edit content" - Role can edit content in workspace. Requires "View" permission for edit functionality and relevant permissions (e.g., "edit article content") for operations.
- "[workspace_name] - Remove content" - Role can delete content in workspace. Requires relevant permissions (e.g., "delete article content") for operations.

**For Live Workspace:**
- " Live - View content" - Role can view content in workspace. Requires relevant permissions (e.g., "view article content") for operations.
- " Live - Add content" - Role can create content in workspace. Requires "View" permission for add functionality and relevant permissions (e.g., "create article content") for operations.
- " Live - Edit content" - Role can edit content in workspace. Requires "View" permission for edit functionality and relevant permissions (e.g., "edit article content") for operations.
- " Live - Remove content" - Role can delete content in workspace. Requires relevant permissions (e.g., "delete article content") for operations.

### How It Works

#### Non-Live Workspaces
1. **Operation-Specific Permission Check**: Users must have the appropriate operation-specific permission (add/edit/remove) for the workspace
2. **Normal Drupal Permissions**: Apply within allowed workspaces for the specific operation

#### Live Workspace
1. **Operation-Specific Permission Check**: Users must have the appropriate operation-specific permission (add/edit/remove) for Live workspace
2. **Normal Drupal Permissions**: Apply within Live workspace for the specific operation
3. **Access Denied**: Users without the specific Live workspace operation permission are **explicitly denied** access to perform that operation in Live workspace

### Example Scenarios

#### Scenario 1: Standard User
- User is in a role with basic content permissions
- User is not in a role with any Live workspace permissions

Result: **Explicitly denied** access to create, edit, or delete in Live workspace

#### Scenario 2: Content Editor
- User is in a role with basic content permissions
- User is in a role with " Live - View content" and " Live - Edit content" permissions

Result: **Can** edit existing content in Live workspace

#### Scenario 3: Administrator
- User is in a role with " Live - View content", " Live - Add content", " Live - Edit content", and " Live - Remove content" permissions
- User is in a role with basic content permissions

Result: **Can** perform all operations in Live workspace

#### Scenario 4: Granular Workspace Access
- User has "Staging - View content", "Staging - Add content" and "Staging - Edit content" permissions
- User lacks any permissions for "development" workspace

Result:
- Can view, create and edit content when in "staging" workspace
- Cannot perform any operations when in "development" workspace

#### Scenario 5: Content Creator Role
- User has "Staging - View content" and "Staging - Add content" permissions
- User lacks "Staging - Edit content" and "Staging - Remove content" permissions

Result:
- Can view and create new content in "staging" workspace
- Cannot edit or delete existing content in "staging" workspace

## Custom Business Logic Integration

The Workspaces Access module provides an extensible event system that allows you to implement custom business-specific access restrictions in your own modules.

### Adding Business-Specific Restrictions

You can create custom modules that subscribe to workspace access events to implement your own business rules. This allows for clean separation between generic workspace permissions and project-specific access control logic.

#### 1. Create an Event Subscriber

Create a new class that implements `EventSubscriberInterface`:

```php
<?php

declare(strict_types=1);

namespace Drupal\my_business_module\EventSubscriber;

use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for custom workspace access restrictions.
 */
class WorkspaceAccessSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'workspaces_access.workspace_check' => 'onWorkspaceAccessCheck',
    ];
  }

  /**
   * Implements custom workspace access restrictions.
   *
   * @param \Drupal\workspaces_access\Event\WorkspaceAccessEvent $event
   *   The workspace access event.
   */
  public function onWorkspaceAccessCheck(WorkspaceAccessEvent $event): void {
    $workspaceName = $event->getWorkspaceName();
    $operation = $event->getOperation();
    $entityTypeId = $event->getEntityTypeId();
    $bundle = $event->getBundle();

    // Only apply restrictions to editing operations
    if (!in_array($operation, ['create', 'update', 'delete'])) {
      return;
    }

    // Example: Block certain content types in specific workspaces
    if ($workspaceName === 'production' && $bundle === 'unpublished_content') {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Cannot edit unpublished content in production workspace'
        )->cachePerPermissions()
      );
      return;
    }

    // Example: Allow specific operations in certain workspaces
    if ($workspaceName === 'staging' && $entityTypeId === 'media' && $operation === 'create') {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::allowed(
          'Media creation allowed in staging workspace'
        )->cachePerPermissions()
      );
      return;
    }
  }

}
```

#### 2. Register Your Event Subscriber

Create a service definition in your module's `services.yml` file:

```yaml
# my_business_module.services.yml
services:
  my_business_module.workspace_access_subscriber:
    class: Drupal\my_business_module\EventSubscriber\WorkspaceAccessSubscriber
    tags:
      - { name: event_subscriber }
```

#### 3. Enable Your Module

Enable your custom module and the event subscriber will automatically be registered.

### Real-World Example: RAGS/CAT Restrictions

Here's an example of how the PDC module implements business-specific restrictions:

```php
<?php

declare(strict_types=1);

namespace Drupal\pdc\EventSubscriber;

use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for PDC-specific workspace access restrictions.
 */
class WorkspaceAccessSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'workspaces_access.workspace_check' => 'onWorkspaceAccessCheck',
    ];
  }

  /**
   * Implements PDC-specific workspace access restrictions.
   */
  public function onWorkspaceAccessCheck(WorkspaceAccessEvent $event): void {
    $workspaceName = $event->getWorkspaceName();
    $operation = $event->getOperation();
    $entityTypeId = $event->getEntityTypeId();
    $bundle = $event->getBundle();

    // Only apply restrictions to editing operations
    if (!in_array($operation, ['create', 'update', 'delete'])) {
      return;
    }

    $typeToCheck = !empty($bundle) ? $bundle : $entityTypeId;

    // Business rule: RAGS workspaces cannot access CAT content
    if (stripos($workspaceName, 'rags') !== false &&
        stripos($typeToCheck, 'cat') !== false) {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Access blocked: workspace contains "rags" and content contains "cat"'
        )->cachePerPermissions()
      );
      return;
    }

    // Business rule: CAT workspaces cannot access RAGS content
    if (stripos($workspaceName, 'cat') !== false &&
        stripos($typeToCheck, 'rags') !== false) {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Access blocked: workspace contains "cat" and content contains "rags"'
        )->cachePerPermissions()
      );
      return;
    }
  }

}
```

### Event System Benefits

- **Separation of Concerns**: Keep generic workspace logic separate from business rules
- **Extensibility**: Multiple modules can subscribe to the same event
- **Override Capability**: Business rules can override or extend base permissions
- **Clean Architecture**: Event-driven design for maintainable code
- **Performance**: Events are only dispatched when needed

### Available Event Data

The `WorkspaceAccessEvent` provides access to:

- `$event->getWorkspaceName()`: Current workspace name
- `$event->getOperation()`: Operation being performed (view, create, update, delete)
- `$event->getEntityTypeId()`: Entity type machine name
- `$event->getBundle()`: Entity bundle (content type, media type, etc.)
- `$event->getEntity()`: The actual entity object (when available)
- `$event->getAccount()`: The user account performing the operation
- `$event->setAccessResult(AccessResultInterface)`: Set the access result

## Technical Details

### Permission Override Logic

The module implements `hook_entity_access()` and `hook_entity_create_access()` to:

1. Check if the current request is within a workspace
2. Verify if the entity type is supported by workspaces
3. Check if the user has the workspace-specific permission
4. Allow or deny access accordingly

### Dynamic Permissions

Permissions are generated dynamically using `hook_permissions()` and updated when workspaces are created, modified, or deleted through event subscribers.

### Caching

The module properly implements cache invalidation to ensure permissions are updated when workspaces change.

## Dependencies

- Drupal Core: ^10 || ^11
- Workspaces module (core)

## Compatibility

This module is designed to work alongside existing workspace functionality and respects:
- Workspace ownership
- Existing access patterns
- Core workspace permissions

## Testing

The module includes comprehensive tests covering:
- Permission generation
- Access control logic
- Workspace lifecycle events
- Edge cases and error conditions

## Security Considerations

- **Live Workspace Protection**: The "Edit content in Live workspace" permission provides controlled access to production content
- **Granular Access Control**: Administrators can grant Live workspace access to specific roles as needed
- **Principle of Least Privilege**: All workspaces require explicit permissions, preventing accidental production edits
- **Restrictive Permissions**: All workspace permissions are marked as "restrict access" for enhanced security
- **Cache Security**: Proper cache contexts prevent permission leakage
- **Entity-Level Enforcement**: Access control is enforced at the entity level for comprehensive protection

## Support

For issues and feature requests, please create an issue in the project repository.
