<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Hook;

use Drupal\node\NodeInterface;
use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceInformationInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Defines a class for reacting to entity access control hooks.
 */
class EntityAccess {

  public function __construct(
    protected WorkspaceManagerInterface $workspaceManager,
    protected WorkspaceInformationInterface $workspaceInfo,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Gets the workspace name for a given workspace ID.
   *
   * @param string $workspaceId
   *   The workspace ID.
   *
   * @return string
   *   The workspace name/label, or empty string if not found.
   */
  private function getWorkspaceName(string $workspaceId): string {
    if ($workspaceId === 'live') {
      return 'live';
    }

    try {
      $workspace = $this->entityTypeManager->getStorage('workspace')
        ->load($workspaceId);
      return $workspace ? $workspace->label() : '';
    }
    catch (\Exception $e) {
      return '';
    }
  }

  /**
   * Check access blocked based on workspace and entity/bundle restrictions.
   *
   * This method dispatches events to allow other modules to implement custom
   * workspace-specific access restrictions.
   *
   * @param string $workspaceId
   *   The workspace ID.
   * @param string $entityTypeId
   *   The entity type machine name.
   * @param string $operation
   *   The operation being performed.
   * @param string $bundle
   *   The entity bundle (optional, for nodes this would be the node type).
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity being accessed (for event dispatching).
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user account (for event dispatching).
   *
   * @return \Drupal\Core\Access\AccessResultInterface|null
   *   AccessResult::forbidden() if access should be blocked, null otherwise.
   */
  protected function checkWorkspaceEntityRestrictions(
    string $workspaceId,
    string $entityTypeId,
    string $operation,
    string $bundle = '',
    ?EntityInterface $entity = NULL,
    ?AccountInterface $account = NULL,
  ): ?AccessResultInterface {
    // Only apply restrictions to editing operations.
    if (!in_array($operation, ['create', 'update', 'delete'])) {
      return NULL;
    }

    // Get the workspace name for checking.
    $workspaceName = $this->getWorkspaceName($workspaceId);
    if (empty($workspaceName)) {
      // If we can't load the workspace, don't apply restrictions.
      return NULL;
    }

    // Dispatch event to allow other modules to implement custom restrictions.
    if ($account) {
      // For create operations without an entity, create a temporary entity for
      // the event.
      if ($operation === 'create' && !$entity) {
        // Create a temporary entity for event dispatching.
        $entityStorage = $this->entityTypeManager->getStorage($entityTypeId);
        $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
        $bundleKey = $entityType->getKey('bundle');

        // Set the bundle using the correct field name.
        $values = [];
        if ($bundleKey && $bundle) {
          $values[$bundleKey] = $bundle;
        }

        $entity = $entityStorage->create($values);
      }

      if ($entity) {
        $event = new WorkspaceAccessEvent(
          $entity,
          $operation,
          $account,
          $workspaceId,
          $workspaceName
        );

        $this->eventDispatcher->dispatch($event, 'workspaces_access.workspace_check');

        // If any subscriber set an access result, return it.
        if ($event->getAccessResult()) {
          return $event->getAccessResult();
        }
      }
    }

    return NULL;
  }

  /**
   * Implements hook_entity_access().
   */
  #[Hook('entity_access', order: Order::First)]
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {

    // Only apply to supported entities.
    if (!$this->workspaceInfo->isEntitySupported($entity)) {
      return AccessResult::neutral();
    }

    // Determine workspace context.
    $hasActiveWorkspace = $this->workspaceManager->hasActiveWorkspace();
    if ($hasActiveWorkspace) {
      $activeWorkspace = $this->workspaceManager->getActiveWorkspace();
      $workspaceId = $activeWorkspace->id();
    }
    else {
      // We're in Live workspace.
      $workspaceId = 'live';
      // No active workspace object for Live.
      $activeWorkspace = NULL;
    }

    // Check workspace-entity type restrictions first (these override all other
    // permissions)
    $entityTypeId = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $restrictionResult = $this->checkWorkspaceEntityRestrictions(
      $workspaceId,
      $entityTypeId,
      $operation,
      $bundle,
      $entity,
      $account
    );
    if ($restrictionResult) {
      return $restrictionResult;
    }

    // Handle both view and editing operations in workspaces.
    if (in_array($operation, ['view', 'create', 'update', 'delete'])) {
      // Special handling for Live workspace with granular permissions.
      if ($workspaceId === 'live') {
        // Check if user has the operation-specific Live workspace permission.
        $operationName = match($operation) {
          'create' => 'add',
          'update' => 'edit',
          'delete' => 'remove',
          default => $operation,
        };
        $liveOperationPermission = 'workspace_live_' . $operationName .
          '_content';
        if ($account->hasPermission($liveOperationPermission)) {
          // User has Live operation permission, now check basic content
          // permissions.
          $entityType = $entity->getEntityType();
          $basicPermission = $operation . ' ' .
          $entityType->getSingularLabel();

          if ($account->hasPermission($basicPermission) || $account->hasPermission($operation . ' any content')) {
            return AccessResult::allowed('User has Live workspace and content permissions')
              ->cachePerPermissions();
          }

          return AccessResult::neutral();
        }

        // User doesn't have the specific Live workspace operation permission.
        return AccessResult::forbidden('User does not have permission to ' . $operation .
          ' content in Live workspace')
          ->cachePerPermissions();
      }
      else {
        // For non-Live workspaces, check operation-specific workspace
        // permission.
        $operationName = match($operation) {
          'create' => 'add',
           'update' => 'edit',
           'delete' => 'remove',
           default => $operation,
        };
        $operationPermission = 'workspace_' . $workspaceId . '_' . $operationName . '_content';
        if (!$account->hasPermission($operationPermission)) {
          return AccessResult::forbidden('User does not have permission to ' . $operation . ' content in this workspace')
            ->cachePerPermissions();
        }
      }

      // At this point, user has passed workspace permission checks
      // now check if they also have basic content permissions.
      $entityType = $entity->getEntityType();
      $basicPermission = $operation . ' ' . $entityType->getSingularLabel();

      // For node entities, return neutral to allow node access modules to apply
      // restrictions.
      // For other entities, allow if user has workspace and basic permissions.
      if ($entityTypeId === 'node') {
        // For nodes, let node access modules handle the final access decision.
        return AccessResult::neutral();
      }

      // For non-node entities, if user has both workspace permission and basic
      // permission, explicitly allow.
      if ($account->hasPermission($basicPermission) || $account->hasPermission($operation . ' any content')) {
        return AccessResult::allowed('User has workspace and content permissions')
          ->cachePerPermissions();
      }

      // User has workspace permission but lacks basic content permission.
      return AccessResult::neutral();
    }

    // For other operations, return neutral.
    return AccessResult::neutral();
  }

  /**
   * Implements hook_node_access().
   */
  #[Hook('node_access', order: Order::First)]
  public function nodeAccess(NodeInterface $node, $operation, AccountInterface $account): AccessResultInterface {

    // Only apply to supported operations.
    if (!in_array($operation, ['view', 'update', 'delete'])) {
      return AccessResult::neutral();
    }

    // Check if we're in a workspace.
    $hasActiveWorkspace = $this->workspaceManager->hasActiveWorkspace();
    if ($hasActiveWorkspace) {
      // We're in a non-Live workspace.
      $activeWorkspace = $this->workspaceManager->getActiveWorkspace();
      $workspaceId = $activeWorkspace->id();
    }
    else {
      // We're in Live workspace (hasActiveWorkspace = FALSE)
      $workspaceId = 'live';
    }

    // Check workspace-entity type restrictions first (these override all other
    // permissions)
    $entityTypeId = $node->getEntityTypeId();
    $bundle = $node->bundle();
    $restrictionResult = $this->checkWorkspaceEntityRestrictions(
      $workspaceId,
      $entityTypeId,
      $operation,
      $bundle,
      $node,
      $account
    );
    if ($restrictionResult) {
      return $restrictionResult;
    }

    // Handle Live workspace permission.
    if ($workspaceId === 'live') {
      // Check if user has the operation-specific Live workspace permission.
      $operationName = match($operation) {
        'create' => 'add',
        'update' => 'edit',
        'delete' => 'remove',
        default => $operation,
      };
      $liveOperationPermission = 'workspace_live_' . $operationName . '_content';
      if ($account->hasPermission($liveOperationPermission)) {
        // User has Live operation permission, now check basic content
        // permissions.
        $entityType = $node->getEntityType();
        $basicPermission = $operation . ' ' .
        $entityType->getSingularLabel();

        if ($account->hasPermission($basicPermission) || $account->hasPermission($operation . ' any content')) {
          return AccessResult::allowed('User has Live workspace and content permissions')
            ->cachePerPermissions();
        }

        return AccessResult::neutral();
      }

      // User doesn't have the specific Live workspace operation permission.
      return AccessResult::forbidden('User does not have permission to ' . $operation . ' content in Live workspace')
        ->cachePerPermissions();
    }
    else {
      // For non-Live workspaces, check operation-specific workspace permission.
      $operationName = match($operation) {
        'create' => 'add',
        'update' => 'edit',
        'delete' => 'remove',
        default => $operation,
      };
      $operationPermission = 'workspace_' . $workspaceId . '_' . $operationName . '_content';
      if (!$account->hasPermission($operationPermission)) {
        return AccessResult::forbidden('User does not have permission to ' . $operation . ' content in this workspace')
          ->cachePerPermissions();
      }
    }

    // At this point, user has passed workspace permission
    // checks.
    // Now check if they also have basic content permissions.
    $entityType = $node->getEntityType();
    $basicPermission = $operation . ' ' . $entityType->getSingularLabel();

    // For node entities, return neutral to allow node access modules to apply
    // restrictions.
    // This ensures node access module permissions are respected.
    return AccessResult::neutral();
  }

  /**
   * Implements hook_entity_create_access().
   */
  #[Hook('entity_create_access', order: Order::First)]
  public function entityCreateAccess(AccountInterface $account, array $context, $entity_bundle): AccessResultInterface {
    // Only apply to supported entity types.
    $entity_type = $this->entityTypeManager->getDefinition($context['entity_type_id']);
    if (!$this->workspaceInfo->isEntityTypeSupported($entity_type)) {
      return AccessResult::neutral();
    }

    // Determine workspace context.
    $hasActiveWorkspace = $this->workspaceManager->hasActiveWorkspace();
    if ($hasActiveWorkspace) {
      $activeWorkspace = $this->workspaceManager->getActiveWorkspace();
      $workspaceId = $activeWorkspace->id();
    }
    else {
      // We're in Live workspace.
      $workspaceId = 'live';
      // No active workspace object for Live.
      $activeWorkspace = NULL;
    }

    // Check workspace-entity type restrictions first
    // (these override all other permissions).
    $entityTypeId = $context['entity_type_id'];
    $bundle = $entity_bundle ?? '';

    // For entity creation, we don't have an entity object yet, so we pass null
    // but still allow event dispatching for business logic modules.
    $restrictionResult = $this->checkWorkspaceEntityRestrictions(
      $workspaceId,
      $entityTypeId,
      'create',
      $bundle,
      NULL,
      $account
    );
    if ($restrictionResult) {
      return $restrictionResult;
    }

    // Handle Live workspace permission.
    if ($workspaceId === 'live') {
      // Check if user has the create-specific Live workspace permission.
      $liveCreatePermission = 'workspace_live_add_content';
      if ($account->hasPermission($liveCreatePermission)) {
        // User has Live create permission, now check basic create permissions.
        $basicPermission = 'create ' . $entity_type->getSingularLabel();
        if ($account->hasPermission($basicPermission) || $account->hasPermission('create any content')) {
          return AccessResult::allowed('User has Live workspace and create permissions')
            ->cachePerPermissions();
        }
        return AccessResult::neutral();
      }

      // User doesn't have Live workspace create permission.
      return AccessResult::forbidden('User does not have permission to create content in Live workspace')
        ->cachePerPermissions();
    }
    else {
      // For non-Live workspaces, check create-specific workspace permission.
      $createPermission = 'workspace_' . $workspaceId . '_add_content';
      if (!$account->hasPermission($createPermission)) {
        return AccessResult::forbidden('User does not have permission to create content in this workspace')
          ->cachePerPermissions();
      }
    }

    // Get workspace name for special media handling.
    $workspaceName = $this->getWorkspaceName($workspaceId);

    // Special handling for media entities based on workspace name.
    if ($entityTypeId === 'media' && !empty($workspaceName)) {
      // In "rags" workspaces: allow all media types except those with "cat".
      if (stripos($workspaceName, 'rags') !== FALSE) {
        return AccessResult::allowed(
          'User has workspace permission for media creation in rags workspace'
        )->cachePerPermissions();
      }

      // In "cat" workspaces: only allow media types that contain "cat".
      if (stripos($workspaceName, 'cat') !== FALSE) {
        if (!empty($bundle) && stripos($bundle, 'cat') === FALSE) {
          return AccessResult::forbidden(
          'In cat workspaces, only media types containing "cat" can be created'
          )->cachePerPermissions();
        }
        // If bundle is empty (general media creation) or contains "cat", allow.
        return AccessResult::allowed(
          'User has workspace permission for cat media creation in cat workspace'
        )->cachePerPermissions();
      }
    }

    // User has workspace permission - check for basic create permission.
    $basicPermission = 'create ' . $entity_type->getSingularLabel();
    if ($account->hasPermission($basicPermission) || $account->hasPermission('create any content')) {
      return AccessResult::allowed(
          'User has workspace and create permissions'
        )->cachePerPermissions();
    }

    // User has workspace permission but lacks basic create permission
    // For media entities, also check for alternative permission names.
    if ($entityTypeId === 'media') {
      $alternativePermissions = [
        'create media',
        'create media entities',
        'create media content',
        'create media item',
      ];

      foreach ($alternativePermissions as $altPermission) {
        if ($account->hasPermission($altPermission)) {
          return AccessResult::allowed(
            'User has workspace and alternative create permissions'
          )->cachePerPermissions();
        }
      }
    }

    return AccessResult::neutral();
  }

}
